/*
* Copyright 2008-2010 Nokia Siemens Networks Oyj
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/
package hudson.plugins.robot;

import hudson.Extension;
import hudson.FilePath;
import hudson.Launcher;
import hudson.model.AbstractBuild;
import hudson.model.AbstractProject;
import hudson.model.Action;
import hudson.model.BuildListener;
import hudson.model.Result;
import hudson.plugins.robot.helper.NumberHelper;
import hudson.plugins.robot.model.RobotResult;
import hudson.plugins.robot.parser.RobotParser;
import hudson.tasks.BuildStepDescriptor;
import hudson.tasks.BuildStepMonitor;
import hudson.tasks.Publisher;
import hudson.tasks.Recorder;
import hudson.util.FormValidation;

import java.io.IOException;
import java.io.PrintStream;

import javax.servlet.ServletException;

import org.apache.commons.lang.StringUtils;
import org.kohsuke.stapler.DataBoundConstructor;
import org.kohsuke.stapler.QueryParameter;

@SuppressWarnings("unchecked")
public class RobotPublisher extends Recorder {

	public static final String DEFAULT_REPORT_FILE = "report.html";
	public static final String FILE_ARCHIVE_DIR = "robot-plugin";
	
	private static final String DEFAULT_OUTPUT_FILE = "output.xml";
	private static final String DEFAULT_LOG_FILE = "log.html";
	
	private String outputPath;
	private String reportFileName;
	private String logFileName;
	private String outputFileName;
	private double passThreshold;
	private double unstableThreshold;
	private boolean onlyCritical;

	@DataBoundConstructor
	public RobotPublisher(String outputPath, String outputFileName, String reportFileName, String logFileName, double passThreshold, double unstableThreshold, boolean onlyCritical) {
		this.outputPath = outputPath;
		this.outputFileName = outputFileName;
		this.reportFileName = reportFileName;
		this.passThreshold = passThreshold;
		this.unstableThreshold = unstableThreshold;
		this.logFileName = logFileName;
		this.onlyCritical = onlyCritical;
	}

	public String getOutputPath() {
		return outputPath;
	}
	
	public String getOutputFileName() {
		if(StringUtils.isBlank(outputFileName))
			return DEFAULT_OUTPUT_FILE;
		return outputFileName;
	}
	
	public String getReportFileName() {
		if(StringUtils.isBlank(reportFileName))
			return DEFAULT_REPORT_FILE;
		return reportFileName;
	}

	public String getLogFileName() {
		if(StringUtils.isBlank(logFileName))
			return DEFAULT_LOG_FILE;
		return logFileName;
	}

	public double getPassThreshold() {
		return forceValidValue(passThreshold);
	}

	public double getUnstableThreshold() {
		return forceValidValue(unstableThreshold);
	}
	
	public boolean getOnlyCritical(){
		return onlyCritical;
	}
	
	@Override
	public Action getProjectAction(AbstractProject<?, ?> project) {
		return new RobotProjectAction(project);
	}

	@Override
	public boolean perform(AbstractBuild<?, ?> build, Launcher launcher,
			BuildListener listener) {
		if (build.getResult() != Result.ABORTED) {
			PrintStream logger = listener.getLogger();
			logger.println("Robot results publisher started...");

			FilePath robotOutputDir = new FilePath(build.getWorkspace(), outputPath);
			RobotParser parser = new RobotParser(new FilePath(robotOutputDir,getOutputFileName()));

			RobotResult result;
			logger.println("-Parsing output xml:");
			try {
				result = robotOutputDir.act(parser);
				logger.println(" Done!");
				logger.println("-Copying log files to build dir:");
				copyRobotFilesToBuildDir(build);
				logger.println(" Done!");
			} catch (Exception e) {
				logger.println(" Failed!");
				e.getCause().printStackTrace(logger);
				build.setResult(Result.FAILURE);
				return true;
			}


			logger.println("-Assigning results to build:");
			RobotBuildAction action = new RobotBuildAction(build, result,
					FILE_ARCHIVE_DIR, getReportFileName());
			build.addAction(action);
			logger.println(" Done!");
			
			logger.println("-Checking thresholds:");
			double passPercentage = (onlyCritical) ? action.getCriticalPassPercentage() : action.getPassPercentage() ;		
			if(build.getResult() != Result.FAILURE) {
				if(passPercentage < getUnstableThreshold()){
					logger.println(" Pass ratio under unstable threshold of " + getUnstableThreshold() + "%, marking build failed");
					build.setResult(Result.FAILURE);
				} else if(passPercentage < getPassThreshold()){
					logger.println(" Pass ratio under pass threshold of " + getPassThreshold() + "%, marking build unstable");
					build.setResult(Result.UNSTABLE);
				}
			}
			logger.println(" Done!");

			
			logger.println("Done publishing Robot results.");
		}
		return true;
	}

	private void copyRobotFilesToBuildDir(AbstractBuild<?, ?> build) throws IOException, InterruptedException{
            FilePath srcDir = new FilePath(build.getWorkspace(), outputPath);
            FilePath destDir = new FilePath(new FilePath(build.getRootDir()), FILE_ARCHIVE_DIR);
            copySplittedFiles(srcDir, destDir, getLogFileName());
            copySplittedFiles(srcDir, destDir, getReportFileName());
            copySplittedFiles(srcDir, destDir, getOutputFileName());
    }
	
	private void copySplittedFiles(FilePath src, FilePath dest, String filename) throws IOException, InterruptedException{
        src.copyRecursiveTo(trimSuffix(filename) + "*" + getSuffix(filename), dest);
	}
	
	private String trimSuffix(String filename){
		int index = filename.lastIndexOf('.');
		if(index > 0){
			filename = filename.substring(0, index);
		}
		return filename;
	}
	
	private String getSuffix(String filename){
		int index = filename.lastIndexOf('.');
		if(index > 0){
			return filename.substring(index); 
		}
		return "";
	}
	
	private double forceValidValue(double value){
		if(value > 100) return 100;
		else if(value < 0) return 0;
		else {
			return NumberHelper.roundToDecimals(value, 1);
		}
	}

	@Extension
	public static final class DescriptorImpl extends
			BuildStepDescriptor<Publisher> {

		@Override
		public boolean isApplicable(Class<? extends AbstractProject> aClass) {
			return true;
		}

		@Override
		public String getDisplayName() {
			return "Publish Robot Framework test results";
		}
        
        public FormValidation doCheckUnstableThreshold(@QueryParameter String value) throws IOException, ServletException {
        	if(isPercentageValue(value)) return FormValidation.ok();
        	else return FormValidation.error("Entry must be percentage value between 0-100");
        }
        
        public FormValidation doCheckPassThreshold(@QueryParameter String value) throws IOException, ServletException {
        	if(isPercentageValue(value)) return FormValidation.ok();
        	else return FormValidation.error("Entry must be percentage value between 0-100");
        } 
        
        private boolean isPercentageValue(String value){
        	try{
        		double doubleValue = Double.parseDouble(value);
        		if(doubleValue <= 100 && doubleValue >= 0) return true;
        		else return false;
        	}catch(NumberFormatException e){
        		return false;
        	}
        }
        
        
	}

	public BuildStepMonitor getRequiredMonitorService() {
		return BuildStepMonitor.NONE;
	}
}
