/*
 * Decompiled with CFR 0.152.
 */
package stream.io;

import java.io.IOException;
import java.io.InputStream;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import stream.Data;
import stream.ProcessContext;
import stream.annotations.Parameter;
import stream.audio.PlayWavData;
import stream.data.DataFactory;
import stream.io.AbstractStream;
import stream.io.SourceURL;
import stream.io.WavFileException;
import stream.runtime.ProcessContextImpl;

public class WavStream
extends AbstractStream {
    static Logger log = LoggerFactory.getLogger(WavStream.class);
    private static final int BUFFER_SIZE = 4096;
    private static final int FMT_CHUNK_ID = 544501094;
    private static final int DATA_CHUNK_ID = 1635017060;
    private static final int RIFF_CHUNK_ID = 1179011410;
    private static final int RIFF_TYPE_ID = 1163280727;
    private int bytesPerSample;
    private int numChannels;
    private Long sampleRate;
    private int blockAlign;
    private int validBits;
    private byte[] buffer;
    private int bufferPointer;
    private int bytesRead;
    final SourceURL source;
    InputStream in;
    protected Integer blockSize = 48000;
    protected Long blocksRead = 0L;
    final Data header = DataFactory.create();
    boolean eos = false;
    boolean initialized = false;

    public WavStream(SourceURL source) {
        super(source);
        this.source = source;
        this.buffer = new byte[4096];
    }

    public int getBytesPerSample() {
        return this.bytesPerSample;
    }

    public Long getSampleRate() {
        return this.sampleRate;
    }

    public int getBlockSize() {
        return this.blockSize;
    }

    @Parameter(description="The number of samples (double values) read from the stream for each data item.")
    public void setBlockSize(int blockSize) {
        this.blockSize = blockSize;
    }

    public void init() throws Exception {
        if (this.initialized) {
            log.info("audio-stream already initialized...");
            return;
        }
        log.info("Opening stream from {}", (Object)this.source);
        this.in = this.source.openStream();
        while (this.in.available() < 12) {
            log.info("Waiting for audio-samples to become available...");
            Thread.sleep(100L);
        }
        int bytesRead = this.in.read(this.buffer, 0, 12);
        if (bytesRead != 12) {
            throw new WavFileException("Not enough wav file bytes for header");
        }
        long riffChunkID = WavStream.getLE(this.buffer, 0, 4);
        long chunkSize = WavStream.getLE(this.buffer, 4, 4);
        long riffTypeID = WavStream.getLE(this.buffer, 8, 4);
        if (riffChunkID != 1179011410L) {
            throw new WavFileException("Invalid Wav Header data, incorrect riff chunk ID");
        }
        if (riffTypeID != 1163280727L) {
            throw new WavFileException("Invalid Wav Header data, incorrect riff type ID");
        }
        boolean foundFormat = false;
        boolean foundData = false;
        while (true) {
            long numChunkBytes;
            if ((bytesRead = this.in.read(this.buffer, 0, 8)) == -1) {
                throw new WavFileException("Reached end of file without finding format chunk");
            }
            if (bytesRead != 8) {
                throw new WavFileException("Could not read chunk header");
            }
            long chunkID = WavStream.getLE(this.buffer, 0, 4);
            chunkSize = WavStream.getLE(this.buffer, 4, 4);
            long l = numChunkBytes = chunkSize % 2L == 1L ? chunkSize + 1L : chunkSize;
            if (chunkID == 544501094L) {
                foundFormat = true;
                bytesRead = this.in.read(this.buffer, 0, 16);
                int compressionCode = (int)WavStream.getLE(this.buffer, 0, 2);
                if (compressionCode != 1) {
                    throw new WavFileException("Compression Code " + compressionCode + " not supported");
                }
                this.numChannels = (int)WavStream.getLE(this.buffer, 2, 2);
                this.sampleRate = WavStream.getLE(this.buffer, 4, 4);
                this.blockAlign = (int)WavStream.getLE(this.buffer, 12, 2);
                this.validBits = (int)WavStream.getLE(this.buffer, 14, 2);
                if (this.numChannels == 0) {
                    throw new WavFileException("Number of channels specified in header is equal to zero");
                }
                if (this.blockAlign == 0) {
                    throw new WavFileException("Block Align specified in header is equal to zero");
                }
                if (this.validBits < 2) {
                    throw new WavFileException("Valid Bits specified in header is less than 2");
                }
                if (this.validBits > 64) {
                    throw new WavFileException("Valid Bits specified in header is greater than 64, this is greater than a long can hold");
                }
                this.bytesPerSample = (this.validBits + 7) / 8;
                if (this.bytesPerSample * this.numChannels != this.blockAlign) {
                    throw new WavFileException("Block Align does not agree with bytes required for validBits and number of channels");
                }
                this.header.put((Object)"wav:samplerate", (Object)this.sampleRate);
                this.header.put((Object)"wav:channels", (Object)this.numChannels);
                this.header.put((Object)"wav:bytesPerSample", (Object)this.bytesPerSample);
                if ((numChunkBytes -= 16L) <= 0L) continue;
                this.in.skip(numChunkBytes);
                continue;
            }
            if (chunkID == 1635017060L) {
                if (!foundFormat) {
                    throw new WavFileException("Data chunk found before Format chunk");
                }
                if (chunkSize % (long)this.blockAlign != 0L) {
                    throw new WavFileException("Data Chunk size is not multiple of Block Align");
                }
                break;
            }
            this.in.skip(numChunkBytes);
        }
        foundData = true;
        if (!foundData) {
            throw new WavFileException("Did not find a data chunk");
        }
        this.bufferPointer = 0;
        bytesRead = 0;
        log.info("Sample rate is: {}", (Object)this.sampleRate);
        log.info("  chunk size is: {}", (Object)chunkSize);
        if (this.blockSize == null) {
            this.blockSize = new Long(chunkSize).intValue();
        }
        log.info("  each sample is for {} seconds", (Object)(1.0 / this.sampleRate.doubleValue()));
        log.info("  stream block size is: {} ({} seconds for each block)", (Object)this.blockSize, (Object)((double)this.blockSize.intValue() / this.sampleRate.doubleValue()));
        this.initialized = true;
    }

    private static long getLE(byte[] buffer, int pos, int numBytes) {
        long val = buffer[pos += --numBytes] & 0xFF;
        for (int b = 0; b < numBytes; ++b) {
            val = (val << 8) + (long)(buffer[--pos] & 0xFF);
        }
        return val;
    }

    public double readSample() throws IOException, WavFileException {
        double val = 0.0;
        for (int b = 0; b < this.bytesPerSample; ++b) {
            if (this.bufferPointer == this.bytesRead) {
                int read = this.in.read(this.buffer, 0, 4096);
                if (read < 0) {
                    this.eos = true;
                    return Double.NaN;
                }
                this.bytesRead = read;
                this.bufferPointer = 0;
            }
            int v = this.buffer[this.bufferPointer];
            if (b < this.bytesPerSample - 1 || this.bytesPerSample == 1) {
                v &= 0xFF;
            }
            val += (double)(v << b * 8);
            ++this.bufferPointer;
        }
        return val;
    }

    public Data readNext() throws Exception {
        if (this.eos) {
            return null;
        }
        Data item = DataFactory.create();
        if (this.blocksRead == 0L) {
            item.putAll((Map)this.header);
        }
        double min = 255.0;
        double max = 0.0;
        double avg = 0.0;
        double[] block = new double[this.blockSize.intValue()];
        int last = 0;
        int read = 0;
        for (int i = 0; i < block.length; ++i) {
            double sample = this.readSample();
            if (sample == Double.NaN) {
                this.eos = true;
                log.info("EOS! Last successful block was {}", (Object)i);
                break;
            }
            block[i] = sample;
            ++last;
            ++read;
            min = Math.min(min, block[i]);
            max = Math.max(max, block[i]);
            avg += block[i];
        }
        if (last < block.length) {
            double[] nb = new double[last];
            for (int i = 0; i < last; ++i) {
                nb[i] = block[i];
            }
            block = nb;
        }
        avg /= (double)block.length;
        double var = 0.0;
        for (int i = 0; i < block.length; ++i) {
            var += (avg - block[i]) * (avg - block[i]);
        }
        var = Math.sqrt(var) / (double)block.length;
        Long i = this.blocksRead;
        Long l = this.blocksRead = Long.valueOf(this.blocksRead + 1L);
        double rate = 1.0 / this.sampleRate.doubleValue();
        item.put((Object)"wav:position", (Object)(this.blocksRead.doubleValue() * (double)this.blockSize.intValue() * rate));
        item.put((Object)"wav:samples", (Object)block);
        item.put((Object)"wav:blocklength", (Object)block.length);
        item.put((Object)"wav:max", (Object)max);
        item.put((Object)"wav:min", (Object)min);
        item.put((Object)"wav:variance", (Object)var);
        item.put((Object)"wav:avg", (Object)avg);
        return item;
    }

    public void close() throws Exception {
        if (this.in != null) {
            this.in.close();
            this.in = null;
        }
    }

    public static void main(String[] args) {
        try {
            int limit = 100000;
            SourceURL url = new SourceURL("file:/Users/chris/tagesschau.wav.gz");
            url = new SourceURL("classpath:/tagesschau.wav.gz");
            WavStream stream = new WavStream(url);
            stream.setBlockSize(1200);
            stream.init();
            PlayWavData sound = new PlayWavData();
            sound.setVolume(0.25);
            sound.init((ProcessContext)new ProcessContextImpl());
            Data item = stream.read();
            for (int count = 0; item != null && count < limit; ++count) {
                log.debug("sample: {}", (Object)item);
                sound.process(item);
                item = stream.read();
            }
        }
        catch (Exception e) {
            e.printStackTrace();
        }
    }
}

