/* 
 * Copyright 2019 The Kathra Authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Contributors:
 *
 *    IRT SystemX (https://www.kathra.org/)    
 *
 */

package org.kathra.codegen.client;

import com.google.gson.reflect.TypeToken;
import org.kathra.client.*;
import org.kathra.utils.ApiException;
import org.kathra.utils.ApiResponse;
import org.kathra.utils.KathraSessionManager;

import java.io.File;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class CodegenClient {
    private ApiClient apiClient;

    public CodegenClient() {
        this.apiClient = new ApiClient().setUserAgent("CodegenClient 1.1.0-SNAPSHOT");
    }

    public CodegenClient(String serviceUrl) {
        this();
        apiClient.setBasePath(serviceUrl);
    }

    public CodegenClient(String serviceUrl, KathraSessionManager sessionManager) {
        this(serviceUrl);
        apiClient.setSessionManager(sessionManager);
    }

    public ApiClient getApiClient() {
        return apiClient;
    }

    private void setApiClient(ApiClient apiClient) {
        this.apiClient = apiClient;
    }

    /**
     * Build call for generateClient
     * @param apiFile The API definition file (required)
     * @param language The desired programming language (required)
     * @param artifactName The name of artifact to generate (optional)
     * @param groupId The groupId of artifact to generate (optional)
     * @param artifactVersion The version of artifact to generate (optional)
     * @param progressListener Progress listener
     * @param progressRequestListener Progress request listener
     * @return Call to execute
     * @throws ApiException If fail to serialize the request body object
     */
    public com.squareup.okhttp.Call generateClientCall(File apiFile, String language, String artifactName, String groupId, String artifactVersion, final ProgressResponseBody.ProgressListener progressListener, final ProgressRequestBody.ProgressRequestListener progressRequestListener) throws ApiException {
        Object localVarPostBody = null;
        
        // create path and map variables
        String localVarPath = "/client";

        List<Pair> localVarQueryParams = new ArrayList();
        List<Pair> localVarCollectionQueryParams = new ArrayList();
        if (artifactName != null)
        localVarQueryParams.addAll(apiClient.parameterToPair("artifactName", artifactName));
        if (groupId != null)
        localVarQueryParams.addAll(apiClient.parameterToPair("groupId", groupId));
        if (artifactVersion != null)
        localVarQueryParams.addAll(apiClient.parameterToPair("artifactVersion", artifactVersion));
        if (language != null)
        localVarQueryParams.addAll(apiClient.parameterToPair("language", language));

        Map<String, String> localVarHeaderParams = new HashMap();

        Map<String, Object> localVarFormParams = new HashMap();
        if (apiFile != null)
        localVarFormParams.put("apiFile", apiFile);

        final String[] localVarAccepts = {
            "application/zip"
        };
        final String localVarAccept = apiClient.selectHeaderAccept(localVarAccepts);
        if (localVarAccept != null) localVarHeaderParams.put("Accept", localVarAccept);

        final String[] localVarContentTypes = {
            "multipart/form-data"
        };
        final String localVarContentType = apiClient.selectHeaderContentType(localVarContentTypes);
        localVarHeaderParams.put("Content-Type", localVarContentType);

        if(progressListener != null) {
            apiClient.getHttpClient().networkInterceptors().add(chain -> {
                com.squareup.okhttp.Response originalResponse = chain.proceed(chain.request());
                return originalResponse.newBuilder()
                .body(new ProgressResponseBody(originalResponse.body(), progressListener))
                .build();
            });
        }

        String[] localVarAuthNames = new String[] {  };
        return apiClient.buildCall(localVarPath, "POST", localVarQueryParams, localVarCollectionQueryParams, localVarPostBody, localVarHeaderParams, localVarFormParams, localVarAuthNames, progressRequestListener);
    }
    
    @SuppressWarnings("rawtypes")
    private com.squareup.okhttp.Call generateClientValidateBeforeCall(File apiFile, String language, String artifactName, String groupId, String artifactVersion, final ProgressResponseBody.ProgressListener progressListener, final ProgressRequestBody.ProgressRequestListener progressRequestListener) throws ApiException {
        
        // verify the required parameter 'apiFile' is set
        if (apiFile == null) {
            throw new ApiException("Missing the required parameter 'apiFile' when calling generateClient(Async)");
        }
        
        // verify the required parameter 'language' is set
        if (language == null) {
            throw new ApiException("Missing the required parameter 'language' when calling generateClient(Async)");
        }
        
        
        com.squareup.okhttp.Call call = generateClientCall(apiFile, language, artifactName, groupId, artifactVersion, progressListener, progressRequestListener);
        return call;

    }

    /**
     * Generate Client
     * 
     * @param apiFile The API definition file (required)
     * @param language The desired programming language (required)
     * @param artifactName The name of artifact to generate (optional)
     * @param groupId The groupId of artifact to generate (optional)
     * @param artifactVersion The version of artifact to generate (optional)
     * @return File
     * @throws ApiException If fail to call the API, e.g. server error or cannot deserialize the response body
     */
    public File generateClient(File apiFile, String language, String artifactName, String groupId, String artifactVersion) throws ApiException {
        ApiResponse<File> resp = generateClientWithHttpInfo(apiFile, language, artifactName, groupId, artifactVersion);
        return resp.getData();
    }

    /**
     * Generate Client
     * 
     * @param apiFile The API definition file (required)
     * @param language The desired programming language (required)
     * @param artifactName The name of artifact to generate (optional)
     * @param groupId The groupId of artifact to generate (optional)
     * @param artifactVersion The version of artifact to generate (optional)
     * @return ApiResponse&lt;File&gt;
     * @throws ApiException If fail to call the API, e.g. server error or cannot deserialize the response body
     */
    public ApiResponse<File> generateClientWithHttpInfo(File apiFile, String language, String artifactName, String groupId, String artifactVersion) throws ApiException {
        com.squareup.okhttp.Call call = generateClientValidateBeforeCall(apiFile, language, artifactName, groupId, artifactVersion, null, null);
        Type localVarReturnType = new TypeToken<File>(){}.getType();
        return apiClient.execute(call, localVarReturnType);
    }

    /**
     * Generate Client (asynchronously)
     * 
     * @param apiFile The API definition file (required)
     * @param language The desired programming language (required)
     * @param artifactName The name of artifact to generate (optional)
     * @param groupId The groupId of artifact to generate (optional)
     * @param artifactVersion The version of artifact to generate (optional)
     * @param callback The callback to be executed when the API call finishes
     * @return The request call
     * @throws ApiException If fail to process the API call, e.g. serializing the request body object
     */
    public com.squareup.okhttp.Call generateClientAsync(File apiFile, String language, String artifactName, String groupId, String artifactVersion, final ApiCallback<File> callback) throws ApiException {

        ProgressResponseBody.ProgressListener progressListener = null;
        ProgressRequestBody.ProgressRequestListener progressRequestListener = null;

        if (callback != null) {
            progressListener = (bytesRead, contentLength, done) -> callback.onDownloadProgress(bytesRead, contentLength, done);
            progressRequestListener = (bytesWritten, contentLength, done) -> callback.onUploadProgress(bytesWritten, contentLength, done);
        }

        com.squareup.okhttp.Call call = generateClientValidateBeforeCall(apiFile, language, artifactName, groupId, artifactVersion, progressListener, progressRequestListener);
        Type localVarReturnType = new TypeToken<File>(){}.getType();
        apiClient.executeAsync(call, localVarReturnType, callback);
        return call;
    }
    /**
     * Build call for generateImplementation
     * @param apiFile The API definition file (required)
     * @param implemName The name of artifact to generate (required)
     * @param language The desired programming language (required)
     * @param artifactName The name of artifact to generate (optional)
     * @param groupId The groupId of artifact to generate (optional)
     * @param artifactVersion The version of artifact to generate (optional)
     * @param progressListener Progress listener
     * @param progressRequestListener Progress request listener
     * @return Call to execute
     * @throws ApiException If fail to serialize the request body object
     */
    public com.squareup.okhttp.Call generateImplementationCall(File apiFile, String implemName, String language, String artifactName, String groupId, String artifactVersion, final ProgressResponseBody.ProgressListener progressListener, final ProgressRequestBody.ProgressRequestListener progressRequestListener) throws ApiException {
        Object localVarPostBody = null;
        
        // create path and map variables
        String localVarPath = "/implem";

        List<Pair> localVarQueryParams = new ArrayList();
        List<Pair> localVarCollectionQueryParams = new ArrayList();
        if (implemName != null)
        localVarQueryParams.addAll(apiClient.parameterToPair("implemName", implemName));
        if (artifactName != null)
        localVarQueryParams.addAll(apiClient.parameterToPair("artifactName", artifactName));
        if (groupId != null)
        localVarQueryParams.addAll(apiClient.parameterToPair("groupId", groupId));
        if (artifactVersion != null)
        localVarQueryParams.addAll(apiClient.parameterToPair("artifactVersion", artifactVersion));
        if (language != null)
        localVarQueryParams.addAll(apiClient.parameterToPair("language", language));

        Map<String, String> localVarHeaderParams = new HashMap();

        Map<String, Object> localVarFormParams = new HashMap();
        if (apiFile != null)
        localVarFormParams.put("apiFile", apiFile);

        final String[] localVarAccepts = {
            "application/zip"
        };
        final String localVarAccept = apiClient.selectHeaderAccept(localVarAccepts);
        if (localVarAccept != null) localVarHeaderParams.put("Accept", localVarAccept);

        final String[] localVarContentTypes = {
            "multipart/form-data"
        };
        final String localVarContentType = apiClient.selectHeaderContentType(localVarContentTypes);
        localVarHeaderParams.put("Content-Type", localVarContentType);

        if(progressListener != null) {
            apiClient.getHttpClient().networkInterceptors().add(chain -> {
                com.squareup.okhttp.Response originalResponse = chain.proceed(chain.request());
                return originalResponse.newBuilder()
                .body(new ProgressResponseBody(originalResponse.body(), progressListener))
                .build();
            });
        }

        String[] localVarAuthNames = new String[] {  };
        return apiClient.buildCall(localVarPath, "POST", localVarQueryParams, localVarCollectionQueryParams, localVarPostBody, localVarHeaderParams, localVarFormParams, localVarAuthNames, progressRequestListener);
    }
    
    @SuppressWarnings("rawtypes")
    private com.squareup.okhttp.Call generateImplementationValidateBeforeCall(File apiFile, String implemName, String language, String artifactName, String groupId, String artifactVersion, final ProgressResponseBody.ProgressListener progressListener, final ProgressRequestBody.ProgressRequestListener progressRequestListener) throws ApiException {
        
        // verify the required parameter 'apiFile' is set
        if (apiFile == null) {
            throw new ApiException("Missing the required parameter 'apiFile' when calling generateImplementation(Async)");
        }
        
        // verify the required parameter 'implemName' is set
        if (implemName == null) {
            throw new ApiException("Missing the required parameter 'implemName' when calling generateImplementation(Async)");
        }
        
        // verify the required parameter 'language' is set
        if (language == null) {
            throw new ApiException("Missing the required parameter 'language' when calling generateImplementation(Async)");
        }
        
        
        com.squareup.okhttp.Call call = generateImplementationCall(apiFile, implemName, language, artifactName, groupId, artifactVersion, progressListener, progressRequestListener);
        return call;

    }

    /**
     * Generate Implementation
     * 
     * @param apiFile The API definition file (required)
     * @param implemName The name of artifact to generate (required)
     * @param language The desired programming language (required)
     * @param artifactName The name of artifact to generate (optional)
     * @param groupId The groupId of artifact to generate (optional)
     * @param artifactVersion The version of artifact to generate (optional)
     * @return File
     * @throws ApiException If fail to call the API, e.g. server error or cannot deserialize the response body
     */
    public File generateImplementation(File apiFile, String implemName, String language, String artifactName, String groupId, String artifactVersion) throws ApiException {
        ApiResponse<File> resp = generateImplementationWithHttpInfo(apiFile, implemName, language, artifactName, groupId, artifactVersion);
        return resp.getData();
    }

    /**
     * Generate Implementation
     * 
     * @param apiFile The API definition file (required)
     * @param implemName The name of artifact to generate (required)
     * @param language The desired programming language (required)
     * @param artifactName The name of artifact to generate (optional)
     * @param groupId The groupId of artifact to generate (optional)
     * @param artifactVersion The version of artifact to generate (optional)
     * @return ApiResponse&lt;File&gt;
     * @throws ApiException If fail to call the API, e.g. server error or cannot deserialize the response body
     */
    public ApiResponse<File> generateImplementationWithHttpInfo(File apiFile, String implemName, String language, String artifactName, String groupId, String artifactVersion) throws ApiException {
        com.squareup.okhttp.Call call = generateImplementationValidateBeforeCall(apiFile, implemName, language, artifactName, groupId, artifactVersion, null, null);
        Type localVarReturnType = new TypeToken<File>(){}.getType();
        return apiClient.execute(call, localVarReturnType);
    }

    /**
     * Generate Implementation (asynchronously)
     * 
     * @param apiFile The API definition file (required)
     * @param implemName The name of artifact to generate (required)
     * @param language The desired programming language (required)
     * @param artifactName The name of artifact to generate (optional)
     * @param groupId The groupId of artifact to generate (optional)
     * @param artifactVersion The version of artifact to generate (optional)
     * @param callback The callback to be executed when the API call finishes
     * @return The request call
     * @throws ApiException If fail to process the API call, e.g. serializing the request body object
     */
    public com.squareup.okhttp.Call generateImplementationAsync(File apiFile, String implemName, String language, String artifactName, String groupId, String artifactVersion, final ApiCallback<File> callback) throws ApiException {

        ProgressResponseBody.ProgressListener progressListener = null;
        ProgressRequestBody.ProgressRequestListener progressRequestListener = null;

        if (callback != null) {
            progressListener = (bytesRead, contentLength, done) -> callback.onDownloadProgress(bytesRead, contentLength, done);
            progressRequestListener = (bytesWritten, contentLength, done) -> callback.onUploadProgress(bytesWritten, contentLength, done);
        }

        com.squareup.okhttp.Call call = generateImplementationValidateBeforeCall(apiFile, implemName, language, artifactName, groupId, artifactVersion, progressListener, progressRequestListener);
        Type localVarReturnType = new TypeToken<File>(){}.getType();
        apiClient.executeAsync(call, localVarReturnType, callback);
        return call;
    }
    /**
     * Build call for generateInterface
     * @param apiFile The API definition file (required)
     * @param language The desired programming language (required)
     * @param artifactName The name of artifact to generate (optional)
     * @param groupId The groupId of artifact to generate (optional)
     * @param artifactVersion The version of artifact to generate (optional)
     * @param progressListener Progress listener
     * @param progressRequestListener Progress request listener
     * @return Call to execute
     * @throws ApiException If fail to serialize the request body object
     */
    public com.squareup.okhttp.Call generateInterfaceCall(File apiFile, String language, String artifactName, String groupId, String artifactVersion, final ProgressResponseBody.ProgressListener progressListener, final ProgressRequestBody.ProgressRequestListener progressRequestListener) throws ApiException {
        Object localVarPostBody = null;
        
        // create path and map variables
        String localVarPath = "/interface";

        List<Pair> localVarQueryParams = new ArrayList();
        List<Pair> localVarCollectionQueryParams = new ArrayList();
        if (artifactName != null)
        localVarQueryParams.addAll(apiClient.parameterToPair("artifactName", artifactName));
        if (groupId != null)
        localVarQueryParams.addAll(apiClient.parameterToPair("groupId", groupId));
        if (artifactVersion != null)
        localVarQueryParams.addAll(apiClient.parameterToPair("artifactVersion", artifactVersion));
        if (language != null)
        localVarQueryParams.addAll(apiClient.parameterToPair("language", language));

        Map<String, String> localVarHeaderParams = new HashMap();

        Map<String, Object> localVarFormParams = new HashMap();
        if (apiFile != null)
        localVarFormParams.put("apiFile", apiFile);

        final String[] localVarAccepts = {
            "application/zip"
        };
        final String localVarAccept = apiClient.selectHeaderAccept(localVarAccepts);
        if (localVarAccept != null) localVarHeaderParams.put("Accept", localVarAccept);

        final String[] localVarContentTypes = {
            "multipart/form-data"
        };
        final String localVarContentType = apiClient.selectHeaderContentType(localVarContentTypes);
        localVarHeaderParams.put("Content-Type", localVarContentType);

        if(progressListener != null) {
            apiClient.getHttpClient().networkInterceptors().add(chain -> {
                com.squareup.okhttp.Response originalResponse = chain.proceed(chain.request());
                return originalResponse.newBuilder()
                .body(new ProgressResponseBody(originalResponse.body(), progressListener))
                .build();
            });
        }

        String[] localVarAuthNames = new String[] {  };
        return apiClient.buildCall(localVarPath, "POST", localVarQueryParams, localVarCollectionQueryParams, localVarPostBody, localVarHeaderParams, localVarFormParams, localVarAuthNames, progressRequestListener);
    }
    
    @SuppressWarnings("rawtypes")
    private com.squareup.okhttp.Call generateInterfaceValidateBeforeCall(File apiFile, String language, String artifactName, String groupId, String artifactVersion, final ProgressResponseBody.ProgressListener progressListener, final ProgressRequestBody.ProgressRequestListener progressRequestListener) throws ApiException {
        
        // verify the required parameter 'apiFile' is set
        if (apiFile == null) {
            throw new ApiException("Missing the required parameter 'apiFile' when calling generateInterface(Async)");
        }
        
        // verify the required parameter 'language' is set
        if (language == null) {
            throw new ApiException("Missing the required parameter 'language' when calling generateInterface(Async)");
        }
        
        
        com.squareup.okhttp.Call call = generateInterfaceCall(apiFile, language, artifactName, groupId, artifactVersion, progressListener, progressRequestListener);
        return call;

    }

    /**
     * Generate Interface
     * 
     * @param apiFile The API definition file (required)
     * @param language The desired programming language (required)
     * @param artifactName The name of artifact to generate (optional)
     * @param groupId The groupId of artifact to generate (optional)
     * @param artifactVersion The version of artifact to generate (optional)
     * @return File
     * @throws ApiException If fail to call the API, e.g. server error or cannot deserialize the response body
     */
    public File generateInterface(File apiFile, String language, String artifactName, String groupId, String artifactVersion) throws ApiException {
        ApiResponse<File> resp = generateInterfaceWithHttpInfo(apiFile, language, artifactName, groupId, artifactVersion);
        return resp.getData();
    }

    /**
     * Generate Interface
     * 
     * @param apiFile The API definition file (required)
     * @param language The desired programming language (required)
     * @param artifactName The name of artifact to generate (optional)
     * @param groupId The groupId of artifact to generate (optional)
     * @param artifactVersion The version of artifact to generate (optional)
     * @return ApiResponse&lt;File&gt;
     * @throws ApiException If fail to call the API, e.g. server error or cannot deserialize the response body
     */
    public ApiResponse<File> generateInterfaceWithHttpInfo(File apiFile, String language, String artifactName, String groupId, String artifactVersion) throws ApiException {
        com.squareup.okhttp.Call call = generateInterfaceValidateBeforeCall(apiFile, language, artifactName, groupId, artifactVersion, null, null);
        Type localVarReturnType = new TypeToken<File>(){}.getType();
        return apiClient.execute(call, localVarReturnType);
    }

    /**
     * Generate Interface (asynchronously)
     * 
     * @param apiFile The API definition file (required)
     * @param language The desired programming language (required)
     * @param artifactName The name of artifact to generate (optional)
     * @param groupId The groupId of artifact to generate (optional)
     * @param artifactVersion The version of artifact to generate (optional)
     * @param callback The callback to be executed when the API call finishes
     * @return The request call
     * @throws ApiException If fail to process the API call, e.g. serializing the request body object
     */
    public com.squareup.okhttp.Call generateInterfaceAsync(File apiFile, String language, String artifactName, String groupId, String artifactVersion, final ApiCallback<File> callback) throws ApiException {

        ProgressResponseBody.ProgressListener progressListener = null;
        ProgressRequestBody.ProgressRequestListener progressRequestListener = null;

        if (callback != null) {
            progressListener = (bytesRead, contentLength, done) -> callback.onDownloadProgress(bytesRead, contentLength, done);
            progressRequestListener = (bytesWritten, contentLength, done) -> callback.onUploadProgress(bytesWritten, contentLength, done);
        }

        com.squareup.okhttp.Call call = generateInterfaceValidateBeforeCall(apiFile, language, artifactName, groupId, artifactVersion, progressListener, progressRequestListener);
        Type localVarReturnType = new TypeToken<File>(){}.getType();
        apiClient.executeAsync(call, localVarReturnType, callback);
        return call;
    }
    /**
     * Build call for generateModel
     * @param apiFile The API definition file (required)
     * @param language The desired programming language (required)
     * @param artifactName The name of artifact to generate (optional)
     * @param groupId The groupId of artifact to generate (optional)
     * @param artifactVersion The version of artifact to generate (optional)
     * @param progressListener Progress listener
     * @param progressRequestListener Progress request listener
     * @return Call to execute
     * @throws ApiException If fail to serialize the request body object
     */
    public com.squareup.okhttp.Call generateModelCall(File apiFile, String language, String artifactName, String groupId, String artifactVersion, final ProgressResponseBody.ProgressListener progressListener, final ProgressRequestBody.ProgressRequestListener progressRequestListener) throws ApiException {
        Object localVarPostBody = null;
        
        // create path and map variables
        String localVarPath = "/model";

        List<Pair> localVarQueryParams = new ArrayList();
        List<Pair> localVarCollectionQueryParams = new ArrayList();
        if (artifactName != null)
        localVarQueryParams.addAll(apiClient.parameterToPair("artifactName", artifactName));
        if (groupId != null)
        localVarQueryParams.addAll(apiClient.parameterToPair("groupId", groupId));
        if (artifactVersion != null)
        localVarQueryParams.addAll(apiClient.parameterToPair("artifactVersion", artifactVersion));
        if (language != null)
        localVarQueryParams.addAll(apiClient.parameterToPair("language", language));

        Map<String, String> localVarHeaderParams = new HashMap();

        Map<String, Object> localVarFormParams = new HashMap();
        if (apiFile != null)
        localVarFormParams.put("apiFile", apiFile);

        final String[] localVarAccepts = {
            "application/zip"
        };
        final String localVarAccept = apiClient.selectHeaderAccept(localVarAccepts);
        if (localVarAccept != null) localVarHeaderParams.put("Accept", localVarAccept);

        final String[] localVarContentTypes = {
            "multipart/form-data"
        };
        final String localVarContentType = apiClient.selectHeaderContentType(localVarContentTypes);
        localVarHeaderParams.put("Content-Type", localVarContentType);

        if(progressListener != null) {
            apiClient.getHttpClient().networkInterceptors().add(chain -> {
                com.squareup.okhttp.Response originalResponse = chain.proceed(chain.request());
                return originalResponse.newBuilder()
                .body(new ProgressResponseBody(originalResponse.body(), progressListener))
                .build();
            });
        }

        String[] localVarAuthNames = new String[] {  };
        return apiClient.buildCall(localVarPath, "POST", localVarQueryParams, localVarCollectionQueryParams, localVarPostBody, localVarHeaderParams, localVarFormParams, localVarAuthNames, progressRequestListener);
    }
    
    @SuppressWarnings("rawtypes")
    private com.squareup.okhttp.Call generateModelValidateBeforeCall(File apiFile, String language, String artifactName, String groupId, String artifactVersion, final ProgressResponseBody.ProgressListener progressListener, final ProgressRequestBody.ProgressRequestListener progressRequestListener) throws ApiException {
        
        // verify the required parameter 'apiFile' is set
        if (apiFile == null) {
            throw new ApiException("Missing the required parameter 'apiFile' when calling generateModel(Async)");
        }
        
        // verify the required parameter 'language' is set
        if (language == null) {
            throw new ApiException("Missing the required parameter 'language' when calling generateModel(Async)");
        }
        
        
        com.squareup.okhttp.Call call = generateModelCall(apiFile, language, artifactName, groupId, artifactVersion, progressListener, progressRequestListener);
        return call;

    }

    /**
     * Generate Model
     * 
     * @param apiFile The API definition file (required)
     * @param language The desired programming language (required)
     * @param artifactName The name of artifact to generate (optional)
     * @param groupId The groupId of artifact to generate (optional)
     * @param artifactVersion The version of artifact to generate (optional)
     * @return File
     * @throws ApiException If fail to call the API, e.g. server error or cannot deserialize the response body
     */
    public File generateModel(File apiFile, String language, String artifactName, String groupId, String artifactVersion) throws ApiException {
        ApiResponse<File> resp = generateModelWithHttpInfo(apiFile, language, artifactName, groupId, artifactVersion);
        return resp.getData();
    }

    /**
     * Generate Model
     * 
     * @param apiFile The API definition file (required)
     * @param language The desired programming language (required)
     * @param artifactName The name of artifact to generate (optional)
     * @param groupId The groupId of artifact to generate (optional)
     * @param artifactVersion The version of artifact to generate (optional)
     * @return ApiResponse&lt;File&gt;
     * @throws ApiException If fail to call the API, e.g. server error or cannot deserialize the response body
     */
    public ApiResponse<File> generateModelWithHttpInfo(File apiFile, String language, String artifactName, String groupId, String artifactVersion) throws ApiException {
        com.squareup.okhttp.Call call = generateModelValidateBeforeCall(apiFile, language, artifactName, groupId, artifactVersion, null, null);
        Type localVarReturnType = new TypeToken<File>(){}.getType();
        return apiClient.execute(call, localVarReturnType);
    }

    /**
     * Generate Model (asynchronously)
     * 
     * @param apiFile The API definition file (required)
     * @param language The desired programming language (required)
     * @param artifactName The name of artifact to generate (optional)
     * @param groupId The groupId of artifact to generate (optional)
     * @param artifactVersion The version of artifact to generate (optional)
     * @param callback The callback to be executed when the API call finishes
     * @return The request call
     * @throws ApiException If fail to process the API call, e.g. serializing the request body object
     */
    public com.squareup.okhttp.Call generateModelAsync(File apiFile, String language, String artifactName, String groupId, String artifactVersion, final ApiCallback<File> callback) throws ApiException {

        ProgressResponseBody.ProgressListener progressListener = null;
        ProgressRequestBody.ProgressRequestListener progressRequestListener = null;

        if (callback != null) {
            progressListener = (bytesRead, contentLength, done) -> callback.onDownloadProgress(bytesRead, contentLength, done);
            progressRequestListener = (bytesWritten, contentLength, done) -> callback.onUploadProgress(bytesWritten, contentLength, done);
        }

        com.squareup.okhttp.Call call = generateModelValidateBeforeCall(apiFile, language, artifactName, groupId, artifactVersion, progressListener, progressRequestListener);
        Type localVarReturnType = new TypeToken<File>(){}.getType();
        apiClient.executeAsync(call, localVarReturnType, callback);
        return call;
    }
}
