/* 
 * Copyright 2019 The Kathra Authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Contributors:
 *
 *    IRT SystemX (https://www.kathra.org/)    
 *
 */

package org.kathra.codegen;

import org.apache.camel.builder.RouteBuilder;
import org.apache.camel.cdi.ContextName;
import org.apache.camel.model.rest.RestBindingMode;
import static org.apache.camel.model.rest.RestParamType.*;
import org.kathra.iface.KathraRequestHandler;

@ContextName("Codegen")
public class CodegenApi extends RouteBuilder {

    @Override
    public void configure() throws Exception {
        // configure we want to use servlet as the component for the rest DSL
        // and we enable json binding mode
        restConfiguration().component(KathraRequestHandler.HTTP_SERVER)
        // use json binding mode so Camel automatic binds json <--> pojo
        .bindingMode(RestBindingMode.off)
        // and output using pretty print
        .dataFormatProperty("prettyPrint", "true")
        .dataFormatProperty("json.in.disableFeatures", "FAIL_ON_UNKNOWN_PROPERTIES")
        // setup context path on localhost and port number that netty will use
        .contextPath("/api/v1")
        .port("{{env:HTTP_PORT:8080}}")
        // add swagger api-doc out of the box
        .apiContextPath("/swagger.json")
        .apiProperty("api.title", "Kathra Codegen")
        .apiProperty("api.version", "1.1.0-SNAPSHOT")
        .apiProperty("api.description", "Codegen")
        // and enable CORS
        .apiProperty("cors", "true")
        .enableCORS(true).corsAllowCredentials(true)
        .corsHeaderProperty("Access-Control-Allow-Headers", "Origin, Accept, X-Requested-With, Content-Type," +
                "Access-Control-Request-Method, Access-Control-Request-Headers, Authorization");

        rest()

        .post("/client").consumes("multipart/form-data").produces("application/zip")
            .description("Generate Client")
                .param()
                    .required(true)
                    .dataType("file")
                    .name("apiFile")
                    .type(formData)
                    .description("The API definition file")
                .endParam()
                .param()
                    .required(true)
                    .dataType("string")
                    .allowableValues("java","python")
                    .name("language")
                    .type(query)
                    .description("The desired programming language")
                .endParam()
                .param()
                    .required(false)
                    .dataType("string")
                    .name("artifactName")
                    .type(query)
                    .description("The name of artifact to generate")
                .endParam()
                .param()
                    .required(false)
                    .dataType("string")
                    .name("groupId")
                    .type(query)
                    .description("The groupId of artifact to generate")
                .endParam()
                .param()
                    .required(false)
                    .dataType("string")
                    .name("artifactVersion")
                    .type(query)
                    .description("The version of artifact to generate")
                .endParam()
            .route().onException(Exception.class)
                    .to("bean:CodegenController?method=handleException")
                    .to("bean:CodegenController?method=deleteSession")
                    .handled(true).stop().end()
                .to("bean:CodegenController?method=handleRequest")
                .to("bean:CodegenController?method=generateClient(${header.apiFile},${header.language},${header.artifactName},${header.groupId},${header.artifactVersion})")
                .to("bean:CodegenController?method=postProcessResponse")
            .endRest()

        .post("/implem").consumes("multipart/form-data").produces("application/zip")
            .description("Generate Implementation")
                .param()
                    .required(true)
                    .dataType("file")
                    .name("apiFile")
                    .type(formData)
                    .description("The API definition file")
                .endParam()
                .param()
                    .required(true)
                    .dataType("string")
                    .name("implemName")
                    .type(query)
                    .description("The name of artifact to generate")
                .endParam()
                .param()
                    .required(true)
                    .dataType("string")
                    .allowableValues("java","python")
                    .name("language")
                    .type(query)
                    .description("The desired programming language")
                .endParam()
                .param()
                    .required(false)
                    .dataType("string")
                    .name("artifactName")
                    .type(query)
                    .description("The name of artifact to generate")
                .endParam()
                .param()
                    .required(false)
                    .dataType("string")
                    .name("groupId")
                    .type(query)
                    .description("The groupId of artifact to generate")
                .endParam()
                .param()
                    .required(false)
                    .dataType("string")
                    .name("artifactVersion")
                    .type(query)
                    .description("The version of artifact to generate")
                .endParam()
            .route().onException(Exception.class)
                    .to("bean:CodegenController?method=handleException")
                    .to("bean:CodegenController?method=deleteSession")
                    .handled(true).stop().end()
                .to("bean:CodegenController?method=handleRequest")
                .to("bean:CodegenController?method=generateImplementation(${header.apiFile},${header.implemName},${header.language},${header.artifactName},${header.groupId},${header.artifactVersion})")
                .to("bean:CodegenController?method=postProcessResponse")
            .endRest()

        .post("/interface").consumes("multipart/form-data").produces("application/zip")
            .description("Generate Interface")
                .param()
                    .required(true)
                    .dataType("file")
                    .name("apiFile")
                    .type(formData)
                    .description("The API definition file")
                .endParam()
                .param()
                    .required(true)
                    .dataType("string")
                    .allowableValues("java","python")
                    .name("language")
                    .type(query)
                    .description("The desired programming language")
                .endParam()
                .param()
                    .required(false)
                    .dataType("string")
                    .name("artifactName")
                    .type(query)
                    .description("The name of artifact to generate")
                .endParam()
                .param()
                    .required(false)
                    .dataType("string")
                    .name("groupId")
                    .type(query)
                    .description("The groupId of artifact to generate")
                .endParam()
                .param()
                    .required(false)
                    .dataType("string")
                    .name("artifactVersion")
                    .type(query)
                    .description("The version of artifact to generate")
                .endParam()
            .route().onException(Exception.class)
                    .to("bean:CodegenController?method=handleException")
                    .to("bean:CodegenController?method=deleteSession")
                    .handled(true).stop().end()
                .to("bean:CodegenController?method=handleRequest")
                .to("bean:CodegenController?method=generateInterface(${header.apiFile},${header.language},${header.artifactName},${header.groupId},${header.artifactVersion})")
                .to("bean:CodegenController?method=postProcessResponse")
            .endRest()

        .post("/model").consumes("multipart/form-data").produces("application/zip")
            .description("Generate Model")
                .param()
                    .required(true)
                    .dataType("file")
                    .name("apiFile")
                    .type(formData)
                    .description("The API definition file")
                .endParam()
                .param()
                    .required(true)
                    .dataType("string")
                    .allowableValues("java","python")
                    .name("language")
                    .type(query)
                    .description("The desired programming language")
                .endParam()
                .param()
                    .required(false)
                    .dataType("string")
                    .name("artifactName")
                    .type(query)
                    .description("The name of artifact to generate")
                .endParam()
                .param()
                    .required(false)
                    .dataType("string")
                    .name("groupId")
                    .type(query)
                    .description("The groupId of artifact to generate")
                .endParam()
                .param()
                    .required(false)
                    .dataType("string")
                    .name("artifactVersion")
                    .type(query)
                    .description("The version of artifact to generate")
                .endParam()
            .route().onException(Exception.class)
                    .to("bean:CodegenController?method=handleException")
                    .to("bean:CodegenController?method=deleteSession")
                    .handled(true).stop().end()
                .to("bean:CodegenController?method=handleRequest")
                .to("bean:CodegenController?method=generateModel(${header.apiFile},${header.language},${header.artifactName},${header.groupId},${header.artifactVersion})")
                .to("bean:CodegenController?method=postProcessResponse")
            .endRest();
    }
}