/*
 * Decompiled with CFR 0.152.
 */
package org.kendar.sync.lib.twoway;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import org.kendar.sync.lib.twoway.ConflictItem;
import org.kendar.sync.lib.twoway.LogEntry;
import org.kendar.sync.lib.twoway.SyncAction;
import org.kendar.sync.lib.twoway.SyncActions;
import org.kendar.sync.lib.twoway.SyncItem;
import org.kendar.sync.lib.utils.FileUtils;

public class StatusAnalyzer {
    private static final String LAST_UPDATE_LOG = ".lastupdate.log";
    private static final String OPERATION_LOG = ".operation.log";
    private static final String LAST_COMPACT_LOG = ".lastcompact.log";
    private static final DateTimeFormatter TIMESTAMP_FORMAT = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
    private final Path baseDirectory;
    private final Path lastUpdateLogPath;
    private final Path operationLogPath;
    private final Path lastCompactLogPath;
    private Map<String, FileInfo> previousFileStates;

    public StatusAnalyzer(String baseDirectory) {
        this.baseDirectory = Paths.get(baseDirectory, new String[0]).toAbsolutePath();
        this.lastUpdateLogPath = this.baseDirectory.resolve(LAST_UPDATE_LOG);
        this.operationLogPath = this.baseDirectory.resolve(OPERATION_LOG);
        this.lastCompactLogPath = this.baseDirectory.resolve(LAST_COMPACT_LOG);
        this.previousFileStates = new ConcurrentHashMap<String, FileInfo>();
    }

    public List<LogEntry> analyze() throws IOException {
        Instant runStartTime = Instant.now();
        this.loadPreviousState();
        Map<String, FileInfo> currentFileStates = this.getCurrentFileStates();
        List<LogEntry> changes = this.detectChanges(currentFileStates, runStartTime);
        this.writeOperationLog(changes);
        this.writeLastUpdateLog(runStartTime);
        this.previousFileStates = currentFileStates;
        return changes;
    }

    public void compact() throws IOException {
        Instant compactTime = Instant.now();
        if (Files.exists(this.operationLogPath, new LinkOption[0])) {
            LinkedHashMap<String, LogEntry> latestCreations = new LinkedHashMap<String, LogEntry>();
            try (BufferedReader reader = Files.newBufferedReader(this.operationLogPath);){
                String line;
                while ((line = reader.readLine()) != null) {
                    LogEntry entry = this.parseLogEntry(line);
                    if (entry == null || !"CR".equals(entry.operation)) continue;
                    latestCreations.put(entry.relativePath, entry);
                }
            }
            try (BufferedWriter writer = Files.newBufferedWriter(this.operationLogPath, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);){
                for (LogEntry entry : latestCreations.values()) {
                    writer.write(this.formatLogEntry(entry));
                    writer.newLine();
                }
            }
        }
        this.writeLastCompactLog(compactTime);
    }

    public SyncActions compare(Path otherLogPath) throws IOException {
        Map<String, LogEntry> remoteOperations = this.loadOperationLog(otherLogPath);
        return this.compare(remoteOperations);
    }

    public SyncActions compare(Map<String, LogEntry> remoteOperations) throws IOException {
        Map<String, LogEntry> localOperations = this.loadOperationLog(this.operationLogPath);
        SyncActions actions = new SyncActions();
        HashSet<String> allFiles = new HashSet<String>();
        allFiles.addAll(localOperations.keySet());
        allFiles.addAll(remoteOperations.keySet());
        for (String filePath : allFiles) {
            LogEntry localEntry = localOperations.get(filePath);
            LogEntry remoteEntry = remoteOperations.get(filePath);
            SyncDecision decision = this.decideSyncAction(localEntry, remoteEntry, filePath);
            switch (decision.action) {
                case UPDATE_FROM_REMOTE: {
                    actions.filesToUpdate.add(new SyncItem(filePath, decision.sourceEntry));
                    break;
                }
                case UPDATE_TO_REMOTE: {
                    actions.filesToSend.add(new SyncItem(filePath, decision.sourceEntry));
                    break;
                }
                case DELETE_LOCAL: {
                    actions.filesToDelete.add(filePath);
                    break;
                }
                case DELETE_REMOTE: {
                    actions.filesToDeleteRemote.add(filePath);
                    break;
                }
                case CONFLICT: {
                    actions.conflicts.add(new ConflictItem(filePath, localEntry, remoteEntry));
                    break;
                }
            }
        }
        return actions;
    }

    private Map<String, LogEntry> loadOperationLog(Path logPath) throws IOException {
        HashMap<String, LogEntry> operations = new HashMap<String, LogEntry>();
        if (!Files.exists(logPath, new LinkOption[0])) {
            return operations;
        }
        try (BufferedReader reader = Files.newBufferedReader(logPath);){
            String line;
            while ((line = reader.readLine()) != null) {
                LogEntry entry = this.parseLogEntry(line);
                if (entry == null) continue;
                operations.put(entry.relativePath, entry);
            }
        }
        return operations;
    }

    private SyncDecision decideSyncAction(LogEntry localEntry, LogEntry remoteEntry, String filePath) {
        if (localEntry != null && remoteEntry == null) {
            if ("DE".equals(localEntry.operation)) {
                return new SyncDecision(SyncAction.NO_ACTION, null);
            }
            return new SyncDecision(SyncAction.UPDATE_TO_REMOTE, localEntry);
        }
        if (localEntry == null && remoteEntry != null) {
            if ("DE".equals(remoteEntry.operation)) {
                return new SyncDecision(SyncAction.NO_ACTION, null);
            }
            return new SyncDecision(SyncAction.UPDATE_FROM_REMOTE, remoteEntry);
        }
        if (localEntry != null) {
            if ("DE".equals(localEntry.operation) && "DE".equals(remoteEntry.operation)) {
                return new SyncDecision(SyncAction.NO_ACTION, null);
            }
            if ("DE".equals(localEntry.operation)) {
                return new SyncDecision(SyncAction.DELETE_REMOTE, localEntry);
            }
            if ("DE".equals(remoteEntry.operation)) {
                return new SyncDecision(SyncAction.DELETE_LOCAL, remoteEntry);
            }
            if (!localEntry.creationTime.equals(remoteEntry.creationTime)) {
                return new SyncDecision(SyncAction.CONFLICT, null);
            }
            if (!localEntry.modificationTime.equals(remoteEntry.modificationTime)) {
                if (localEntry.modificationTime.isAfter(remoteEntry.modificationTime)) {
                    return new SyncDecision(SyncAction.UPDATE_TO_REMOTE, localEntry);
                }
                if (remoteEntry.modificationTime.isAfter(localEntry.modificationTime)) {
                    return new SyncDecision(SyncAction.UPDATE_FROM_REMOTE, remoteEntry);
                }
                return new SyncDecision(SyncAction.CONFLICT, null);
            }
        }
        return new SyncDecision(SyncAction.NO_ACTION, null);
    }

    private void writeLastCompactLog(Instant compactTime) throws IOException {
        Files.createDirectories(this.lastCompactLogPath.getParent(), new FileAttribute[0]);
        String timestamp = LocalDateTime.ofInstant(compactTime, ZoneId.systemDefault()).format(TIMESTAMP_FORMAT);
        try (BufferedWriter writer = Files.newBufferedWriter(this.lastCompactLogPath, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);){
            writer.write(timestamp);
        }
    }

    public Optional<Instant> getLastCompactTime() {
        if (!Files.exists(this.lastCompactLogPath, new LinkOption[0])) {
            return Optional.empty();
        }
        try {
            String content = FileUtils.readFile(this.lastCompactLogPath).trim();
            if (content.isEmpty()) {
                return Optional.empty();
            }
            LocalDateTime dateTime = LocalDateTime.parse(content, TIMESTAMP_FORMAT);
            return Optional.of(dateTime.atZone(ZoneId.systemDefault()).toInstant());
        }
        catch (Exception e) {
            return Optional.empty();
        }
    }

    private void loadPreviousState() {
        if (!Files.exists(this.operationLogPath, new LinkOption[0])) {
            return;
        }
        try (BufferedReader reader = Files.newBufferedReader(this.operationLogPath);){
            String line;
            while ((line = reader.readLine()) != null) {
                LogEntry entry = this.parseLogEntry(line);
                if (entry == null || entry.operation.equals("DE")) continue;
                this.previousFileStates.put(entry.relativePath, new FileInfo(entry.creationTime, entry.modificationTime, entry.size));
            }
        }
        catch (IOException e) {
            this.previousFileStates.clear();
        }
    }

    private Map<String, FileInfo> getCurrentFileStates() throws IOException {
        final ConcurrentHashMap<String, FileInfo> currentStates = new ConcurrentHashMap<String, FileInfo>();
        if (!Files.exists(this.baseDirectory, new LinkOption[0])) {
            return currentStates;
        }
        Files.walkFileTree(this.baseDirectory, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
                if (file.equals(StatusAnalyzer.this.lastUpdateLogPath) || file.equals(StatusAnalyzer.this.operationLogPath) || file.equals(StatusAnalyzer.this.lastCompactLogPath)) {
                    return FileVisitResult.CONTINUE;
                }
                String relativePath = StatusAnalyzer.this.baseDirectory.relativize(file).toString().replace('\\', '/');
                Instant creationTime = attrs.creationTime().toInstant();
                Instant modificationTime = attrs.lastModifiedTime().toInstant();
                long size = attrs.size();
                currentStates.put(relativePath, new FileInfo(creationTime, modificationTime, size));
                return FileVisitResult.CONTINUE;
            }
        });
        return currentStates;
    }

    private List<LogEntry> detectChanges(Map<String, FileInfo> currentStates, Instant runStartTime) {
        String path;
        ArrayList<LogEntry> changes = new ArrayList<LogEntry>();
        Instant now = Instant.now();
        for (Map.Entry<String, FileInfo> entry : currentStates.entrySet()) {
            path = entry.getKey();
            FileInfo currentInfo = entry.getValue();
            FileInfo previousInfo = this.previousFileStates.get(path);
            if (previousInfo == null) {
                changes.add(new LogEntry(runStartTime, currentInfo.creationTime, currentInfo.modificationTime, currentInfo.size, "CR", path));
                continue;
            }
            if (currentInfo.modificationTime.equals(previousInfo.modificationTime) && currentInfo.size == previousInfo.size) continue;
            changes.add(new LogEntry(runStartTime, currentInfo.creationTime, currentInfo.modificationTime, currentInfo.size, "MO", path));
        }
        for (Map.Entry<String, FileInfo> entry : this.previousFileStates.entrySet()) {
            path = entry.getKey();
            if (currentStates.containsKey(path)) continue;
            changes.add(new LogEntry(runStartTime, now, now, 0L, "DE", path));
        }
        return changes;
    }

    private void writeOperationLog(List<LogEntry> changes) throws IOException {
        if (changes.isEmpty()) {
            return;
        }
        Files.createDirectories(this.operationLogPath.getParent(), new FileAttribute[0]);
        try (BufferedWriter writer = Files.newBufferedWriter(this.operationLogPath, StandardOpenOption.CREATE, StandardOpenOption.APPEND);){
            for (LogEntry change : changes) {
                writer.write(this.formatLogEntry(change));
                writer.newLine();
            }
        }
    }

    private void writeLastUpdateLog(Instant runStartTime) throws IOException {
        Files.createDirectories(this.lastUpdateLogPath.getParent(), new FileAttribute[0]);
        String timestamp = LocalDateTime.ofInstant(runStartTime, ZoneId.systemDefault()).format(TIMESTAMP_FORMAT);
        try (BufferedWriter writer = Files.newBufferedWriter(this.lastUpdateLogPath, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);){
            writer.write(timestamp);
        }
    }

    private String formatLogEntry(LogEntry entry) {
        String creationTime = LocalDateTime.ofInstant(entry.creationTime, ZoneId.systemDefault()).format(TIMESTAMP_FORMAT);
        String modificationTime = LocalDateTime.ofInstant(entry.modificationTime, ZoneId.systemDefault()).format(TIMESTAMP_FORMAT);
        String runStartTime = LocalDateTime.ofInstant(entry.runStartTime, ZoneId.systemDefault()).format(TIMESTAMP_FORMAT);
        return String.format("%s|%s|%s|%d|%s|%s", runStartTime, creationTime, modificationTime, entry.size, entry.operation, entry.relativePath);
    }

    private LogEntry parseLogEntry(String line) {
        String[] parts = line.split("\\|", 6);
        if (parts.length != 6 && parts.length != 5) {
            return null;
        }
        try {
            Instant runStartTime = LocalDateTime.parse(parts[0], TIMESTAMP_FORMAT).atZone(ZoneId.systemDefault()).toInstant();
            Instant creationTime = LocalDateTime.parse(parts[1], TIMESTAMP_FORMAT).atZone(ZoneId.systemDefault()).toInstant();
            Instant modificationTime = LocalDateTime.parse(parts[2], TIMESTAMP_FORMAT).atZone(ZoneId.systemDefault()).toInstant();
            long size = Long.parseLong(parts[3]);
            String operation = parts[4];
            String relativePath = parts[5];
            return new LogEntry(runStartTime, creationTime, modificationTime, size, operation, relativePath);
        }
        catch (Exception e) {
            return null;
        }
    }

    public Optional<Instant> getLastUpdateTime() {
        if (!Files.exists(this.lastUpdateLogPath, new LinkOption[0])) {
            return Optional.empty();
        }
        try {
            String content = FileUtils.readFile(this.lastUpdateLogPath).trim();
            if (content.isEmpty()) {
                return Optional.empty();
            }
            LocalDateTime dateTime = LocalDateTime.parse(content, TIMESTAMP_FORMAT);
            return Optional.of(dateTime.atZone(ZoneId.systemDefault()).toInstant());
        }
        catch (Exception e) {
            return Optional.empty();
        }
    }

    private static class SyncDecision {
        final SyncAction action;
        final LogEntry sourceEntry;

        SyncDecision(SyncAction action, LogEntry sourceEntry) {
            this.action = action;
            this.sourceEntry = sourceEntry;
        }
    }

    private static class FileInfo {
        final Instant creationTime;
        final Instant modificationTime;
        final long size;

        FileInfo(Instant creationTime, Instant modificationTime, long size) {
            this.creationTime = creationTime;
            this.modificationTime = modificationTime;
            this.size = size;
        }

        public boolean equals(Object o) {
            if (!(o instanceof FileInfo)) {
                return false;
            }
            FileInfo fileInfo = (FileInfo)o;
            return this.size == fileInfo.size && Objects.equals(this.creationTime, fileInfo.creationTime) && Objects.equals(this.modificationTime, fileInfo.modificationTime);
        }

        public int hashCode() {
            return Objects.hash(this.creationTime, this.modificationTime, this.size);
        }
    }
}

