/*
 * Decompiled with CFR 0.152.
 */
package org.kendar.sync.lib.utils;

import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.PathMatcher;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.DosFileAttributes;
import java.nio.file.attribute.FileTime;
import java.nio.file.attribute.PosixFileAttributes;
import java.nio.file.attribute.PosixFilePermission;
import java.text.SimpleDateFormat;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Date;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.kendar.sync.lib.model.FileInfo;
import org.kendar.sync.lib.protocol.BackupType;
import org.kendar.sync.lib.utils.Attributes;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileUtils {
    private static final Logger log = LoggerFactory.getLogger(FileUtils.class);
    private static ConcurrentHashMap<String, Pattern> regexPatterns = new ConcurrentHashMap();
    private static ConcurrentHashMap<String, PathMatcher> globPatterns = new ConcurrentHashMap();

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static String readFile(String filename) throws IOException {
        String content = null;
        File file = new File(filename);
        try (FileReader reader = null;){
            reader = new FileReader(file);
            char[] chars = new char[(int)file.length()];
            reader.read(chars);
            content = new String(chars);
            reader.close();
        }
        return content;
    }

    public static String readFile(Path lastCompactLogPath) throws IOException {
        return FileUtils.readFile(lastCompactLogPath.toAbsolutePath().normalize().toString());
    }

    public static List<FileInfo> listFiles(File directory, String baseDir) throws IOException {
        ArrayList<FileInfo> files = new ArrayList<FileInfo>();
        FileUtils.listFilesRecursive(directory, baseDir, files);
        return files;
    }

    private static void listFilesRecursive(File directory, String baseDir, List<FileInfo> files) throws IOException {
        if (!directory.exists() || !directory.isDirectory()) {
            return;
        }
        File[] fileList = directory.listFiles();
        if (fileList == null) {
            return;
        }
        for (File file : fileList) {
            FileInfo fileInfo = FileInfo.fromFile(file, baseDir);
            if (file.isDirectory()) {
                FileUtils.listFilesRecursive(file, baseDir, files);
                continue;
            }
            files.add(fileInfo);
        }
    }

    public static String makeUniformPath(String path) {
        String res = path.replaceAll("\\\\", "/");
        if (res.startsWith("/")) {
            return res.substring(1);
        }
        return res;
    }

    public static Map<String, List<FileInfo>> calculateFileDifferences(List<FileInfo> sourceFiles, List<FileInfo> targetFiles, BackupType backupType) {
        HashMap<String, List<FileInfo>> result = new HashMap<String, List<FileInfo>>();
        Map<String, FileInfo> sourceMap = sourceFiles.stream().collect(Collectors.toMap(FileInfo::getRelativePath, f -> f));
        Map<String, FileInfo> targetMap = targetFiles.stream().collect(Collectors.toMap(FileInfo::getRelativePath, f -> f));
        List filesToTransfer = sourceFiles.stream().filter(sourceFile -> {
            if (Attributes.isDirectory(sourceFile.getExtendedUmask())) {
                return false;
            }
            FileInfo targetFile = (FileInfo)targetMap.get(sourceFile.getRelativePath());
            if (targetFile == null) {
                return true;
            }
            return !sourceFile.getModificationTime().equals(targetFile.getModificationTime()) || sourceFile.getSize() != targetFile.getSize();
        }).collect(Collectors.toList());
        result.put("transfer", filesToTransfer);
        if (backupType == BackupType.MIRROR) {
            List filesToDelete = targetFiles.stream().filter(targetFile -> !Attributes.isDirectory(targetFile.getExtendedUmask()) && !sourceMap.containsKey(targetFile.getRelativePath())).collect(Collectors.toList());
            result.put("delete", filesToDelete);
        } else {
            result.put("delete", new ArrayList());
        }
        return result;
    }

    public static String getTargetPath(FileInfo file, String targetDir, BackupType backupType) {
        if (backupType != BackupType.DATE_SEPARATED) {
            return targetDir;
        }
        SimpleDateFormat dtf = new SimpleDateFormat("yyyy-MM-dd");
        String dateDir = dtf.format(new Date(file.getModificationTime().toEpochMilli()));
        return Paths.get(targetDir, dateDir).toString();
    }

    public static void createDirectoryIfNotExists(File directory) throws IOException {
        if (!directory.exists() && !directory.mkdirs()) {
            throw new IOException("Failed to create directory 8: " + directory);
        }
    }

    public static void setFileTimes(File file, Instant creationTime, Instant modificationTime) throws IOException {
        Path path = file.toPath();
        Files.setAttribute(path, "creationTime", FileTime.from(creationTime), new LinkOption[0]);
        Files.setAttribute(path, "lastModifiedTime", FileTime.from(modificationTime), new LinkOption[0]);
    }

    public static void setFileTimesToEpoch(File file) throws IOException {
        Instant epoch = Instant.EPOCH;
        FileUtils.setFileTimes(file, epoch, epoch);
    }

    public static void delete(File file) throws IOException {
        File[] files;
        if (file.isDirectory() && (files = file.listFiles()) != null) {
            for (File child : files) {
                FileUtils.delete(child);
            }
        }
        if (!file.delete()) {
            throw new IOException("Failed to delete: " + file);
        }
    }

    public static void copyFile(File source, File target) throws IOException {
        FileUtils.createDirectoryIfNotExists(target.getParentFile());
        Files.copy(source.toPath(), target.toPath(), StandardCopyOption.REPLACE_EXISTING);
        BasicFileAttributes attrs = Files.readAttributes(source.toPath(), BasicFileAttributes.class, new LinkOption[0]);
        FileUtils.setFileTimes(target, attrs.creationTime().toInstant(), attrs.lastModifiedTime().toInstant());
    }

    public static byte[] readFile(File file) throws IOException {
        return Files.readAllBytes(file.toPath());
    }

    public static void writeFile(File file, byte[] data) throws IOException {
        FileUtils.createDirectoryIfNotExists(file.getParentFile());
        Files.write(file.toPath(), data, new OpenOption[0]);
    }

    public static boolean deleteDirectoryContents(Path directory) throws IOException {
        if (!Files.exists(directory, new LinkOption[0]) || !Files.isDirectory(directory, new LinkOption[0])) {
            return false;
        }
        boolean success = true;
        try (Stream<Path> paths = Files.walk(directory, new FileVisitOption[0]);){
            List filesToDelete = paths.filter(path -> !path.equals(directory)).sorted((a, b) -> -a.compareTo((Path)b)).collect(Collectors.toList());
            for (Path path2 : filesToDelete) {
                try {
                    Files.delete(path2);
                }
                catch (IOException e) {
                    success = false;
                    log.error("Failed to delete: {} - {}", (Object)path2, (Object)e.getMessage());
                }
            }
        }
        Files.delete(directory);
        return success;
    }

    public static boolean matches(String path, String pattern) {
        try {
            path = FileUtils.makeUniformPath(path);
            if (pattern.startsWith("@")) {
                regexPatterns.computeIfAbsent(pattern, p -> Pattern.compile(p.substring(1)));
                return regexPatterns.get(pattern).matcher(path).matches();
            }
            globPatterns.computeIfAbsent(pattern, p -> FileSystems.getDefault().getPathMatcher("glob:" + p));
            return globPatterns.get(pattern).matches(Path.of(path, new String[0]));
        }
        catch (Exception e) {
            return false;
        }
    }

    public static Attributes readFileAttributes(Path path) throws IOException {
        BasicFileAttributes attributes = Files.readAttributes(path, BasicFileAttributes.class, new LinkOption[0]);
        int umask = 0;
        File file = path.toFile();
        if (file.isHidden()) {
            umask |= 0x1000;
        }
        if (file.isDirectory()) {
            umask |= 0x8000;
        }
        if (attributes.isSymbolicLink()) {
            umask |= 0x2000;
        }
        if (attributes instanceof PosixFileAttributes) {
            PosixFileAttributes posixAttributes = (PosixFileAttributes)attributes;
            for (PosixFilePermission permission : posixAttributes.permissions()) {
                switch (permission) {
                    case OWNER_READ: {
                        umask |= 0x400;
                        break;
                    }
                    case OWNER_WRITE: {
                        umask |= 0x200;
                        break;
                    }
                    case OWNER_EXECUTE: {
                        umask |= 0x100;
                        break;
                    }
                    case GROUP_READ: {
                        umask |= 0x40;
                        break;
                    }
                    case GROUP_WRITE: {
                        umask |= 0x20;
                        break;
                    }
                    case GROUP_EXECUTE: {
                        umask |= 0x10;
                        break;
                    }
                    case OTHERS_READ: {
                        umask |= 4;
                        break;
                    }
                    case OTHERS_WRITE: {
                        umask |= 2;
                        break;
                    }
                    case OTHERS_EXECUTE: {
                        umask |= 1;
                    }
                }
            }
        } else if (attributes instanceof DosFileAttributes) {
            DosFileAttributes dosAttributes = (DosFileAttributes)attributes;
            if (dosAttributes.isSystem()) {
                umask |= 0x4000;
            }
            if (file.canExecute()) {
                umask |= 1;
            }
            if (file.canRead()) {
                umask |= 0x111;
            }
            if (file.isDirectory()) {
                umask |= 0;
            }
            if (file.canWrite()) {
                umask |= 2;
            }
        }
        return new Attributes(umask, attributes.creationTime().toInstant(), attributes.lastModifiedTime().toInstant(), attributes.size());
    }

    public static void writeFileAttributes(Path path, int umask, BasicFileAttributes attributes) throws IOException {
        File file = path.toFile();
        if (attributes instanceof PosixFileAttributes) {
            PosixFileAttributes posixAttributes = (PosixFileAttributes)attributes;
            EnumSet<PosixFilePermission> permissions = EnumSet.noneOf(PosixFilePermission.class);
            if ((umask & 0x400) != 0) {
                permissions.add(PosixFilePermission.OWNER_READ);
            }
            if ((umask & 0x200) != 0) {
                permissions.add(PosixFilePermission.OWNER_WRITE);
            }
            if ((umask & 0x100) != 0) {
                permissions.add(PosixFilePermission.OWNER_EXECUTE);
            }
            if ((umask & 0x40) != 0) {
                permissions.add(PosixFilePermission.GROUP_READ);
            }
            if ((umask & 0x20) != 0) {
                permissions.add(PosixFilePermission.GROUP_WRITE);
            }
            if ((umask & 0x10) != 0) {
                permissions.add(PosixFilePermission.GROUP_EXECUTE);
            }
            if ((umask & 4) != 0) {
                permissions.add(PosixFilePermission.OTHERS_READ);
            }
            if ((umask & 2) != 0) {
                permissions.add(PosixFilePermission.OTHERS_WRITE);
            }
            if ((umask & 1) != 0) {
                permissions.add(PosixFilePermission.OTHERS_EXECUTE);
            }
            Files.setPosixFilePermissions(path, permissions);
        } else if (attributes instanceof DosFileAttributes) {
            DosFileAttributes dosAttributes = (DosFileAttributes)attributes;
            if ((umask & 0x4000) != 0) {
                Files.setAttribute(path, "dos:system", true, new LinkOption[0]);
            }
            if ((umask & 0x1000) != 0) {
                Files.setAttribute(path, "dos:hidden", true, new LinkOption[0]);
            }
            if ((umask & 0x222) == 0) {
                file.setReadOnly();
            }
        }
    }
}

