/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak;

import java.security.PublicKey;
import java.util.Arrays;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.crypto.SecretKey;
import org.keycloak.common.VerificationException;
import org.keycloak.exceptions.TokenNotActiveException;
import org.keycloak.exceptions.TokenSignatureInvalidException;
import org.keycloak.jose.jws.AlgorithmType;
import org.keycloak.jose.jws.JWSHeader;
import org.keycloak.jose.jws.JWSInput;
import org.keycloak.jose.jws.JWSInputException;
import org.keycloak.jose.jws.crypto.HMACProvider;
import org.keycloak.jose.jws.crypto.RSAProvider;
import org.keycloak.representations.JsonWebToken;

public class TokenVerifier<T extends JsonWebToken> {
    private static final Logger LOG = Logger.getLogger(TokenVerifier.class.getName());
    public static final Predicate<JsonWebToken> SUBJECT_EXISTS_CHECK = new Predicate<JsonWebToken>(){

        @Override
        public boolean test(JsonWebToken t) throws VerificationException {
            String subject = t.getSubject();
            if (subject == null) {
                throw new VerificationException("Subject missing in token");
            }
            return true;
        }
    };
    public static final Predicate<JsonWebToken> IS_ACTIVE = new Predicate<JsonWebToken>(){

        @Override
        public boolean test(JsonWebToken t) throws VerificationException {
            if (!t.isActive()) {
                throw new TokenNotActiveException(t, "Token is not active");
            }
            return true;
        }
    };
    private String tokenString;
    private Class<? extends T> clazz;
    private PublicKey publicKey;
    private SecretKey secretKey;
    private String realmUrl;
    private String expectedTokenType = "Bearer";
    private boolean checkTokenType = true;
    private boolean checkRealmUrl = true;
    private final LinkedList<Predicate<? super T>> checks = new LinkedList();
    private JWSInput jws;
    private T token;

    protected TokenVerifier(String tokenString, Class<T> clazz) {
        this.tokenString = tokenString;
        this.clazz = clazz;
    }

    protected TokenVerifier(T token) {
        this.token = token;
    }

    public static <T extends JsonWebToken> TokenVerifier<T> create(String tokenString, Class<T> clazz) {
        return new TokenVerifier<T>(tokenString, clazz);
    }

    public static <T extends JsonWebToken> TokenVerifier<T> create(T token) {
        return new TokenVerifier<T>(token);
    }

    public TokenVerifier<T> withDefaultChecks() {
        return this.withChecks(RealmUrlCheck.NULL_INSTANCE, SUBJECT_EXISTS_CHECK, TokenTypeCheck.INSTANCE_BEARER, IS_ACTIVE);
    }

    private void removeCheck(Class<? extends Predicate<?>> checkClass) {
        Iterator it = this.checks.iterator();
        while (it.hasNext()) {
            if (((Predicate)it.next()).getClass() != checkClass) continue;
            it.remove();
        }
    }

    private void removeCheck(Predicate<? super T> check) {
        this.checks.remove(check);
    }

    private <P extends Predicate<? super T>> TokenVerifier<T> replaceCheck(Class<? extends Predicate<?>> checkClass, boolean active, P predicate) {
        this.removeCheck(checkClass);
        if (active) {
            this.checks.add(predicate);
        }
        return this;
    }

    private <P extends Predicate<? super T>> TokenVerifier<T> replaceCheck(Predicate<? super T> check, boolean active, P predicate) {
        this.removeCheck(check);
        if (active) {
            this.checks.add(predicate);
        }
        return this;
    }

    public TokenVerifier<T> withChecks(Predicate<? super T> ... checks) {
        if (checks != null) {
            this.checks.addAll(Arrays.asList(checks));
        }
        return this;
    }

    public TokenVerifier<T> publicKey(PublicKey publicKey) {
        this.publicKey = publicKey;
        return this;
    }

    public TokenVerifier<T> secretKey(SecretKey secretKey) {
        this.secretKey = secretKey;
        return this;
    }

    public TokenVerifier<T> realmUrl(String realmUrl) {
        this.realmUrl = realmUrl;
        return this.replaceCheck(RealmUrlCheck.class, this.checkRealmUrl, new RealmUrlCheck(realmUrl));
    }

    public TokenVerifier<T> checkTokenType(boolean checkTokenType) {
        this.checkTokenType = checkTokenType;
        return this.replaceCheck(TokenTypeCheck.class, this.checkTokenType, new TokenTypeCheck(this.expectedTokenType));
    }

    public TokenVerifier<T> tokenType(String tokenType) {
        this.expectedTokenType = tokenType;
        return this.replaceCheck(TokenTypeCheck.class, this.checkTokenType, new TokenTypeCheck(this.expectedTokenType));
    }

    public TokenVerifier<T> checkActive(boolean checkActive) {
        return this.replaceCheck(IS_ACTIVE, checkActive, IS_ACTIVE);
    }

    public TokenVerifier<T> checkRealmUrl(boolean checkRealmUrl) {
        this.checkRealmUrl = checkRealmUrl;
        return this.replaceCheck(RealmUrlCheck.class, this.checkRealmUrl, new RealmUrlCheck(this.realmUrl));
    }

    public TokenVerifier<T> parse() throws VerificationException {
        if (this.jws == null) {
            if (this.tokenString == null) {
                throw new VerificationException("Token not set");
            }
            try {
                this.jws = new JWSInput(this.tokenString);
            }
            catch (JWSInputException e) {
                throw new VerificationException("Failed to parse JWT", e);
            }
            try {
                this.token = (JsonWebToken)this.jws.readJsonContent(this.clazz);
            }
            catch (JWSInputException e) {
                throw new VerificationException("Failed to read access token from JWT", e);
            }
        }
        return this;
    }

    public T getToken() throws VerificationException {
        if (this.token == null) {
            this.parse();
        }
        return this.token;
    }

    public JWSHeader getHeader() throws VerificationException {
        this.parse();
        return this.jws.getHeader();
    }

    public void verifySignature() throws VerificationException {
        AlgorithmType algorithmType = this.getHeader().getAlgorithm().getType();
        if (null == algorithmType) {
            throw new VerificationException("Unknown or unsupported token algorithm");
        }
        switch (algorithmType) {
            case RSA: {
                if (this.publicKey == null) {
                    throw new VerificationException("Public key not set");
                }
                if (RSAProvider.verify(this.jws, this.publicKey)) break;
                throw new TokenSignatureInvalidException((JsonWebToken)this.token, "Invalid token signature");
            }
            case HMAC: {
                if (this.secretKey == null) {
                    throw new VerificationException("Secret key not set");
                }
                if (HMACProvider.verify(this.jws, this.secretKey)) break;
                throw new TokenSignatureInvalidException((JsonWebToken)this.token, "Invalid token signature");
            }
            default: {
                throw new VerificationException("Unknown or unsupported token algorithm");
            }
        }
    }

    public TokenVerifier<T> verify() throws VerificationException {
        if (this.getToken() == null) {
            this.parse();
        }
        if (this.jws != null) {
            this.verifySignature();
        }
        for (Predicate predicate : this.checks) {
            if (predicate.test(this.getToken())) continue;
            throw new VerificationException("JWT check failed for check " + predicate);
        }
        return this;
    }

    public static <T extends JsonWebToken> Predicate<T> optional(final Predicate<T> mandatoryPredicate) {
        return new Predicate<T>(){

            @Override
            public boolean test(T t) throws VerificationException {
                try {
                    if (!mandatoryPredicate.test(t)) {
                        LOG.finer("[optional] predicate failed: " + mandatoryPredicate);
                    }
                    return true;
                }
                catch (VerificationException ex) {
                    LOG.log(Level.FINER, "[optional] predicate " + mandatoryPredicate + " failed.", ex);
                    return true;
                }
            }
        };
    }

    public static <T extends JsonWebToken> Predicate<T> alternative(final Predicate<? super T> ... predicates) {
        return new Predicate<T>(){

            @Override
            public boolean test(T t) throws VerificationException {
                for (Predicate predicate : predicates) {
                    try {
                        if (predicate.test(t)) {
                            return true;
                        }
                        LOG.finer("[alternative] predicate failed: " + predicate);
                    }
                    catch (VerificationException ex) {
                        LOG.log(Level.FINER, "[alternative] predicate " + predicate + " failed.", ex);
                    }
                }
                return false;
            }
        };
    }

    public static class TokenTypeCheck
    implements Predicate<JsonWebToken> {
        private static final TokenTypeCheck INSTANCE_BEARER = new TokenTypeCheck("Bearer");
        private final String tokenType;

        public TokenTypeCheck(String tokenType) {
            this.tokenType = tokenType;
        }

        @Override
        public boolean test(JsonWebToken t) throws VerificationException {
            if (!this.tokenType.equalsIgnoreCase(t.getType())) {
                throw new VerificationException("Token type is incorrect. Expected '" + this.tokenType + "' but was '" + t.getType() + "'");
            }
            return true;
        }
    }

    public static class RealmUrlCheck
    implements Predicate<JsonWebToken> {
        private static final RealmUrlCheck NULL_INSTANCE = new RealmUrlCheck(null);
        private final String realmUrl;

        public RealmUrlCheck(String realmUrl) {
            this.realmUrl = realmUrl;
        }

        @Override
        public boolean test(JsonWebToken t) throws VerificationException {
            if (this.realmUrl == null) {
                throw new VerificationException("Realm URL not set");
            }
            if (!this.realmUrl.equals(t.getIssuer())) {
                throw new VerificationException("Invalid token issuer. Expected '" + this.realmUrl + "', but was '" + t.getIssuer() + "'");
            }
            return true;
        }
    }

    public static interface Predicate<T extends JsonWebToken> {
        public boolean test(T var1) throws VerificationException;
    }
}

