/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak.services.clientpolicy.executor;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.text.SimpleDateFormat;
import java.util.Objects;
import org.jboss.logging.Logger;
import org.keycloak.common.util.Time;
import org.keycloak.models.ClientModel;
import org.keycloak.models.KeycloakSession;
import org.keycloak.models.utils.KeycloakModelUtils;
import org.keycloak.protocol.oidc.OIDCClientSecretConfigWrapper;
import org.keycloak.representations.idm.ClientPolicyExecutorConfigurationRepresentation;
import org.keycloak.services.clientpolicy.ClientPolicyContext;
import org.keycloak.services.clientpolicy.ClientPolicyException;
import org.keycloak.services.clientpolicy.context.ClientCRUDContext;
import org.keycloak.services.clientpolicy.context.ClientSecretRotationContext;
import org.keycloak.services.clientpolicy.context.DynamicClientUpdatedContext;
import org.keycloak.services.clientpolicy.executor.ClientPolicyExecutorProvider;
import org.keycloak.services.clientpolicy.executor.ClientSecretRotationExecutorFactory;

public class ClientSecretRotationExecutor
implements ClientPolicyExecutorProvider<Configuration> {
    private static final Logger logger = Logger.getLogger(ClientSecretRotationExecutor.class);
    private final KeycloakSession session;
    private Configuration configuration;

    public ClientSecretRotationExecutor(KeycloakSession session) {
        this.session = session;
    }

    public String getProviderId() {
        return "secret-rotation";
    }

    public Class<Configuration> getExecutorConfigurationClass() {
        return Configuration.class;
    }

    public void executeOnEvent(ClientPolicyContext context) throws ClientPolicyException {
        if (!this.session.getContext().getClient().isPublicClient() && !this.session.getContext().getClient().isBearerOnly()) {
            this.session.setAttribute("client.secret.rotation.enabled", (Object)Boolean.TRUE);
            switch (context.getEvent()) {
                case REGISTERED: 
                case UPDATED: {
                    this.executeOnClientCreateOrUpdate((ClientCRUDContext)context);
                    break;
                }
                case AUTHORIZATION_REQUEST: 
                case TOKEN_REQUEST: {
                    this.executeOnAuthRequest();
                    return;
                }
                default: {
                    return;
                }
            }
        }
    }

    public void setupConfiguration(Configuration config) {
        this.configuration = config == null ? new Configuration().parseWithDefaultValues() : config.parseWithDefaultValues();
    }

    private void executeOnAuthRequest() {
        ClientModel client = this.session.getContext().getClient();
        OIDCClientSecretConfigWrapper wrapper = OIDCClientSecretConfigWrapper.fromClientModel(client);
        if (!wrapper.hasClientSecretExpirationTime()) {
            this.updatedSecretExpiration(wrapper);
        }
    }

    private void executeOnClientCreateOrUpdate(ClientCRUDContext adminContext) {
        OIDCClientSecretConfigWrapper clientConfigWrapper = OIDCClientSecretConfigWrapper.fromClientModel(adminContext.getTargetClient());
        logger.debugv("Executing policy {0} for client {1}-{2} with configuration [ expirationPeriod: {3}, rotatedExpirationPeriod: {4}, remainExpirationPeriod: {5} ]", new Object[]{this.getName(), clientConfigWrapper.getId(), clientConfigWrapper.getName(), this.configuration.getExpirationPeriod(), this.configuration.getRotatedExpirationPeriod(), this.configuration.getRemainExpirationPeriod()});
        if (adminContext instanceof ClientSecretRotationContext || clientConfigWrapper.isClientSecretExpired() || !clientConfigWrapper.hasClientSecretExpirationTime()) {
            this.rotateSecret(adminContext, clientConfigWrapper);
        } else if (adminContext instanceof DynamicClientUpdatedContext) {
            int startRemainingWindow = clientConfigWrapper.getClientSecretExpirationTime() - this.configuration.remainExpirationPeriod;
            this.debugDynamicInfo(clientConfigWrapper, startRemainingWindow);
            if (Time.currentTime() >= startRemainingWindow) {
                logger.debugv("Executing rotation for the dynamic client {0} due to remaining expiration time that starts at {1}", (Object)adminContext.getTargetClient().getClientId(), (Object)Time.toDate((int)startRemainingWindow));
                this.rotateSecret(adminContext, clientConfigWrapper);
            }
        }
    }

    private void debugDynamicInfo(OIDCClientSecretConfigWrapper clientConfigWrapper, int startRemainingWindow) {
        if (logger.isDebugEnabled()) {
            SimpleDateFormat sdf = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");
            logger.debugv("client expiration time: {0}, remaining time: {1}, current time: {2}, Time offset: {3}", new Object[]{clientConfigWrapper.getClientSecretExpirationTime(), startRemainingWindow, Time.currentTime(), Time.getOffset()});
            logger.debugv("client expiration date: {0}, window remaining date: {1}, current date: {2}", (Object)sdf.format(Time.toDate((int)clientConfigWrapper.getClientSecretExpirationTime())), (Object)sdf.format(Time.toDate((int)startRemainingWindow)), (Object)sdf.format(Time.toDate((int)Time.currentTime())));
        }
    }

    private void rotateSecret(ClientCRUDContext crudContext, OIDCClientSecretConfigWrapper clientConfigWrapper) {
        if (crudContext instanceof ClientSecretRotationContext) {
            ClientSecretRotationContext secretRotationContext = (ClientSecretRotationContext)crudContext;
            if (secretRotationContext.isForceRotation()) {
                logger.debugv("Force rotation for client {0}", (Object)clientConfigWrapper.getId());
                this.updateRotateSecret(clientConfigWrapper, secretRotationContext.getCurrentSecret());
                this.updateClientConfigProperties(clientConfigWrapper);
            }
        } else if (!clientConfigWrapper.hasClientSecretExpirationTime()) {
            logger.debugv("client {0} has no secret rotation expiration time configured", (Object)clientConfigWrapper.getId());
            this.updatedSecretExpiration(clientConfigWrapper);
        } else {
            logger.debugv("Execute typical secret rotation for client {0}", (Object)clientConfigWrapper.getId());
            this.updatedSecretExpiration(clientConfigWrapper);
            this.updateRotateSecret(clientConfigWrapper, clientConfigWrapper.getSecret());
            KeycloakModelUtils.generateSecret((ClientModel)crudContext.getTargetClient());
            this.updateClientConfigProperties(clientConfigWrapper);
        }
        if (Objects.nonNull(crudContext.getProposedClientRepresentation())) {
            clientConfigWrapper.updateClientRepresentationAttributes(crudContext.getProposedClientRepresentation());
        }
        logger.debugv("Client configured: {0}", (Object)clientConfigWrapper.toJson());
    }

    private void updatedSecretExpiration(OIDCClientSecretConfigWrapper clientConfigWrapper) {
        clientConfigWrapper.setClientSecretExpirationTime(Time.currentTime() + this.configuration.getExpirationPeriod());
        logger.debugv("A new secret expiration is configured for client {0}. Expires at {1}", (Object)clientConfigWrapper.getId(), (Object)Time.toDate((int)clientConfigWrapper.getClientSecretExpirationTime()));
    }

    private void updateClientConfigProperties(OIDCClientSecretConfigWrapper clientConfigWrapper) {
        clientConfigWrapper.setClientSecretCreationTime(Time.currentTime());
        this.updatedSecretExpiration(clientConfigWrapper);
    }

    private void updateRotateSecret(OIDCClientSecretConfigWrapper clientConfigWrapper, String secret) {
        if (this.configuration.rotatedExpirationPeriod > 0) {
            clientConfigWrapper.setClientRotatedSecret(secret);
            clientConfigWrapper.setClientRotatedSecretCreationTime();
            clientConfigWrapper.setClientRotatedSecretExpirationTime(Time.currentTime() + this.configuration.getRotatedExpirationPeriod());
            logger.debugv("Rotating the secret for client {0}. Secret creation at {1}. Secret expiration at {2}", (Object)clientConfigWrapper.getId(), (Object)Time.toDate((int)clientConfigWrapper.getClientRotatedSecretCreationTime()), (Object)Time.toDate((int)clientConfigWrapper.getClientRotatedSecretExpirationTime()));
        } else {
            logger.debugv("Removing rotation for client {0}", (Object)clientConfigWrapper.getId());
            clientConfigWrapper.setClientRotatedSecret(null);
            clientConfigWrapper.setClientRotatedSecretCreationTime(null);
            clientConfigWrapper.setClientRotatedSecretExpirationTime(null);
        }
    }

    public static class Configuration
    extends ClientPolicyExecutorConfigurationRepresentation {
        @JsonProperty(value="expiration-period")
        protected Integer expirationPeriod;
        @JsonProperty(value="remaining-rotation-period")
        protected Integer remainExpirationPeriod;
        @JsonProperty(value="rotated-expiration-period")
        private Integer rotatedExpirationPeriod;

        public boolean validateConfig() {
            logger.debugv("Validating configuration: [ expirationPeriod: {0}, rotatedExpirationPeriod: {1}, remainExpirationPeriod: {2} ]", (Object)this.expirationPeriod, (Object)this.rotatedExpirationPeriod, (Object)this.remainExpirationPeriod);
            if (this.expirationPeriod <= 0) {
                return false;
            }
            if (this.rotatedExpirationPeriod > this.expirationPeriod) {
                return false;
            }
            return this.remainExpirationPeriod <= this.expirationPeriod;
        }

        public Integer getExpirationPeriod() {
            return this.expirationPeriod;
        }

        public void setExpirationPeriod(Integer expirationPeriod) {
            this.expirationPeriod = expirationPeriod;
        }

        public Integer getRemainExpirationPeriod() {
            return this.remainExpirationPeriod;
        }

        public void setRemainExpirationPeriod(Integer remainExpirationPeriod) {
            this.remainExpirationPeriod = remainExpirationPeriod;
        }

        public Integer getRotatedExpirationPeriod() {
            return this.rotatedExpirationPeriod;
        }

        public void setRotatedExpirationPeriod(Integer rotatedExpirationPeriod) {
            this.rotatedExpirationPeriod = rotatedExpirationPeriod;
        }

        public Configuration parseWithDefaultValues() {
            if (this.getExpirationPeriod() == null) {
                this.setExpirationPeriod(ClientSecretRotationExecutorFactory.DEFAULT_SECRET_EXPIRATION_PERIOD);
            }
            if (this.getRemainExpirationPeriod() == null) {
                this.setRemainExpirationPeriod(ClientSecretRotationExecutorFactory.DEFAULT_SECRET_REMAINING_ROTATION_PERIOD);
            }
            if (this.getRotatedExpirationPeriod() == null) {
                this.setRotatedExpirationPeriod(ClientSecretRotationExecutorFactory.DEFAULT_SECRET_ROTATED_EXPIRATION_PERIOD);
            }
            return this;
        }
    }
}

