/*
 * Decompiled with CFR 0.152.
 */
package org.infinispan.server.core.backup;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;
import org.infinispan.commons.CacheException;
import org.infinispan.commons.util.concurrent.CompletableFutures;
import org.infinispan.configuration.parsing.ParserRegistry;
import org.infinispan.lock.EmbeddedClusteredLockManagerFactory;
import org.infinispan.lock.api.ClusteredLock;
import org.infinispan.lock.api.ClusteredLockManager;
import org.infinispan.manager.DefaultCacheManager;
import org.infinispan.manager.EmbeddedCacheManager;
import org.infinispan.security.AuthorizationPermission;
import org.infinispan.security.Security;
import org.infinispan.server.core.BackupManager;
import org.infinispan.server.core.backup.BackupManagerResources;
import org.infinispan.server.core.backup.BackupReader;
import org.infinispan.server.core.backup.BackupWriter;
import org.infinispan.server.core.backup.SecurityActions;
import org.infinispan.server.core.logging.Log;
import org.infinispan.util.concurrent.BlockingManager;
import org.infinispan.util.concurrent.CompletionStages;
import org.infinispan.util.logging.LogFactory;

public class BackupManagerImpl
implements BackupManager {
    private static final Log log = (Log)LogFactory.getLog(BackupManagerImpl.class, Log.class);
    final ParserRegistry parserRegistry;
    final BlockingManager blockingManager;
    final Path rootDir;
    final BackupReader reader;
    final Lock backupLock;
    final Lock restoreLock;
    private final EmbeddedCacheManager cacheManager;
    final Map<String, DefaultCacheManager> cacheManagers;
    final Map<String, BackupRequest> backupMap;
    final Map<String, CompletionStage<Void>> restoreMap;

    public BackupManagerImpl(BlockingManager blockingManager, DefaultCacheManager cm, Path dataRoot) {
        this.blockingManager = blockingManager;
        this.rootDir = dataRoot.resolve("backups");
        this.cacheManager = cm;
        this.cacheManagers = Collections.singletonMap(cm.getName(), cm);
        this.parserRegistry = new ParserRegistry();
        this.reader = new BackupReader(blockingManager, this.cacheManagers, this.parserRegistry);
        this.backupLock = new Lock("backup", (EmbeddedCacheManager)cm);
        this.restoreLock = new Lock("restore", (EmbeddedCacheManager)cm);
        this.backupMap = new ConcurrentHashMap<String, BackupRequest>();
        this.restoreMap = new ConcurrentHashMap<String, CompletionStage<Void>>();
    }

    @Override
    public void init() throws IOException {
        Files.createDirectories(this.rootDir, new FileAttribute[0]);
    }

    @Override
    public Set<String> getBackupNames() {
        SecurityActions.checkPermission(this.cacheManager.withSubject(Security.getSubject()), AuthorizationPermission.ADMIN);
        return new HashSet<String>(this.backupMap.keySet());
    }

    @Override
    public BackupManager.Status getBackupStatus(String name) {
        SecurityActions.checkPermission(this.cacheManager.withSubject(Security.getSubject()), AuthorizationPermission.ADMIN);
        BackupManager.Status status = this.getBackupStatus(this.backupMap.get(name));
        log.tracef("Backup status %s = %s", name, (Object)status);
        return status;
    }

    @Override
    public Path getBackupLocation(String name) {
        SecurityActions.checkPermission(this.cacheManager.withSubject(Security.getSubject()), AuthorizationPermission.ADMIN);
        BackupRequest request = this.backupMap.get(name);
        BackupManager.Status status = this.getBackupStatus(request);
        if (status != BackupManager.Status.COMPLETE) {
            return null;
        }
        return request.future.join();
    }

    private BackupManager.Status getBackupStatus(BackupRequest request) {
        CompletableFuture<Path> future = request == null ? null : request.future;
        return this.getFutureStatus(future);
    }

    private BackupManager.Status getFutureStatus(CompletionStage<?> stage) {
        if (stage == null) {
            return BackupManager.Status.NOT_FOUND;
        }
        CompletableFuture<?> future = stage.toCompletableFuture();
        if (future.isCompletedExceptionally()) {
            return BackupManager.Status.FAILED;
        }
        return future.isDone() ? BackupManager.Status.COMPLETE : BackupManager.Status.IN_PROGRESS;
    }

    @Override
    public CompletionStage<BackupManager.Status> removeBackup(String name) {
        SecurityActions.checkPermission(this.cacheManager.withSubject(Security.getSubject()), AuthorizationPermission.ADMIN);
        BackupRequest request = this.backupMap.get(name);
        BackupManager.Status status = this.getBackupStatus(request);
        switch (status) {
            case NOT_FOUND: {
                this.backupMap.remove(name);
                return CompletableFuture.completedFuture(status);
            }
            case COMPLETE: 
            case FAILED: {
                this.backupMap.remove(name);
                return this.blockingManager.supplyBlocking(() -> {
                    request.writer.cleanup();
                    return BackupManager.Status.COMPLETE;
                }, (Object)"remove-completed-backup");
            }
            case IN_PROGRESS: {
                this.blockingManager.handleBlocking(request.future, (path, t) -> {
                    request.writer.cleanup();
                    return null;
                }, (Object)"remove-inprogress-backup");
                return CompletableFuture.completedFuture(BackupManager.Status.IN_PROGRESS);
            }
        }
        throw new IllegalStateException();
    }

    @Override
    public CompletionStage<Path> create(String name, Path workingDir) {
        return this.create(name, workingDir, this.cacheManagers.entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, p -> new BackupManagerResources.Builder().includeAll().build())));
    }

    @Override
    public CompletionStage<Path> create(String name, Path workingDir, Map<String, BackupManager.Resources> params) {
        SecurityActions.checkPermission(this.cacheManager.withSubject(Security.getSubject()), AuthorizationPermission.ADMIN);
        if (this.getBackupStatus(name) != BackupManager.Status.NOT_FOUND) {
            return CompletableFutures.completedExceptionFuture((Throwable)log.backupAlreadyExists(name));
        }
        BackupWriter writer = new BackupWriter(name, this.blockingManager, this.cacheManagers, this.parserRegistry, workingDir == null ? this.rootDir : workingDir);
        CompletionStage backupStage = this.backupLock.lock().thenCompose(lockAcquired -> {
            log.tracef("Backup %s locked = %s", this.backupLock, lockAcquired);
            if (!lockAcquired.booleanValue()) {
                return CompletableFutures.completedExceptionFuture((Throwable)log.backupInProgress());
            }
            log.initiatingBackup(name);
            return writer.create(params);
        });
        backupStage = CompletionStages.handleAndCompose(backupStage, (path, t) -> {
            CompletionStage<Void> unlock = this.backupLock.unlock().thenAccept(v -> log.tracef("Backup %s unlocked", this.backupLock));
            if (t != null) {
                CacheException backupErr = log.errorCreatingBackup((Throwable)t);
                log.errorf(backupErr.getCause(), "%s:", backupErr.getMessage());
                return unlock.thenCompose(ignore -> CompletableFutures.completedExceptionFuture((Throwable)backupErr));
            }
            log.backupComplete(path.getFileName().toString());
            return unlock.thenCompose(ignore -> CompletableFuture.completedFuture(path));
        });
        this.backupMap.put(name, new BackupRequest(writer, backupStage));
        return backupStage;
    }

    @Override
    public CompletionStage<BackupManager.Status> removeRestore(String name) {
        SecurityActions.checkPermission(this.cacheManager.withSubject(Security.getSubject()), AuthorizationPermission.ADMIN);
        CompletionStage<Void> stage = this.restoreMap.remove(name);
        BackupManager.Status status = this.getFutureStatus(stage);
        return CompletableFuture.completedFuture(status);
    }

    @Override
    public BackupManager.Status getRestoreStatus(String name) {
        SecurityActions.checkPermission(this.cacheManager.withSubject(Security.getSubject()), AuthorizationPermission.ADMIN);
        return this.getFutureStatus(this.restoreMap.get(name));
    }

    @Override
    public Set<String> getRestoreNames() {
        SecurityActions.checkPermission(this.cacheManager.withSubject(Security.getSubject()), AuthorizationPermission.ADMIN);
        return new HashSet<String>(this.restoreMap.keySet());
    }

    @Override
    public CompletionStage<Void> restore(String name, Path backup) {
        return this.restore(name, backup, this.cacheManagers.entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, p -> new BackupManagerResources.Builder().includeAll().build())));
    }

    @Override
    public CompletionStage<Void> restore(String name, Path backup, Map<String, BackupManager.Resources> params) {
        SecurityActions.checkPermission(this.cacheManager.withSubject(Security.getSubject()), AuthorizationPermission.ADMIN);
        if (this.getRestoreStatus(name) != BackupManager.Status.NOT_FOUND) {
            return CompletableFutures.completedExceptionFuture((Throwable)log.restoreAlreadyExists(name));
        }
        if (!Files.exists(backup, new LinkOption[0])) {
            CacheException e = log.errorRestoringBackup(backup, new FileNotFoundException(backup.toString()));
            log.errorf(e.getCause(), "%s:", e.getMessage());
            return CompletableFutures.completedExceptionFuture((Throwable)e);
        }
        CompletionStage restoreStage = this.restoreLock.lock().thenCompose(lockAcquired -> {
            if (!lockAcquired.booleanValue()) {
                return CompletableFutures.completedExceptionFuture((Throwable)log.restoreInProgress());
            }
            log.initiatingRestore(name, backup);
            return this.reader.restore(backup, params);
        });
        restoreStage = CompletionStages.handleAndCompose(restoreStage, (path, t) -> {
            CompletionStage<Void> unlock = this.restoreLock.unlock();
            if (t != null) {
                CacheException restoreErr = log.errorRestoringBackup(backup, (Throwable)t);
                log.errorf(restoreErr.getCause(), "%s:", restoreErr.getMessage());
                return unlock.thenCompose(ignore -> CompletableFutures.completedExceptionFuture((Throwable)restoreErr));
            }
            log.restoreComplete(name);
            return unlock.thenCompose(ignore -> CompletableFuture.completedFuture(path));
        });
        this.restoreMap.put(name, restoreStage);
        return restoreStage;
    }

    static class Lock {
        final String name;
        final EmbeddedCacheManager cm;
        final boolean isClustered;
        volatile ClusteredLock clusteredLock;
        volatile AtomicBoolean localLock;

        Lock(String name, EmbeddedCacheManager cm) {
            this.name = String.format("%s-%s", BackupManagerImpl.class.getSimpleName(), name);
            this.cm = cm;
            this.isClustered = SecurityActions.getGlobalConfiguration(cm).isClustered();
        }

        CompletionStage<Boolean> lock() {
            if (this.isClustered) {
                return this.getClusteredLock().tryLock();
            }
            return CompletableFuture.completedFuture(this.getLocalLock().compareAndSet(false, true));
        }

        CompletionStage<Void> unlock() {
            if (this.isClustered) {
                return this.getClusteredLock().unlock();
            }
            this.getLocalLock().compareAndSet(true, false);
            return CompletableFutures.completedNull();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private ClusteredLock getClusteredLock() {
            if (this.clusteredLock == null) {
                Lock lock = this;
                synchronized (lock) {
                    if (this.clusteredLock == null) {
                        ClusteredLockManager lockManager = EmbeddedClusteredLockManagerFactory.from((EmbeddedCacheManager)this.cm);
                        boolean isDefined = lockManager.isDefined(this.name);
                        if (!isDefined) {
                            lockManager.defineLock(this.name);
                        }
                        this.clusteredLock = lockManager.get(this.name);
                    }
                }
            }
            return this.clusteredLock;
        }

        private AtomicBoolean getLocalLock() {
            if (this.localLock == null) {
                this.localLock = new AtomicBoolean();
            }
            return this.localLock;
        }

        public String toString() {
            return "Lock{name='" + this.name + "', isClustered=" + this.isClustered + "}";
        }
    }

    static class BackupRequest {
        final BackupWriter writer;
        final CompletableFuture<Path> future;

        BackupRequest(BackupWriter writer, CompletionStage<Path> stage) {
            this.writer = writer;
            this.future = stage.toCompletableFuture();
        }
    }
}

