/*
 * Decompiled with CFR 0.152.
 */
package org.infinispan.server.core.backup;

import java.io.IOException;
import java.io.OutputStream;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.CompletionStage;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;
import org.infinispan.commons.CacheException;
import org.infinispan.commons.configuration.io.ConfigurationWriter;
import org.infinispan.commons.util.Util;
import org.infinispan.commons.util.Version;
import org.infinispan.configuration.global.GlobalConfiguration;
import org.infinispan.configuration.parsing.ParserRegistry;
import org.infinispan.factories.GlobalComponentRegistry;
import org.infinispan.manager.DefaultCacheManager;
import org.infinispan.manager.EmbeddedCacheManager;
import org.infinispan.server.core.BackupManager;
import org.infinispan.server.core.backup.ContainerResource;
import org.infinispan.server.core.backup.SecurityActions;
import org.infinispan.server.core.backup.resources.ContainerResourceFactory;
import org.infinispan.server.core.logging.Log;
import org.infinispan.util.concurrent.AggregateCompletionStage;
import org.infinispan.util.concurrent.BlockingManager;
import org.infinispan.util.concurrent.CompletionStages;
import org.infinispan.util.logging.LogFactory;

class BackupWriter {
    private static final Log log = (Log)LogFactory.getLog(BackupWriter.class, Log.class);
    private final String name;
    private final BlockingManager blockingManager;
    private final Map<String, DefaultCacheManager> cacheManagers;
    private final ParserRegistry parserRegistry;
    private final Path rootDir;

    BackupWriter(String name, BlockingManager blockingManager, Map<String, DefaultCacheManager> cacheManagers, ParserRegistry parserRegistry, Path rootDir) {
        this.name = name;
        this.blockingManager = blockingManager;
        this.cacheManagers = cacheManagers;
        this.parserRegistry = parserRegistry;
        this.rootDir = rootDir.resolve(name);
    }

    void cleanup() {
        log.backupDeleted(this.name);
        Util.recursiveFileRemove((Path)this.rootDir);
    }

    CompletionStage<Path> create(Map<String, BackupManager.Resources> params) {
        AggregateCompletionStage stages = CompletionStages.aggregateCompletionStage();
        for (Map.Entry<String, BackupManager.Resources> e : params.entrySet()) {
            String containerName = e.getKey();
            EmbeddedCacheManager cm = (EmbeddedCacheManager)this.cacheManagers.get(containerName);
            stages.dependsOn(this.createBackup(containerName, cm, e.getValue()));
        }
        stages.dependsOn(this.writeManifest(this.cacheManagers.keySet()));
        return this.blockingManager.thenApplyBlocking(stages.freeze(), Void2 -> this.createZip(), (Object)"create");
    }

    private CompletionStage<Void> createBackup(String containerName, EmbeddedCacheManager cm, BackupManager.Resources params) {
        Path containerRoot = this.rootDir.resolve("containers").resolve(containerName);
        try {
            Files.createDirectories(containerRoot, new FileAttribute[0]);
        }
        catch (IOException e) {
            throw new CacheException("Unable to create directories " + containerRoot);
        }
        GlobalComponentRegistry gcr = SecurityActions.getGlobalComponentRegistry(cm);
        BlockingManager blockingManager = (BlockingManager)gcr.getComponent(BlockingManager.class);
        Collection<ContainerResource> resources = ContainerResourceFactory.getResources(params, blockingManager, cm, gcr, this.parserRegistry, containerRoot);
        CompletionStage prepareStage = blockingManager.runBlocking(() -> resources.forEach(ContainerResource::prepareAndValidateBackup), (Object)"backupWriter - prepare");
        return prepareStage.thenCompose(ignore -> {
            AggregateCompletionStage stages = CompletionStages.aggregateCompletionStage();
            for (ContainerResource cr : resources) {
                stages.dependsOn(cr.backup());
            }
            stages.dependsOn(blockingManager.runBlocking(() -> this.writeGlobalConfig(SecurityActions.getGlobalConfiguration(cm), containerRoot), (Object)"backupWriter - writeGlobalConfig"));
            return blockingManager.thenRunBlocking(stages.freeze(), () -> {
                Properties manifest = new Properties();
                resources.forEach(r -> r.writeToManifest(manifest));
                this.storeProperties(manifest, "Container Properties", containerRoot.resolve("container.properties"));
            }, (Object)"backupWriter - createManifest");
        });
    }

    private CompletionStage<Void> writeManifest(Set<String> containers) {
        return this.blockingManager.runBlocking(() -> {
            Properties manifest = new Properties();
            manifest.put("containers", String.join((CharSequence)",", containers));
            manifest.put("version", Version.getVersion());
            this.storeProperties(manifest, "Backup Manifest", this.rootDir.resolve("manifest.properties"));
        }, (Object)"write-manifest");
    }

    private void writeGlobalConfig(GlobalConfiguration configuration, Path root) {
        Path xmlPath = root.resolve("global.xml");
        try (ConfigurationWriter writer = ConfigurationWriter.to((OutputStream)Files.newOutputStream(xmlPath, new OpenOption[0])).clearTextSecrets(true).prettyPrint(true).build();){
            this.parserRegistry.serialize(writer, configuration, Collections.emptyMap());
        }
        catch (Exception e) {
            throw new CacheException(String.format("Unable to create global configuration file '%s'", xmlPath), (Throwable)e);
        }
    }

    private Path createZip() {
        final Path zipFile = this.rootDir.resolve(this.name + ".zip");
        try (final ZipOutputStream zs = new ZipOutputStream(Files.newOutputStream(Files.createFile(zipFile, new FileAttribute[0]), new OpenOption[0]));){
            Files.walkFileTree(this.rootDir, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult visitFile(Path path, BasicFileAttributes attrs) throws IOException {
                    if (!path.equals(zipFile)) {
                        String name = BackupWriter.this.rootDir.relativize(path).toString();
                        zs.putNextEntry(new ZipEntry(name));
                        Files.copy(path, zs);
                        zs.closeEntry();
                        Files.delete(path);
                    }
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                    if (exc != null) {
                        throw new IllegalStateException(exc);
                    }
                    if (!dir.equals(BackupWriter.this.rootDir)) {
                        Files.delete(dir);
                    }
                    return FileVisitResult.CONTINUE;
                }
            });
        }
        catch (IOException e) {
            throw new CacheException((Throwable)e);
        }
        return zipFile;
    }

    private void storeProperties(Properties properties, String description, Path dest) {
        try (OutputStream os = Files.newOutputStream(dest, new OpenOption[0]);){
            properties.store(os, description);
        }
        catch (IOException e) {
            throw new CacheException(String.format("Unable to create %s file", description), (Throwable)e);
        }
    }
}

