/*
 * Decompiled with CFR 0.152.
 */
package org.apache.activemq.artemis.core.journal.impl;

import java.security.AccessController;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.BlockingDeque;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Executor;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.activemq.artemis.api.core.ActiveMQIOErrorException;
import org.apache.activemq.artemis.core.io.SequentialFile;
import org.apache.activemq.artemis.core.io.SequentialFileFactory;
import org.apache.activemq.artemis.core.journal.impl.JournalFile;
import org.apache.activemq.artemis.core.journal.impl.JournalFileImpl;
import org.apache.activemq.artemis.core.journal.impl.JournalImpl;
import org.apache.activemq.artemis.journal.ActiveMQJournalBundle;
import org.apache.activemq.artemis.journal.ActiveMQJournalLogger;

public class JournalFilesRepository {
    private static final boolean trace = ActiveMQJournalLogger.LOGGER.isTraceEnabled();
    private static final boolean CHECK_CONSISTENCE = false;
    private final SequentialFileFactory fileFactory;
    private final JournalImpl journal;
    private final BlockingDeque<JournalFile> dataFiles = new LinkedBlockingDeque<JournalFile>();
    private final ConcurrentLinkedQueue<JournalFile> freeFiles = new ConcurrentLinkedQueue();
    private final BlockingQueue<JournalFile> openedFiles = new LinkedBlockingQueue<JournalFile>();
    private final AtomicLong nextFileID = new AtomicLong(0L);
    private final int maxAIO;
    private final int minFiles;
    private final int poolSize;
    private final int fileSize;
    private final String filePrefix;
    private final String fileExtension;
    private final int userVersion;
    private final AtomicInteger freeFilesCount = new AtomicInteger(0);
    private Executor openFilesExecutor;
    private final Runnable pushOpenRunnable = new Runnable(){

        @Override
        public void run() {
            try {
                JournalFilesRepository.this.pushOpenedFile();
            }
            catch (Exception e) {
                ActiveMQJournalLogger.LOGGER.errorPushingFile(e);
            }
        }
    };

    private static void trace(String message) {
        ActiveMQJournalLogger.LOGGER.trace(message);
    }

    public JournalFilesRepository(SequentialFileFactory fileFactory, JournalImpl journal, String filePrefix, String fileExtension, int userVersion, int maxAIO, int fileSize, int minFiles, int poolSize) {
        if (filePrefix == null) {
            throw new IllegalArgumentException("filePrefix cannot be null");
        }
        if (fileExtension == null) {
            throw new IllegalArgumentException("fileExtension cannot be null");
        }
        if (maxAIO <= 0) {
            throw new IllegalArgumentException("maxAIO must be a positive number");
        }
        this.fileFactory = fileFactory;
        this.maxAIO = maxAIO;
        this.filePrefix = filePrefix;
        this.fileExtension = fileExtension;
        this.minFiles = minFiles;
        this.fileSize = fileSize;
        this.poolSize = poolSize;
        this.userVersion = userVersion;
        this.journal = journal;
    }

    public void setExecutor(Executor fileExecutor) {
        this.openFilesExecutor = fileExecutor;
    }

    public void clear() throws Exception {
        this.dataFiles.clear();
        this.freeFiles.clear();
        this.freeFilesCount.set(0);
        for (JournalFile file : this.openedFiles) {
            try {
                file.getFile().close();
            }
            catch (Exception e) {
                ActiveMQJournalLogger.LOGGER.errorClosingFile(e);
            }
        }
        this.openedFiles.clear();
    }

    public int getMaxAIO() {
        return this.maxAIO;
    }

    public String getFileExtension() {
        return this.fileExtension;
    }

    public String getFilePrefix() {
        return this.filePrefix;
    }

    public void calculateNextfileID(List<JournalFile> files) {
        for (JournalFile file : files) {
            long fileIdFromFile = file.getFileID();
            long fileIdFromName = this.getFileNameID(file.getFile().getFileName());
            this.setNextFileID(Math.max(fileIdFromName, fileIdFromFile));
        }
    }

    public void setNextFileID(long targetUpdate) {
        long current;
        do {
            if ((current = this.nextFileID.get()) < targetUpdate) continue;
            return;
        } while (!this.nextFileID.compareAndSet(current, targetUpdate));
    }

    public void ensureMinFiles() throws Exception {
        int filesToCreate = this.minFiles - (this.dataFiles.size() + this.freeFilesCount.get());
        if (filesToCreate > 0) {
            for (int i = 0; i < filesToCreate; ++i) {
                this.freeFiles.add(this.createFile(false, false, true, false, -1L));
                this.freeFilesCount.getAndIncrement();
            }
        }
    }

    public void openFile(JournalFile file, boolean multiAIO) throws Exception {
        if (multiAIO) {
            file.getFile().open();
        } else {
            file.getFile().open(1, false);
        }
        file.getFile().position(file.getFile().calculateBlockStart(16));
    }

    public JournalFile[] getDataFilesArray() {
        return this.dataFiles.toArray(new JournalFile[this.dataFiles.size()]);
    }

    public JournalFile pollLastDataFile() {
        return (JournalFile)this.dataFiles.pollLast();
    }

    public void removeDataFile(JournalFile file) {
        if (!this.dataFiles.remove(file)) {
            ActiveMQJournalLogger.LOGGER.couldNotRemoveFile(file);
        }
    }

    public int getDataFilesCount() {
        return this.dataFiles.size();
    }

    public Collection<JournalFile> getDataFiles() {
        return this.dataFiles;
    }

    public void clearDataFiles() {
        this.dataFiles.clear();
    }

    public void addDataFileOnTop(JournalFile file) {
        this.dataFiles.addFirst(file);
    }

    public String debugFiles() {
        StringBuilder buffer = new StringBuilder();
        buffer.append("**********\nCurrent File = " + this.journal.getCurrentFile() + "\n");
        buffer.append("**********\nDataFiles:\n");
        for (JournalFile file : this.dataFiles) {
            buffer.append(file.toString() + "\n");
        }
        buffer.append("*********\nFreeFiles:\n");
        for (JournalFile file : this.freeFiles) {
            buffer.append(file.toString() + "\n");
        }
        return buffer.toString();
    }

    public synchronized void checkDataFiles() {
        long seq = -1L;
        for (JournalFile file : this.dataFiles) {
            if (file.getFileID() <= seq) {
                ActiveMQJournalLogger.LOGGER.checkFiles();
                ActiveMQJournalLogger.LOGGER.info(this.debugFiles());
                ActiveMQJournalLogger.LOGGER.seqOutOfOrder();
                throw new IllegalStateException("Sequence out of order");
            }
            if (this.journal.getCurrentFile() != null && this.journal.getCurrentFile().getFileID() <= file.getFileID()) {
                ActiveMQJournalLogger.LOGGER.checkFiles();
                ActiveMQJournalLogger.LOGGER.info(this.debugFiles());
                ActiveMQJournalLogger.LOGGER.currentFile(file.getFileID(), this.journal.getCurrentFile().getFileID(), file.getFileID(), this.journal.getCurrentFile() == file);
            }
            if (this.journal.getCurrentFile() == file) {
                throw new RuntimeException("Check failure! Current file listed as data file!");
            }
            seq = file.getFileID();
        }
        long lastFreeId = -1L;
        for (JournalFile file : this.freeFiles) {
            if (file.getFileID() <= lastFreeId) {
                ActiveMQJournalLogger.LOGGER.checkFiles();
                ActiveMQJournalLogger.LOGGER.info(this.debugFiles());
                ActiveMQJournalLogger.LOGGER.fileIdOutOfOrder();
                throw new RuntimeException("Check failure!");
            }
            lastFreeId = file.getFileID();
            if (file.getFileID() >= seq) continue;
            ActiveMQJournalLogger.LOGGER.checkFiles();
            ActiveMQJournalLogger.LOGGER.info(this.debugFiles());
            ActiveMQJournalLogger.LOGGER.fileTooSmall();
        }
    }

    public void addDataFileOnBottom(JournalFile file) {
        this.dataFiles.add(file);
    }

    public int getFreeFilesCount() {
        return this.freeFilesCount.get();
    }

    public synchronized void addFreeFile(JournalFile file, boolean renameTmp) throws Exception {
        this.addFreeFile(file, renameTmp, true);
    }

    public synchronized void addFreeFile(JournalFile file, boolean renameTmp, boolean checkDelete) throws Exception {
        long calculatedSize = 0L;
        try {
            calculatedSize = file.getFile().size();
        }
        catch (Exception e) {
            throw new IllegalStateException(e.getMessage() + " file: " + file);
        }
        if (calculatedSize != (long)this.fileSize) {
            ActiveMQJournalLogger.LOGGER.deletingFile(file);
            file.getFile().delete();
        } else if (!checkDelete || this.freeFilesCount.get() + this.dataFiles.size() + 1 + this.openedFiles.size() < this.poolSize || this.poolSize < 0) {
            if (trace) {
                JournalFilesRepository.trace("Adding free file " + file);
            }
            JournalFile jf = this.reinitializeFile(file);
            if (renameTmp) {
                jf.getFile().renameTo(JournalImpl.renameExtensionFile(jf.getFile().getFileName(), ".tmp"));
            }
            this.freeFiles.add(jf);
            this.freeFilesCount.getAndIncrement();
        } else {
            if (trace) {
                ActiveMQJournalLogger.LOGGER.trace("DataFiles.size() = " + this.dataFiles.size());
                ActiveMQJournalLogger.LOGGER.trace("openedFiles.size() = " + this.openedFiles.size());
                ActiveMQJournalLogger.LOGGER.trace("minfiles = " + this.minFiles + ", poolSize = " + this.poolSize);
                ActiveMQJournalLogger.LOGGER.trace("Free Files = " + this.freeFilesCount.get());
                ActiveMQJournalLogger.LOGGER.trace("File " + file + " being deleted as freeFiles.size() + dataFiles.size() + 1 + openedFiles.size() (" + (this.freeFilesCount.get() + this.dataFiles.size() + 1 + this.openedFiles.size()) + ") < minFiles (" + this.minFiles + ")");
            }
            file.getFile().delete();
        }
    }

    public Collection<JournalFile> getFreeFiles() {
        return this.freeFiles;
    }

    public JournalFile getFreeFile() {
        JournalFile file = (JournalFile)this.freeFiles.remove();
        this.freeFilesCount.getAndDecrement();
        return file;
    }

    public int getOpenedFilesCount() {
        return this.openedFiles.size();
    }

    public JournalFile openFile() throws InterruptedException, ActiveMQIOErrorException {
        if (trace) {
            JournalFilesRepository.trace("enqueueOpenFile with openedFiles.size=" + this.openedFiles.size());
        }
        if (this.openFilesExecutor == null) {
            this.pushOpenRunnable.run();
        } else {
            this.openFilesExecutor.execute(this.pushOpenRunnable);
        }
        JournalFile nextFile = this.openedFiles.poll(5L, TimeUnit.SECONDS);
        if (nextFile == null) {
            this.fileFactory.onIOError((Exception)ActiveMQJournalBundle.BUNDLE.fileNotOpened(), "unable to open ", null);
            this.fileFactory.activateBuffer(this.journal.getCurrentFile().getFile());
            throw ActiveMQJournalBundle.BUNDLE.fileNotOpened();
        }
        if (trace) {
            JournalFilesRepository.trace("Returning file " + nextFile);
        }
        return nextFile;
    }

    public void pushOpenedFile() throws Exception {
        JournalFile nextOpenedFile = this.takeFile(true, true, true, false);
        if (trace) {
            JournalFilesRepository.trace("pushing openFile " + nextOpenedFile);
        }
        if (!this.openedFiles.offer(nextOpenedFile)) {
            ActiveMQJournalLogger.LOGGER.failedToAddFile(nextOpenedFile);
        }
    }

    public void closeFile(JournalFile file) throws Exception {
        this.fileFactory.deactivateBuffer();
        file.getFile().close();
        if (!this.dataFiles.contains(file)) {
            this.dataFiles.add(file);
        }
    }

    public JournalFile takeFile(boolean keepOpened, boolean multiAIO, boolean initFile, boolean tmpCompactExtension) throws Exception {
        JournalFile nextFile = null;
        nextFile = this.freeFiles.poll();
        if (nextFile != null) {
            this.freeFilesCount.getAndDecrement();
        }
        if (nextFile == null) {
            nextFile = this.createFile(keepOpened, multiAIO, initFile, tmpCompactExtension, -1L);
        } else {
            if (tmpCompactExtension) {
                SequentialFile sequentialFile = nextFile.getFile();
                sequentialFile.renameTo(sequentialFile.getFileName() + ".cmp");
            }
            if (keepOpened) {
                this.openFile(nextFile, multiAIO);
            }
        }
        return nextFile;
    }

    public JournalFile createRemoteBackupSyncFile(long fileID) throws Exception {
        return this.createFile(false, false, true, false, fileID);
    }

    private JournalFile createFile(final boolean keepOpened, final boolean multiAIO, final boolean init, final boolean tmpCompact, final long fileIdPreSet) throws Exception {
        if (System.getSecurityManager() == null) {
            return this.createFile0(keepOpened, multiAIO, init, tmpCompact, fileIdPreSet);
        }
        try {
            return AccessController.doPrivileged(new PrivilegedExceptionAction<JournalFile>(){

                @Override
                public JournalFile run() throws Exception {
                    return JournalFilesRepository.this.createFile0(keepOpened, multiAIO, init, tmpCompact, fileIdPreSet);
                }
            });
        }
        catch (PrivilegedActionException e) {
            throw this.unwrapException(e);
        }
    }

    private RuntimeException unwrapException(PrivilegedActionException e) throws Exception {
        Throwable c = e.getCause();
        if (c instanceof RuntimeException) {
            throw (RuntimeException)c;
        }
        if (c instanceof Error) {
            throw (Error)c;
        }
        throw new RuntimeException(c);
    }

    private JournalFile createFile0(boolean keepOpened, boolean multiAIO, boolean init, boolean tmpCompact, long fileIdPreSet) throws Exception {
        long fileID = fileIdPreSet != -1L ? fileIdPreSet : this.generateFileID();
        String fileName = this.createFileName(tmpCompact, fileID);
        if (trace) {
            JournalFilesRepository.trace("Creating file " + fileName);
        }
        String tmpFileName = fileName + ".tmp";
        SequentialFile sequentialFile = this.fileFactory.createSequentialFile(tmpFileName);
        sequentialFile.open(1, false);
        if (init) {
            sequentialFile.fill(this.fileSize);
            JournalImpl.initFileHeader(this.fileFactory, sequentialFile, this.userVersion, fileID);
        }
        long position = sequentialFile.position();
        sequentialFile.close();
        if (trace) {
            JournalFilesRepository.trace("Renaming file " + tmpFileName + " as " + fileName);
        }
        sequentialFile.renameTo(fileName);
        if (keepOpened) {
            if (multiAIO) {
                sequentialFile.open();
            } else {
                sequentialFile.open(1, false);
            }
            sequentialFile.position(position);
        }
        return new JournalFileImpl(sequentialFile, fileID, 2);
    }

    private String createFileName(boolean tmpCompact, long fileID) {
        String fileName = tmpCompact ? this.filePrefix + "-" + fileID + "." + this.fileExtension + ".cmp" : this.filePrefix + "-" + fileID + "." + this.fileExtension;
        return fileName;
    }

    private long generateFileID() {
        return this.nextFileID.incrementAndGet();
    }

    private long getFileNameID(String fileName) {
        try {
            return Long.parseLong(fileName.substring(this.filePrefix.length() + 1, fileName.indexOf(46)));
        }
        catch (Throwable e) {
            ActiveMQJournalLogger.LOGGER.errorRetrievingID(e, fileName);
            return 0L;
        }
    }

    private JournalFile reinitializeFile(JournalFile file) throws Exception {
        long newFileID = this.generateFileID();
        SequentialFile sf = file.getFile();
        sf.open(1, false);
        int position = JournalImpl.initFileHeader(this.fileFactory, sf, this.userVersion, newFileID);
        JournalFileImpl jf = new JournalFileImpl(sf, newFileID, 2);
        sf.position(position);
        sf.close();
        return jf;
    }

    public String toString() {
        return "JournalFilesRepository(dataFiles=" + this.dataFiles + ", freeFiles=" + this.freeFiles + ", openedFiles=" + this.openedFiles + ")";
    }
}

