/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak.models;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.keycloak.models.UserCredentialValueModel;
import org.keycloak.models.UserModel;
import org.keycloak.models.utils.Pbkdf2PasswordEncoder;

public class PasswordPolicy {
    public static final String INVALID_PASSWORD_MIN_LENGTH_MESSAGE = "invalidPasswordMinLengthMessage";
    public static final String INVALID_PASSWORD_MIN_DIGITS_MESSAGE = "invalidPasswordMinDigitsMessage";
    public static final String INVALID_PASSWORD_MIN_LOWER_CASE_CHARS_MESSAGE = "invalidPasswordMinLowerCaseCharsMessage";
    public static final String INVALID_PASSWORD_MIN_UPPER_CASE_CHARS_MESSAGE = "invalidPasswordMinUpperCaseCharsMessage";
    public static final String INVALID_PASSWORD_MIN_SPECIAL_CHARS_MESSAGE = "invalidPasswordMinSpecialCharsMessage";
    public static final String INVALID_PASSWORD_NOT_USERNAME = "invalidPasswordNotUsernameMessage";
    public static final String INVALID_PASSWORD_REGEX_PATTERN = "invalidPasswordRegexPatternMessage";
    public static final String INVALID_PASSWORD_HISTORY = "invalidPasswordHistoryMessage";
    private List<Policy> policies;
    private String policyString;

    public PasswordPolicy(String policyString) {
        if (policyString == null || policyString.length() == 0) {
            this.policyString = null;
            this.policies = Collections.emptyList();
        } else {
            this.policyString = policyString;
            this.policies = PasswordPolicy.parse(policyString);
        }
    }

    private static List<Policy> parse(String policyString) {
        String[] policies;
        LinkedList<Policy> list = new LinkedList<Policy>();
        for (String policy : policies = policyString.split(" and ")) {
            String name;
            policy = policy.trim();
            String[] args = null;
            int i = policy.indexOf(40);
            if (i == -1) {
                name = policy.trim();
            } else {
                name = policy.substring(0, i).trim();
                args = policy.substring(i + 1, policy.length() - 1).split(",");
                for (int j = 0; j < args.length; ++j) {
                    args[j] = args[j].trim();
                }
            }
            if (name.equals("length")) {
                list.add(new Length(args));
                continue;
            }
            if (name.equals("digits")) {
                list.add(new Digits(args));
                continue;
            }
            if (name.equals("lowerCase")) {
                list.add(new LowerCase(args));
                continue;
            }
            if (name.equals("upperCase")) {
                list.add(new UpperCase(args));
                continue;
            }
            if (name.equals("specialChars")) {
                list.add(new SpecialChars(args));
                continue;
            }
            if (name.equals("notUsername")) {
                list.add(new NotUsername(args));
                continue;
            }
            if (name.equals("hashIterations")) {
                list.add(new HashIterations(args));
                continue;
            }
            if (name.equals("regexPatterns")) {
                for (String regexPattern : args) {
                    Pattern.compile(regexPattern);
                }
                list.add(new RegexPatterns(args));
                continue;
            }
            if (name.equals("passwordHistory")) {
                list.add(new PasswordHistory(args));
                continue;
            }
            if (!name.equals("forceExpiredPasswordChange")) continue;
            list.add(new ForceExpiredPasswordChange(args));
        }
        return list;
    }

    public int getHashIterations() {
        if (this.policies == null) {
            return -1;
        }
        for (Policy p : this.policies) {
            if (!(p instanceof HashIterations)) continue;
            return ((HashIterations)p).iterations;
        }
        return -1;
    }

    public int getExpiredPasswords() {
        if (this.policies == null) {
            return -1;
        }
        for (Policy p : this.policies) {
            if (!(p instanceof PasswordHistory)) continue;
            return ((PasswordHistory)p).passwordHistoryPolicyValue;
        }
        return -1;
    }

    public int getDaysToExpirePassword() {
        if (this.policies == null) {
            return -1;
        }
        for (Policy p : this.policies) {
            if (!(p instanceof ForceExpiredPasswordChange)) continue;
            return ((ForceExpiredPasswordChange)p).daysToExpirePassword;
        }
        return -1;
    }

    public Error validate(UserModel user, String password) {
        for (Policy p : this.policies) {
            Error error = p.validate(user, password);
            if (error == null) continue;
            return error;
        }
        return null;
    }

    public Error validate(String user, String password) {
        for (Policy p : this.policies) {
            Error error = p.validate(user, password);
            if (error == null) continue;
            return error;
        }
        return null;
    }

    private static int intArg(String policy, int defaultValue, String ... args) {
        if (args == null || args.length == 0) {
            return defaultValue;
        }
        if (args.length == 1) {
            return Integer.parseInt(args[0]);
        }
        throw new IllegalArgumentException("Invalid arguments to " + policy + ", expect no argument or single integer");
    }

    public String toString() {
        return this.policyString;
    }

    private static class ForceExpiredPasswordChange
    implements Policy {
        private static final String NAME = "forceExpiredPasswordChange";
        private int daysToExpirePassword;

        public ForceExpiredPasswordChange(String[] args) {
            this.daysToExpirePassword = PasswordPolicy.intArg(NAME, 365, args);
        }

        @Override
        public Error validate(String username, String password) {
            return null;
        }

        @Override
        public Error validate(UserModel user, String password) {
            return null;
        }
    }

    private static class PasswordHistory
    implements Policy {
        private static final String NAME = "passwordHistory";
        private int passwordHistoryPolicyValue;

        public PasswordHistory(String[] args) {
            this.passwordHistoryPolicyValue = PasswordPolicy.intArg(NAME, 3, args);
        }

        @Override
        public Error validate(String user, String password) {
            return null;
        }

        @Override
        public Error validate(UserModel user, String password) {
            if (this.passwordHistoryPolicyValue != -1) {
                UserCredentialValueModel cred = this.getCredentialValueModel(user, "password");
                if (cred != null && new Pbkdf2PasswordEncoder(cred.getSalt()).verify(password, cred.getValue(), cred.getHashIterations())) {
                    return new Error(PasswordPolicy.INVALID_PASSWORD_HISTORY, new Object[]{this.passwordHistoryPolicyValue});
                }
                List<UserCredentialValueModel> passwordExpiredCredentials = this.getCredentialValueModels(user, this.passwordHistoryPolicyValue - 1, "password-history");
                for (UserCredentialValueModel credential : passwordExpiredCredentials) {
                    if (!new Pbkdf2PasswordEncoder(credential.getSalt()).verify(password, credential.getValue(), credential.getHashIterations())) continue;
                    return new Error(PasswordPolicy.INVALID_PASSWORD_HISTORY, new Object[]{this.passwordHistoryPolicyValue});
                }
            }
            return null;
        }

        private UserCredentialValueModel getCredentialValueModel(UserModel user, String credType) {
            for (UserCredentialValueModel model : user.getCredentialsDirectly()) {
                if (!model.getType().equals(credType)) continue;
                return model;
            }
            return null;
        }

        private List<UserCredentialValueModel> getCredentialValueModels(UserModel user, int expiredPasswordsPolicyValue, String credType) {
            ArrayList<UserCredentialValueModel> credentialModels = new ArrayList<UserCredentialValueModel>();
            for (UserCredentialValueModel model : user.getCredentialsDirectly()) {
                if (!model.getType().equals(credType)) continue;
                credentialModels.add(model);
            }
            Collections.sort(credentialModels, new Comparator<UserCredentialValueModel>(){

                @Override
                public int compare(UserCredentialValueModel credFirst, UserCredentialValueModel credSecond) {
                    if (credFirst.getCreatedDate() > credSecond.getCreatedDate()) {
                        return -1;
                    }
                    if (credFirst.getCreatedDate() < credSecond.getCreatedDate()) {
                        return 1;
                    }
                    return 0;
                }
            });
            if (credentialModels.size() > expiredPasswordsPolicyValue) {
                return credentialModels.subList(0, expiredPasswordsPolicyValue);
            }
            return credentialModels;
        }
    }

    private static class RegexPatterns
    implements Policy {
        private static final String NAME = "regexPatterns";
        private String[] regexPatterns;

        public RegexPatterns(String[] args) {
            this.regexPatterns = args;
        }

        @Override
        public Error validate(String username, String password) {
            Pattern pattern = null;
            Matcher matcher = null;
            for (String regexPattern : this.regexPatterns) {
                pattern = Pattern.compile(regexPattern);
                matcher = pattern.matcher(password);
                if (matcher.matches()) continue;
                return new Error(PasswordPolicy.INVALID_PASSWORD_REGEX_PATTERN, new Object[]{this.regexPatterns});
            }
            return null;
        }

        @Override
        public Error validate(UserModel user, String password) {
            return this.validate(user.getUsername(), password);
        }
    }

    private static class SpecialChars
    implements Policy {
        private static final String NAME = "specialChars";
        private int min;

        public SpecialChars(String[] args) {
            this.min = PasswordPolicy.intArg(NAME, 1, args);
        }

        @Override
        public Error validate(String username, String password) {
            int count = 0;
            for (char c : password.toCharArray()) {
                if (Character.isLetterOrDigit(c)) continue;
                ++count;
            }
            return count < this.min ? new Error(PasswordPolicy.INVALID_PASSWORD_MIN_SPECIAL_CHARS_MESSAGE, new Object[]{this.min}) : null;
        }

        @Override
        public Error validate(UserModel user, String password) {
            return this.validate(user.getUsername(), password);
        }
    }

    private static class UpperCase
    implements Policy {
        private static final String NAME = "upperCase";
        private int min;

        public UpperCase(String[] args) {
            this.min = PasswordPolicy.intArg(NAME, 1, args);
        }

        @Override
        public Error validate(String username, String password) {
            int count = 0;
            for (char c : password.toCharArray()) {
                if (!Character.isUpperCase(c)) continue;
                ++count;
            }
            return count < this.min ? new Error(PasswordPolicy.INVALID_PASSWORD_MIN_UPPER_CASE_CHARS_MESSAGE, new Object[]{this.min}) : null;
        }

        @Override
        public Error validate(UserModel user, String password) {
            return this.validate(user.getUsername(), password);
        }
    }

    private static class LowerCase
    implements Policy {
        private static final String NAME = "lowerCase";
        private int min;

        public LowerCase(String[] args) {
            this.min = PasswordPolicy.intArg(NAME, 1, args);
        }

        @Override
        public Error validate(String username, String password) {
            int count = 0;
            for (char c : password.toCharArray()) {
                if (!Character.isLowerCase(c)) continue;
                ++count;
            }
            return count < this.min ? new Error(PasswordPolicy.INVALID_PASSWORD_MIN_LOWER_CASE_CHARS_MESSAGE, new Object[]{this.min}) : null;
        }

        @Override
        public Error validate(UserModel user, String password) {
            return this.validate(user.getUsername(), password);
        }
    }

    private static class Digits
    implements Policy {
        private static final String NAME = "digits";
        private int min;

        public Digits(String[] args) {
            this.min = PasswordPolicy.intArg(NAME, 1, args);
        }

        @Override
        public Error validate(String username, String password) {
            int count = 0;
            for (char c : password.toCharArray()) {
                if (!Character.isDigit(c)) continue;
                ++count;
            }
            return count < this.min ? new Error(PasswordPolicy.INVALID_PASSWORD_MIN_DIGITS_MESSAGE, new Object[]{this.min}) : null;
        }

        @Override
        public Error validate(UserModel user, String password) {
            return this.validate(user.getUsername(), password);
        }
    }

    private static class Length
    implements Policy {
        private static final String NAME = "length";
        private int min;

        public Length(String[] args) {
            this.min = PasswordPolicy.intArg(NAME, 8, args);
        }

        @Override
        public Error validate(String username, String password) {
            return password.length() < this.min ? new Error(PasswordPolicy.INVALID_PASSWORD_MIN_LENGTH_MESSAGE, new Object[]{this.min}) : null;
        }

        @Override
        public Error validate(UserModel user, String password) {
            return this.validate(user.getUsername(), password);
        }
    }

    private static class NotUsername
    implements Policy {
        private static final String NAME = "notUsername";

        public NotUsername(String[] args) {
        }

        @Override
        public Error validate(String username, String password) {
            return username.equals(password) ? new Error(PasswordPolicy.INVALID_PASSWORD_NOT_USERNAME, new Object[0]) : null;
        }

        @Override
        public Error validate(UserModel user, String password) {
            return this.validate(user.getUsername(), password);
        }
    }

    private static class HashIterations
    implements Policy {
        private static final String NAME = "hashIterations";
        private int iterations;

        public HashIterations(String[] args) {
            this.iterations = PasswordPolicy.intArg(NAME, 1, args);
        }

        @Override
        public Error validate(String user, String password) {
            return null;
        }

        @Override
        public Error validate(UserModel user, String password) {
            return null;
        }
    }

    public static class Error {
        private String message;
        private Object[] parameters;

        private Error(String message, Object ... parameters) {
            this.message = message;
            this.parameters = parameters;
        }

        public String getMessage() {
            return this.message;
        }

        public Object[] getParameters() {
            return this.parameters;
        }
    }

    private static interface Policy {
        public Error validate(UserModel var1, String var2);

        public Error validate(String var1, String var2);
    }
}

