/*
 * Copyright 2021 Red Hat, Inc. and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.kie.kogito.explainability.model;

import java.util.Collections;
import java.util.List;
import java.util.UUID;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

/**
 * The prediction generated by a {@link PredictionProvider}.
 */
public class CounterfactualPrediction extends BasePrediction {

    private final PredictionFeatureDomain domain;
    private final List<Boolean> constraints;
    private final DataDistribution dataDistribution;

    public CounterfactualPrediction(PredictionInput input,
            PredictionOutput output) {
        super(input, output);
        this.domain = new PredictionFeatureDomain(Collections.emptyList());
        this.constraints = IntStream.range(0, input.getFeatures().size())
                .mapToObj(x -> false)
                .collect(Collectors.toList());
        this.dataDistribution = null;
    }

    public CounterfactualPrediction(PredictionInput input,
            PredictionOutput output,
            PredictionFeatureDomain domain,
            List<Boolean> constraints,
            DataDistribution dataDistribution,
            UUID executionId) {
        super(input, output, executionId);
        this.domain = domain;
        this.constraints = constraints;
        this.dataDistribution = dataDistribution;
    }

    public PredictionFeatureDomain getDomain() {
        return domain;
    }

    public List<Boolean> getConstraints() {
        return constraints;
    }

    public DataDistribution getDataDistribution() {
        return dataDistribution;
    }

}
