/*
 * Decompiled with CFR 0.152.
 */
package org.kie.kogito.explainability.utils;

import java.net.URI;
import java.nio.ByteBuffer;
import java.time.Duration;
import java.time.LocalTime;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Currency;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Random;
import java.util.stream.Collectors;
import java.util.stream.DoubleStream;
import org.kie.kogito.explainability.model.DataDistribution;
import org.kie.kogito.explainability.model.Feature;
import org.kie.kogito.explainability.model.FeatureDistribution;
import org.kie.kogito.explainability.model.FeatureFactory;
import org.kie.kogito.explainability.model.PredictionInput;
import org.kie.kogito.explainability.model.Type;

public class DataUtils {
    private static final Random random = new Random();

    private DataUtils() {
    }

    public static void setSeed(long seed) {
        random.setSeed(seed);
    }

    public static double[] generateData(double mean, double stdDeviation, int size) {
        double[] data = new double[size];
        for (int i = 0; i < size; ++i) {
            data[i] = random.nextGaussian() * stdDeviation + mean;
        }
        double m = DataUtils.getMean(data);
        double d = DataUtils.getStdDev(data, m);
        double d1 = stdDeviation / d;
        int i = 0;
        while (i < size) {
            int n = i++;
            data[n] = data[n] * d1;
        }
        double m1 = m * stdDeviation / d;
        int i2 = 0;
        while (i2 < size) {
            int n = i2++;
            data[n] = data[n] + (mean - m1);
        }
        return data;
    }

    static double getMean(double[] data) {
        double m = 0.0;
        for (double datum : data) {
            m += datum;
        }
        return m /= (double)data.length;
    }

    static double getStdDev(double[] data, double mean) {
        double d = 0.0;
        for (double datum : data) {
            d += Math.pow(datum - mean, 2.0);
        }
        d /= (double)data.length;
        d = Math.sqrt(d);
        return d;
    }

    public static double[] generateSamples(double min, double max, int size) {
        double[] data = new double[size];
        double val = min;
        double sum = max / (double)size;
        for (int i = 0; i < size; ++i) {
            data[i] = val;
            val += sum;
        }
        return data;
    }

    public static List<Feature> doublesToFeatures(double[] inputs) {
        return DoubleStream.of(inputs).mapToObj(DataUtils::doubleToFeature).collect(Collectors.toList());
    }

    static Feature doubleToFeature(double d) {
        return FeatureFactory.newNumericalFeature(String.valueOf(d), d);
    }

    public static PredictionInput perturbFeatures(PredictionInput input, int noOfPerturbations) {
        int[] indexesToBePerturbed;
        List<Feature> originalFeatures = input.getFeatures();
        ArrayList<Feature> newFeatures = new ArrayList<Feature>(originalFeatures);
        PredictionInput perturbedInput = new PredictionInput(newFeatures);
        int perturbationSize = Math.min(noOfPerturbations, originalFeatures.size());
        for (int value : indexesToBePerturbed = random.ints(0, perturbedInput.getFeatures().size()).distinct().limit(perturbationSize).toArray()) {
            perturbedInput.getFeatures().set(value, DataUtils.perturbFeature(perturbedInput.getFeatures().get(value)));
        }
        return perturbedInput;
    }

    private static Feature perturbFeature(Feature feature) {
        Feature f;
        Type type = feature.getType();
        String featureName = feature.getName();
        switch (type) {
            case COMPOSITE: {
                List composite = (List)feature.getValue().getUnderlyingObject();
                HashMap<String, Object> featuresMap = new HashMap<String, Object>();
                for (Feature cf : composite) {
                    if (random.nextBoolean()) {
                        featuresMap.put(cf.getName(), DataUtils.perturbFeature(cf));
                        continue;
                    }
                    featuresMap.put(cf.getName(), cf);
                }
                f = FeatureFactory.newCompositeFeature(featureName, featuresMap);
                break;
            }
            case TEXT: {
                String stringValue;
                String newStringValue = "";
                if (random.nextBoolean() && (stringValue = feature.getValue().asString()).indexOf(32) != -1) {
                    ArrayList<String> words = new ArrayList<String>(Arrays.asList(stringValue.split(" ")));
                    if (!words.isEmpty()) {
                        int featuresToDrop = random.nextInt(Math.min(2, words.size() / 2));
                        for (int i = 0; i < 1 + featuresToDrop; ++i) {
                            int dropIdx = random.nextInt(words.size());
                            words.remove(dropIdx);
                        }
                    }
                    newStringValue = String.join((CharSequence)" ", words);
                }
                f = FeatureFactory.newTextFeature(featureName, newStringValue);
                break;
            }
            case NUMBER: {
                double originalFeatureValue = feature.getValue().asNumber();
                boolean intValue = originalFeatureValue % 1.0 == 0.0;
                double normalDistributionSample = random.nextGaussian();
                if (originalFeatureValue != 0.0) {
                    normalDistributionSample = normalDistributionSample * originalFeatureValue + originalFeatureValue;
                }
                if (intValue && (normalDistributionSample = (double)((int)normalDistributionSample)) == originalFeatureValue) {
                    normalDistributionSample = (double)((int)normalDistributionSample) * 10.0;
                }
                f = FeatureFactory.newNumericalFeature(featureName, normalDistributionSample);
                break;
            }
            case BOOLEAN: {
                f = FeatureFactory.newBooleanFeature(featureName, !Boolean.getBoolean(feature.getValue().asString()));
                break;
            }
            case TIME: {
                LocalTime featureValue = (LocalTime)feature.getValue().getUnderlyingObject();
                f = FeatureFactory.newTimeFeature(featureName, featureValue.minusHours(random.nextInt(24)));
                break;
            }
            case DURATION: {
                f = FeatureFactory.newDurationFeature(featureName, Duration.of(0L, ChronoUnit.SECONDS));
                break;
            }
            case CURRENCY: {
                f = FeatureFactory.newCurrencyFeature(featureName, Currency.getInstance(Locale.getDefault()));
                break;
            }
            case CATEGORICAL: {
                String category = feature.getValue().asString();
                category = !"0".equals(category) ? "0" : "1";
                f = FeatureFactory.newCategoricalFeature(featureName, category);
                break;
            }
            case BINARY: {
                ByteBuffer byteBuffer = ByteBuffer.allocate(0);
                f = FeatureFactory.newBinaryFeature(featureName, byteBuffer);
                break;
            }
            case URI: {
                f = FeatureFactory.newURIFeature(featureName, URI.create(""));
                break;
            }
            case VECTOR: {
                double[] values = feature.getValue().asVector();
                if (values.length > 1) {
                    int idx = random.nextInt(values.length - 1);
                    if (values[idx] != 0.0) {
                        values[idx] = 0.0;
                    } else {
                        int n = idx;
                        values[n] = values[n] - 1.0;
                    }
                }
                f = FeatureFactory.newVectorFeature(featureName, values);
                break;
            }
            case UNDEFINED: {
                if (feature.getValue().getUnderlyingObject() instanceof Feature) {
                    f = DataUtils.perturbFeature((Feature)feature.getValue().getUnderlyingObject());
                    break;
                }
                f = feature;
                break;
            }
            default: {
                throw new IllegalStateException("Unexpected value: " + type);
            }
        }
        return f;
    }

    public static Feature dropFeature(Feature feature, List<String> names) {
        Type type = feature.getType();
        Feature f = feature;
        String featureName = feature.getName();
        switch (type) {
            case COMPOSITE: {
                List composite = (List)feature.getValue().getUnderlyingObject();
                HashMap<String, Object> featuresMap = new HashMap<String, Object>();
                for (Feature cf : composite) {
                    featuresMap.put(cf.getName(), DataUtils.dropFeature(cf, names));
                }
                f = FeatureFactory.newCompositeFeature(featureName, featuresMap);
                break;
            }
            case TEXT: {
                if (names.contains(featureName)) {
                    f = FeatureFactory.newTextFeature(featureName, "");
                    break;
                }
                String stringValue = feature.getValue().asString();
                if (stringValue.indexOf(32) != -1) {
                    ArrayList<String> words = new ArrayList<String>(Arrays.asList(stringValue.split(" ")));
                    List matchingWords = names.stream().map(n -> n.contains(" (") ? n.substring(0, n.indexOf(" (")) : "").filter(words::contains).collect(Collectors.toList());
                    if (words.removeAll(matchingWords)) {
                        stringValue = String.join((CharSequence)" ", words);
                    }
                }
                f = FeatureFactory.newTextFeature(featureName, stringValue);
                break;
            }
            case NUMBER: {
                if (!names.contains(featureName)) break;
                if (feature.getValue().asNumber() == 0.0) {
                    f = FeatureFactory.newNumericalFeature(featureName, Double.NaN);
                    break;
                }
                f = FeatureFactory.newNumericalFeature(featureName, 0);
                break;
            }
            case BOOLEAN: {
                if (!names.contains(featureName)) break;
                f = FeatureFactory.newBooleanFeature(featureName, null);
                break;
            }
            case TIME: {
                if (!names.contains(featureName)) break;
                f = FeatureFactory.newTimeFeature(featureName, null);
                break;
            }
            case DURATION: {
                if (!names.contains(featureName)) break;
                f = FeatureFactory.newDurationFeature(featureName, null);
                break;
            }
            case CURRENCY: {
                if (!names.contains(featureName)) break;
                f = FeatureFactory.newCurrencyFeature(featureName, null);
                break;
            }
            case CATEGORICAL: {
                if (!names.contains(featureName)) break;
                f = FeatureFactory.newCategoricalFeature(featureName, "");
                break;
            }
            case BINARY: {
                if (!names.contains(featureName)) break;
                ByteBuffer byteBuffer = ByteBuffer.allocate(0);
                f = FeatureFactory.newBinaryFeature(featureName, byteBuffer);
                break;
            }
            case URI: {
                if (!names.contains(featureName)) break;
                f = FeatureFactory.newURIFeature(featureName, URI.create(""));
                break;
            }
            case VECTOR: {
                if (!names.contains(featureName)) break;
                double[] values = feature.getValue().asVector();
                if (values.length > 0) {
                    Arrays.fill(values, 0.0);
                }
                f = FeatureFactory.newVectorFeature(featureName, values);
                break;
            }
            case UNDEFINED: {
                if (feature.getValue().getUnderlyingObject() instanceof Feature) {
                    f = DataUtils.dropFeature((Feature)feature.getValue().getUnderlyingObject(), names);
                    break;
                }
                f = FeatureFactory.newObjectFeature(featureName, null);
                break;
            }
            default: {
                throw new IllegalStateException("Unexpected value: " + type);
            }
        }
        return f;
    }

    public static double hammingDistance(double[] x, double[] y) {
        if (x.length != y.length) {
            return Double.NaN;
        }
        double h = 0.0;
        for (int i = 0; i < x.length; ++i) {
            if (x[i] == y[i]) continue;
            h += 1.0;
        }
        return h;
    }

    public static double hammingDistance(String x, String y) {
        if (x.length() != y.length()) {
            return Double.NaN;
        }
        double h = 0.0;
        for (int i = 0; i < x.length(); ++i) {
            if (x.charAt(i) == y.charAt(i)) continue;
            h += 1.0;
        }
        return h;
    }

    public static double euclideanDistance(double[] x, double[] y) {
        if (x.length != y.length) {
            return Double.NaN;
        }
        double e = 0.0;
        for (int i = 0; i < x.length; ++i) {
            e += Math.pow(x[i] - y[i], 2.0);
        }
        return Math.sqrt(e);
    }

    public static double gaussianKernel(double x, double mu, double sigma) {
        return Math.exp(-Math.pow((x - mu) / sigma, 2.0) / 2.0) / (sigma * Math.sqrt(Math.PI * 2));
    }

    public static double exponentialSmoothingKernel(double x, double width) {
        return Math.sqrt(Math.exp(-Math.pow(x, 2.0) / Math.pow(width, 2.0)));
    }

    public static FeatureDistribution getFeatureDistribution(double[] doubles) {
        double min = DoubleStream.of(doubles).min().orElse(0.0);
        double max = DoubleStream.of(doubles).max().orElse(0.0);
        double mean = DataUtils.getMean(doubles);
        double stdDev = DataUtils.getStdDev(doubles, mean);
        return new FeatureDistribution(min, max, mean, stdDev);
    }

    public static DataDistribution generateRandomDataDistribution(int noOfFeatures, int distributionSize) {
        LinkedList<FeatureDistribution> featureDistributions = new LinkedList<FeatureDistribution>();
        for (int i = 0; i < noOfFeatures; ++i) {
            double[] doubles = DataUtils.generateData(random.nextDouble(), random.nextDouble(), distributionSize);
            FeatureDistribution featureDistribution = DataUtils.getFeatureDistribution(doubles);
            featureDistributions.add(featureDistribution);
        }
        return new DataDistribution(featureDistributions);
    }

    public static List<PredictionInput> linearizeInputs(List<PredictionInput> predictionInputs) {
        LinkedList<PredictionInput> newInputs = new LinkedList<PredictionInput>();
        for (PredictionInput predictionInput : predictionInputs) {
            List<Feature> originalFeatures = predictionInput.getFeatures();
            List<Feature> flattenedFeatures = DataUtils.getLinearizedFeatures(originalFeatures);
            newInputs.add(new PredictionInput(flattenedFeatures));
        }
        return newInputs;
    }

    public static List<Feature> getLinearizedFeatures(List<Feature> originalFeatures) {
        LinkedList<Feature> flattenedFeatures = new LinkedList<Feature>();
        for (Feature f : originalFeatures) {
            DataUtils.linearizeFeature(flattenedFeatures, f);
        }
        return flattenedFeatures;
    }

    private static void linearizeFeature(List<Feature> flattenedFeatures, Feature f) {
        if (Type.UNDEFINED.equals((Object)f.getType())) {
            if (f.getValue().getUnderlyingObject() instanceof Feature) {
                DataUtils.linearizeFeature(flattenedFeatures, (Feature)f.getValue().getUnderlyingObject());
            } else {
                flattenedFeatures.add(f);
            }
        } else if (Type.COMPOSITE.equals((Object)f.getType())) {
            List features = (List)f.getValue().getUnderlyingObject();
            for (Feature feature : features) {
                DataUtils.linearizeFeature(flattenedFeatures, feature);
            }
        } else if (Type.TEXT.equals((Object)f.getType())) {
            for (String w : f.getValue().asString().split(" ")) {
                Feature outputFeature = FeatureFactory.newTextFeature(w + " (" + f.getName() + ")", w);
                flattenedFeatures.add(outputFeature);
            }
        } else {
            flattenedFeatures.add(f);
        }
    }
}

