
package org.jbpm.serverless.workflow.api.switchconditions;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;
import javax.validation.Valid;
import javax.validation.constraints.NotNull;
import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonPropertyDescription;
import com.fasterxml.jackson.annotation.JsonPropertyOrder;
import com.fasterxml.jackson.annotation.JsonValue;
import org.jbpm.serverless.workflow.api.interfaces.SwitchCondition;
import org.jbpm.serverless.workflow.api.transitions.Transition;


/**
 * Switch state data based condition
 * 
 */
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonPropertyOrder({
    "path",
    "value",
    "operator",
    "transition"
})
public class DataCondition implements Serializable, SwitchCondition
{

    /**
     * JSONPath expression that selects elements of state data
     * (Required)
     * 
     */
    @JsonProperty("path")
    @JsonPropertyDescription("JSONPath expression that selects elements of state data")
    @NotNull
    private String path;
    /**
     * Matching value
     * (Required)
     * 
     */
    @JsonProperty("value")
    @JsonPropertyDescription("Matching value")
    @NotNull
    private String value;
    /**
     * Condition operator
     * (Required)
     * 
     */
    @JsonProperty("operator")
    @JsonPropertyDescription("Condition operator")
    @NotNull
    private DataCondition.Operator operator;
    /**
     * 
     * (Required)
     * 
     */
    @JsonProperty("transition")
    @Valid
    @NotNull
    private Transition transition;
    private final static long serialVersionUID = 878469266715403371L;

    /**
     * No args constructor for use in serialization
     * 
     */
    public DataCondition() {
    }

    /**
     * 
     * @param path
     * @param value
     * @param operator
     * @param transition
     */
    public DataCondition(String path, String value, DataCondition.Operator operator, Transition transition) {
        super();
        this.path = path;
        this.value = value;
        this.operator = operator;
        this.transition = transition;
    }

    /**
     * JSONPath expression that selects elements of state data
     * (Required)
     * 
     */
    @JsonProperty("path")
    public String getPath() {
        return path;
    }

    /**
     * JSONPath expression that selects elements of state data
     * (Required)
     * 
     */
    @JsonProperty("path")
    public void setPath(String path) {
        this.path = path;
    }

    public DataCondition withPath(String path) {
        this.path = path;
        return this;
    }

    /**
     * Matching value
     * (Required)
     * 
     */
    @JsonProperty("value")
    public String getValue() {
        return value;
    }

    /**
     * Matching value
     * (Required)
     * 
     */
    @JsonProperty("value")
    public void setValue(String value) {
        this.value = value;
    }

    public DataCondition withValue(String value) {
        this.value = value;
        return this;
    }

    /**
     * Condition operator
     * (Required)
     * 
     */
    @JsonProperty("operator")
    public DataCondition.Operator getOperator() {
        return operator;
    }

    /**
     * Condition operator
     * (Required)
     * 
     */
    @JsonProperty("operator")
    public void setOperator(DataCondition.Operator operator) {
        this.operator = operator;
    }

    public DataCondition withOperator(DataCondition.Operator operator) {
        this.operator = operator;
        return this;
    }

    /**
     * 
     * (Required)
     * 
     */
    @JsonProperty("transition")
    public Transition getTransition() {
        return transition;
    }

    /**
     * 
     * (Required)
     * 
     */
    @JsonProperty("transition")
    public void setTransition(Transition transition) {
        this.transition = transition;
    }

    public DataCondition withTransition(Transition transition) {
        this.transition = transition;
        return this;
    }

    public enum Operator {

        EXISTS("exists"),
        NOTEXISTS("notexists"),
        NULL("null"),
        NOTNULL("notnull"),
        EQUALS("equals"),
        NOTEQUALS("notequals"),
        LESSTHAN("lessthan"),
        LESSTHANOREQUALS("lessthanorequals"),
        GREATERTHAN("greaterthan"),
        GREATERTHANOREQUALS("greaterthanorequals"),
        MATCHES("matches"),
        NOTMATCHES("notmatches"),
        CUSTOM("custom");
        private final String value;
        private final static Map<String, DataCondition.Operator> CONSTANTS = new HashMap<String, DataCondition.Operator>();

        static {
            for (DataCondition.Operator c: values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private Operator(String value) {
            this.value = value;
        }

        @Override
        public String toString() {
            return this.value;
        }

        @JsonValue
        public String value() {
            return this.value;
        }

        @JsonCreator
        public static DataCondition.Operator fromValue(String value) {
            DataCondition.Operator constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

    }

}
