
package org.jbpm.serverless.workflow.api.choices;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;
import javax.validation.Valid;
import javax.validation.constraints.NotNull;
import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonPropertyDescription;
import com.fasterxml.jackson.annotation.JsonPropertyOrder;
import com.fasterxml.jackson.annotation.JsonValue;
import org.jbpm.serverless.workflow.api.interfaces.Choice;
import org.jbpm.serverless.workflow.api.transitions.Transition;


/**
 * Default Choice
 * 
 */
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonPropertyOrder({
    "path",
    "value",
    "operator",
    "transition"
})
public class DefaultChoice implements Serializable, Choice
{

    /**
     * JSON Path that selects the value of the input data to be matched
     * (Required)
     * 
     */
    @JsonProperty("path")
    @JsonPropertyDescription("JSON Path that selects the value of the input data to be matched")
    @NotNull
    private String path;
    /**
     * Matching value
     * (Required)
     * 
     */
    @JsonProperty("value")
    @JsonPropertyDescription("Matching value")
    @NotNull
    private String value;
    /**
     * Specifies how the input data is compared with the values
     * (Required)
     * 
     */
    @JsonProperty("operator")
    @JsonPropertyDescription("Specifies how the input data is compared with the values")
    @NotNull
    private DefaultChoice.Operator operator;
    @JsonProperty("transition")
    @Valid
    private Transition transition;
    private final static long serialVersionUID = 119747170163935975L;

    /**
     * No args constructor for use in serialization
     * 
     */
    public DefaultChoice() {
    }

    /**
     * 
     * @param path
     * @param value
     * @param operator
     */
    public DefaultChoice(String path, String value, DefaultChoice.Operator operator) {
        super();
        this.path = path;
        this.value = value;
        this.operator = operator;
    }

    /**
     * JSON Path that selects the value of the input data to be matched
     * (Required)
     * 
     */
    @JsonProperty("path")
    public String getPath() {
        return path;
    }

    /**
     * JSON Path that selects the value of the input data to be matched
     * (Required)
     * 
     */
    @JsonProperty("path")
    public void setPath(String path) {
        this.path = path;
    }

    public DefaultChoice withPath(String path) {
        this.path = path;
        return this;
    }

    /**
     * Matching value
     * (Required)
     * 
     */
    @JsonProperty("value")
    public String getValue() {
        return value;
    }

    /**
     * Matching value
     * (Required)
     * 
     */
    @JsonProperty("value")
    public void setValue(String value) {
        this.value = value;
    }

    public DefaultChoice withValue(String value) {
        this.value = value;
        return this;
    }

    /**
     * Specifies how the input data is compared with the values
     * (Required)
     * 
     */
    @JsonProperty("operator")
    public DefaultChoice.Operator getOperator() {
        return operator;
    }

    /**
     * Specifies how the input data is compared with the values
     * (Required)
     * 
     */
    @JsonProperty("operator")
    public void setOperator(DefaultChoice.Operator operator) {
        this.operator = operator;
    }

    public DefaultChoice withOperator(DefaultChoice.Operator operator) {
        this.operator = operator;
        return this;
    }

    @JsonProperty("transition")
    public Transition getTransition() {
        return transition;
    }

    @JsonProperty("transition")
    public void setTransition(Transition transition) {
        this.transition = transition;
    }

    public DefaultChoice withTransition(Transition transition) {
        this.transition = transition;
        return this;
    }

    public enum Operator {

        EQUALS("Equals"),
        LESS_THAN("LessThan"),
        LESS_THAN_EQUALS("LessThanEquals"),
        GREATER_THAN("GreaterThan"),
        GREATER_THAN_EQUALS("GreaterThanEquals");
        private final String value;
        private final static Map<String, DefaultChoice.Operator> CONSTANTS = new HashMap<String, DefaultChoice.Operator>();

        static {
            for (DefaultChoice.Operator c: values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private Operator(String value) {
            this.value = value;
        }

        @Override
        public String toString() {
            return this.value;
        }

        @JsonValue
        public String value() {
            return this.value;
        }

        @JsonCreator
        public static DefaultChoice.Operator fromValue(String value) {
            DefaultChoice.Operator constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

    }

}
