package org.kink_lang.kink;

import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;

/**
 * The component registry of the VM.
 *
 * <p>Components are any objects which are registered to the VM.
 * A component is registered with its class.
 * For a class, only one component can be registered.
 * The second or later attempts of registration do not succeed.</p>
 *
 * @see Vm#component
 */
public class ComponentRegistry {

    /** The vm. */
    private final Vm vm;

    /** The reference to the components mappings. */
    private final AtomicReference<Map<Class<?>, Object>> mappingRef
        = new AtomicReference<>(Map.of());

    /**
     * Constructs the registry.
     */
    ComponentRegistry(Vm vm) {
        this.vm = vm;
    }

    /**
     * Returns a component registered with the class.
     *
     * @param <T> the type of the class of the component.
     * @param klass the class of the component.
     * @return the component.
     * @throws IllegalStateException if it is not registered yet.
     */
    @SuppressWarnings("unchecked")
    public <T> T get(Class<T> klass) {
        Map<Class<?>, Object> mapping = this.mappingRef.get();
        T result = (T) mapping.get(klass);
        if (result == null) {
            throw new IllegalStateException(String.format(Locale.ROOT,
                        "component for class %s is not registered", klass.getName()));
        }
        return result;
    }

    /**
     * Registers the component with the class, only if no component is registered with the class.
     *
     * @param <T> the type of the class of the component.
     * @param klass the class of the component.
     * @param component the component.
     */
    public <T> void register(Class<T> klass, T component) {
        while (true) {
            Map<Class<?>, Object> mapping = this.mappingRef.get();
            if (mapping.containsKey(klass)) {
                return;
            }
            Map<Class<?>, Object> newMapping = new HashMap<>(mapping);
            newMapping.put(klass, component);
            this.mappingRef.compareAndSet(mapping, newMapping);
        }
    }

    /**
     * Returns a component registered with the class;
     * if it is not registered yet, makes one via {@code makeComponent},
     * then registers and returns it.
     *
     * <p>Example:</p>
     *
     * <pre>
     * class X {
     *   X(Vm vm) {}
     * }
     *
     * X x = vm.component.getOrRegister(X.class, X::new);
     * </pre>
     *
     * @param <T> the type of the class of the component.
     * @param klass the class of the component.
     * @param makeComponent a function to make a component.
     * @return the component.
     */
    @SuppressWarnings("unchecked")
    public <T> T getOrRegister(Class<T> klass, Function<Vm, T> makeComponent) {
        T component = (T) this.mappingRef.get().get(klass);
        if (component != null) {
            return component;
        }
        register(klass, makeComponent.apply(vm));
        return (T) this.mappingRef.get().get(klass);
    }

}

// vim: et sw=4 sts=4 fdm=marker
