package org.kink_lang.kink;

import java.lang.reflect.Array;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.function.Supplier;
import java.util.function.UnaryOperator;

import javax.annotation.Nullable;

import org.kink_lang.kink.internal.function.ThrowingFunction0;
import org.kink_lang.kink.internal.function.ThrowingFunction1;
import org.kink_lang.kink.internal.function.ThrowingFunction2;
import org.kink_lang.kink.internal.function.ThrowingFunction3;
import org.kink_lang.kink.internal.function.ThrowingFunction4;
import org.kink_lang.kink.hostfun.HostFunBuilder;
import org.kink_lang.kink.hostfun.HostResult;
import org.kink_lang.kink.hostfun.graph.GraphNode;
import org.kink_lang.kink.hostfun.HostContext;
import org.kink_lang.kink.hostfun.CallContext;
import org.kink_lang.kink.internal.num.NumOperations;

/**
 * The helper of {@link JavaVal}s.
 */
public class JavaHelper {

    /** The vm. */
    private final Vm vm;

    /** Shared vars of java_val vals.. */
    SharedVars sharedVars;

    /** The handle of "read_conf". */
    private int readConfHandle;

    /** The default success cont of invoecation. */
    private FunVal invDefSuccessCont;

    /** The default error cont of invoecation. */
    private FunVal invDefErrorCont;

    /**
     * Makes the helper.
     */
    JavaHelper(Vm vm) {
        this.vm = vm;
    }

    /**
     * Returns a java val.
     *
     * <p>The {@code object} must be able to typed as {@code staticType}.
     * Namely,</p>
     *
     * <ul>
     *   <li>{@code object} is null and {@code staticType} is not a primitive type,</li>
     *   <li>{@code object} is a boxed value and {@code staticType}
     *   is the corresponding primitive type,</li>
     *   <li>or {@code object} is not null and {@code staticType.isInstance(object)} is true.</li>
     * </ul>
     *
     * @param object the nullable java object reference
     * @param staticType the static type.
     * @return a java val.
     * @throws IllegalArgumentException if {@code object} cannot be typed as {@code staticType}.
     */
    public JavaVal of(@Nullable Object object, Class<?> staticType) {
        return new JavaVal(vm, object, staticType);
    }

    /**
     * Intializes the helper.
     */
    void init() {
        Map<Integer, Val> map = new HashMap<>();
        map.put(vm.sym.handleFor("static_type"),
                method0("Java_val.static_type", this::staticTypeFun));
        map.put(vm.sym.handleFor("dynamic_type"),
                method0("Java_val.dynamic_type", this::dynamicTypeFun));
        addMethod1(map, "Java_val", "typable_as?", "Klass", this::typableAsFun);
        addMethod1(map, "Java_val", "as", "Klass", this::asFun);
        addMethod1(map, "Java_val", "eq_eq?", "Another", this::eqEqPFun);
        map.put(vm.sym.handleFor("null?"),
                method0("Java_val.null?", (c, v) -> vm.bool.of(v.objectReference() == null)));
        map.put(vm.sym.handleFor("to_kink_str"),
                method0("Java_val.to_kink_str", this::toKinkStrFun));
        map.put(vm.sym.handleFor("to_kink_bool"),
                method0("Java_val.to_kink_bool", this::toKinkBoolFun));
        map.put(vm.sym.handleFor("to_kink_num"),
                method0("Java_val.to_kink_num", this::toKinkNumFun));
        map.put(vm.sym.handleFor("to_kink_bin"),
                method0("Java_val.to_kink_bin", this::toKinkBinFun));
        map.put(vm.sym.handleFor("to_kink_exception"),
                method0("Java_val.to_kink_exception", this::toKinkExceptionMethod));
        map.put(vm.sym.handleFor("unwrap"),
                method0("Java_val.unwrap", this::unwrapFun));
        map.put(vm.sym.handleFor("array_class"),
                method0("Java_val.array_class", this::arrayClassFun));
        addMethod1(map, "Java_val", "array_new", "Size", this::arrayNewFun);
        map.put(vm.sym.handleFor("array_of"),
                method("Java_val.array_of", this::arrayOfFun));
        map.put(vm.sym.handleFor("array_length"),
                method0("Java_val.array_length", this::arrayLengthFun));
        addMethod1(map, "Java_val", "array_get", "Ind", this::arrayGetFun);
        map.put(vm.sym.handleFor("array_set"),
                method2("Java_val.array_set(Ind Elem)", this::arraySetFun));
        addMethod1(map, "Java_val", "get_field", "Field_name", this::getFieldFun);
        map.put(vm.sym.handleFor("set_field"),
                method2("Java_val.set_field(Field_name Content)", this::setFieldFun));
        addMethod1(map, "Java_val", "get_static", "Field_name", this::getStaticFun);
        map.put(vm.sym.handleFor("set_static"),
                method2("Java_val.set_static(Field_name Content)", this::setStaticFun));
        map.put(vm.sym.handleFor("call_method"),
                methodMinMax("Java_val.call_method(Method_name Args ...[$config])",
                    2, 3, this::callMethodFun));
        map.put(vm.sym.handleFor("call_static"),
                methodMinMax("Java_val.call_static(Method_name Args ...[$config])",
                    2, 3, this::callStaticFun));
        map.put(vm.sym.handleFor("new"), methodMinMax("Java_val.new(Args ...[$config])",
                    1, 2, this::newFun));
        map.put(vm.sym.handleFor("repr"), method0("Java_val.repr", this::reprFun));

        this.sharedVars = vm.sharedVars.of(map);
        this.readConfHandle = vm.sym.handleFor("read_conf");
        this.invDefSuccessCont = vm.fun.make().takeMinMax(0, 1).action(this::invDefSuccessCont);
        this.invDefErrorCont = vm.fun.make().take(1).action(this::invDefErrorCont);
    }

    // Java_val.static_type {{{1

    /**
     * Implementation of Java_val.static_type.
     */
    private HostResult staticTypeFun(HostContext c, JavaVal jv) {
        return vm.java.of(jv.staticType(), Class.class);
    }

    // }}}1
    // Java_val.dynamic_type {{{1

    /**
     * Implementation of Java_val.dynamic_type.
     */
    private HostResult dynamicTypeFun(HostContext c, JavaVal jv) {
        Object obj = jv.objectReference();
        if (obj == null) {
            return c.raise(
                    "Java_val.dynamic_type: required non-null java_val as \\recv, but got null");
        }
        return vm.java.of(obj.getClass(), Class.class);
    }

    // }}}1
    // Java_val.typable_as? {{{1

    /**
     * Implementation of Java_val.typable_as?.
     */
    private HostResult typableAsFun(String desc, HostContext c, JavaVal jv, Val klassVal) {
        Class<?> klass = asClass(klassVal);
        if (klass == null) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Klass must be a Java class, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(klassVal)));
        }
        return vm.bool.of(isTypable(jv.objectReference(), klass));
    }

    // }}}1
    // Java_val.as {{{1

    /**
     * Implementation of Java_val.as.
     */
    private HostResult asFun(String desc, HostContext c, JavaVal jv, Val klassVal) {
        Class<?> klass = asClass(klassVal);
        if (klass == null) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Klass must be a Java class, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(klassVal)));
        }

        Object obj = jv.objectReference();
        if (! isTypable(obj, klass)) {
            String msg = jv.objectReference() == null
                ? String.format(Locale.ROOT,
                        "%s: null is not typable as %s", desc, klass.getName())
                : String.format(Locale.ROOT,
                        "%s: instance of %s is not typable as %s",
                        desc,
                        obj.getClass().getName(), klass.getName());
            return c.raise(msg);
        }
        return vm.java.of(jv.objectReference(), klass);
    }

    // }}}1
    // Java_val.eq_eq? {{{1

    /**
     * Implementation of Java_val.eq_eq?.
     */
    private HostResult eqEqPFun(String desc, HostContext c, JavaVal jv, Val arg) {
        if (! (arg instanceof JavaVal)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: required java_val for Another, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(arg)));
        }
        JavaVal argJval = (JavaVal) arg;
        return vm.bool.of(jv.objectReference() == argJval.objectReference());
    }

    // }}}1
    // Java_val.to_kink_str {{{1

    /**
     * Implementation of Java_val.to_kink_str.
     */
    private HostResult toKinkStrFun(HostContext c, JavaVal jv) {
        if (! (jv.objectReference() instanceof String)) {
            return c.call(vm.graph.raiseFormat(
                        "Java_val.to_kink_str: required string java_val as \\recv, but got {}",
                        vm.graph.repr(jv)));
        }
        return vm.str.of((String) jv.objectReference());
    }

    // }}}1
    // Java_val.to_kink_bool {{{1

    /**
     * Implementation of Java_val.to_kink_bool.
     */
    private HostResult toKinkBoolFun(HostContext c, JavaVal jv) {
        if (! (jv.objectReference() instanceof Boolean)) {
            return c.call(vm.graph.raiseFormat(
                        "Java_val.to_kink_bool: required boolean java_val as \\recv, but got {}",
                        vm.graph.repr(jv)));
        }
        return vm.bool.of((Boolean) jv.objectReference());
    }

    // }}}1
    // Java_val.to_kink_num {{{1

    /**
     * Implementation of Java_val.to_kink_num.
     */
    private HostResult toKinkNumFun(HostContext c, JavaVal jv) {
        Object obj = jv.objectReference();

        if (obj instanceof Character) {
            return vm.num.of((Character) obj);
        } else if (obj instanceof Byte) {
            return vm.num.of((Byte) obj);
        } else if (obj instanceof Short) {
            return vm.num.of((Short) obj);
        } else if (obj instanceof Integer) {
            return vm.num.of((Integer) obj);
        } else if (obj instanceof Long) {
            return vm.num.of((Long) obj);
        } else if (obj instanceof Float) {
            float num = (Float) obj;
            if (! Float.isFinite(num)) {
                return c.raise(String.format(Locale.ROOT,
                            "Java_val.to_kink_num: cannot convert %s to kink num", num));
            }
            return vm.num.of(new BigDecimal((double) num));
        } else if (obj instanceof Double) {
            double num = (Double) obj;
            if (! Double.isFinite(num)) {
                return c.raise(String.format(Locale.ROOT,
                            "Java_val.to_kink_num: cannot convert %s to kink num", num));
            }
            return vm.num.of(new BigDecimal((Double) obj));
        } else if (obj instanceof BigInteger) {
            return vm.num.of((BigInteger) obj);
        } else if (obj instanceof BigDecimal) {
            return vm.num.of((BigDecimal) obj);
        }

        return c.call(vm.graph.raiseFormat(
                    "Java_val.to_kink_num: required char, byte, short, int, long,"
                    + " finite float, finite double, BigInteger or BigDecimal as \\recv,"
                    + " but got {}",
                    vm.graph.repr(jv)));
    }

    // }}}1

    // Java_val.to_kink_bin {{{1

    /**
     * Implementation of Java_val.to_kink_bin.
     */
    private HostResult toKinkBinFun(HostContext c, JavaVal jv) {
        if (! (jv.objectReference() instanceof byte[])) {
            return c.call(vm.graph.raiseFormat(
                        "Java_val.to_kink_bin: required byte[] java_val as \\recv, but got {}",
                        vm.graph.repr(jv)));
        }
        byte[] bytes = (byte[]) jv.objectReference();
        return vm.bin.of(bytes);
    }

    // }}}1

    // Java_val.to_kink_exception {{{

    /**
     * Implementation of Java_val.to_kink_exception.
     */
    private HostResult toKinkExceptionMethod(CallContext c, JavaVal recv) {
        String desc = "Java_val.to_kink_exception";
        if (! (recv.objectReference() instanceof Throwable th)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Java_val must be java val of Throwable, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(recv)));
        }
        return vm.exception.of(th);
    }

    // }}}

    // Java_val.unwrap {{{1

    /**
     * Implementation of Java_val.unwrap.
     */
    private HostResult unwrapFun(HostContext c, JavaVal jv) {
        if (! (jv.objectReference() instanceof Val)) {
            return c.call(vm.graph.raiseFormat(
                        "Java_val.unwrap: required java_val of org.kink_lang.kink.Val as \\recv,"
                        + " but got {}",
                        vm.graph.repr(jv)));
        }

        Val val = (Val) jv.objectReference();
        if (val.vm != this.vm) {
            return c.raise("Java_val.unwrap: could not unwrap val from another Kink VM");
        }
        return val;
    }

    // }}}1
    // Java_val.array_class {{{1

    /**
     * Implementation of Java_val.array_class.
     */
    private HostResult arrayClassFun(HostContext c, JavaVal klassVal) {
        Object klassObj = klassVal.objectReference();
        Optional<String> errorMsg = checkErrorForArrayComponentClass(
                "Java_val.array_class", klassObj);
        return errorMsg.map(c::raise).orElseGet(() -> {
            Object array = Array.newInstance((Class<?>) klassObj, 0);
            return vm.java.of(array.getClass(), Class.class);
        });
    }

    // }}}1
    // Java_val.array_new(Size) {{{1

    /**
     * Implementation of Java_val.array_new(Size).
     */
    private HostResult arrayNewFun(String desc, HostContext c, JavaVal klassVal, Val sizeVal) {
        Object klassObj = klassVal.objectReference();
        Optional<String> errorMsg = checkErrorForArrayComponentClass(desc, klassObj);
        if (errorMsg.isPresent()) {
            return c.raise(errorMsg.get());
        }
        int size = NumOperations.getPosIndex(sizeVal, Integer.MAX_VALUE);
        if (size < 0) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Size must be an int num in [0, {}], but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.of(vm.num.of(Integer.MAX_VALUE)),
                        vm.graph.repr(sizeVal)));
        }
        Object array = Array.newInstance((Class<?>) klassObj, size);
        return vm.java.of(array, array.getClass());
    }

    // }}}1
    // Java_val.array_of(E0 E1 ...) {{{1

    /**
     * Implementation of Java_val.array_of(E0 E1 ...).
     */
    private HostResult arrayOfFun(CallContext c, JavaVal klassVal) {
        Object klassObj = klassVal.objectReference();
        Optional<String> errorMsg = checkErrorForArrayComponentClass(
                "Java_val.array_of", klassObj);
        if (errorMsg.isPresent()) {
            return c.raise(errorMsg.get());
        }
        Class<?> klass = (Class<?>) klassObj;
        Object array = Array.newInstance(klass, c.argCount());
        for (int i = 0; i < c.argCount(); ++ i) {
            Val elem = c.arg(i);
            if (! (elem instanceof JavaVal)) {
                return c.call(vm.graph.raiseFormat(
                            "Java_val.array_of: required java_val as \\{}, but got {}",
                            vm.graph.of(vm.num.of(i)),
                            vm.graph.repr(elem)));
            }
            Object elemObj = ((JavaVal) elem).objectReference();
            if (! isTypable(elemObj, klass)) {
                return c.call(vm.graph.raiseFormat(
                            "Java_val.array_of: required java_val typable as {}, as \\{},"
                            + " but got {}",
                            vm.graph.of(vm.str.of(klass.getName())),
                            vm.graph.of(vm.num.of(i)),
                            vm.graph.repr(elem)));
            }
            Array.set(array, i, elemObj);
        }
        return vm.java.of(array, array.getClass());
    }

    // }}}1
    // Java_val.array_length {{{1

    /**
     * Implementation of Java_val.array_length.
     */
    private HostResult arrayLengthFun(HostContext c, JavaVal jval) {
        Object obj = jval.objectReference();
        return checkErrorForArray("Java_val.array_length", obj).map(c::raise)
            .orElseGet(() -> vm.num.of(Array.getLength(obj)));
    }

    // }}}1
    // Java_val.array_get(Index) {{{1

    /**
     * Implementation of Java_val.array_get(Index).
     */
    private HostResult arrayGetFun(String desc, HostContext c, JavaVal jv, Val indexVal) {
        Object arrayObj = jv.objectReference();
        Optional<String> errorMsg = checkErrorForArray(desc, arrayObj);
        if (errorMsg.isPresent()) {
            return c.raise(errorMsg.get());
        }

        int size = Array.getLength(arrayObj);
        int index = NumOperations.getElemIndex(indexVal, size);
        if (index < 0) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Ind must be an int num in [0, {}), but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.of(vm.num.of(size)),
                        vm.graph.repr(indexVal)));
        }
        return vm.java.of(Array.get(arrayObj, index), arrayObj.getClass().getComponentType());
    }

    // }}}1
    // Java_val.array_set(Index Elem) {{{1

    /**
     * Implementation of Java_val.array_set(Index Elem).
     */
    private HostResult arraySetFun(
            HostContext c, JavaVal jv, Val indexVal, Val elemVal
            ) throws Throwable {
        Object arrayObj = jv.objectReference();
        String desc = "Java_val.array_set(Ind Elem)";
        Optional<String> errorMsg = checkErrorForArray(desc, arrayObj);
        if (errorMsg.isPresent()) {
            return c.raise(errorMsg.get());
        }

        int size = Array.getLength(arrayObj);
        int index = NumOperations.getElemIndex(indexVal, size);
        if (index < 0) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Ind must be an int num in [0, {}), but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.of(vm.num.of(size)),
                        vm.graph.repr(indexVal)));
        }

        Class<?> componentType = arrayObj.getClass().getComponentType();
        return forValOf(componentType, elemVal, () -> c.call(vm.graph.raiseFormat(
                        "{}: Elem msut be a java_val typable as {}, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.of(vm.str.of(componentType.getName())),
                        vm.graph.repr(elemVal))),
                elem -> {
                    Array.set(arrayObj, index, elem);
                    return vm.nada;
                });
    }

    // }}}1
    // Java_val.get_field {{{1

    /**
     * Implementation of Java_val.get_field.
     */
    private HostResult getFieldFun(
            String desc, HostContext c, JavaVal jv, Val nameVal
            ) throws ReflectiveOperationException {
        if (jv.objectReference() == null) {
            return c.raise(String.format(Locale.ROOT,
                        "%s: required non-null java_val for \\recv, but got null",
                        desc));
        }

        if (! (nameVal instanceof StrVal)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Field_name must be a str, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(nameVal)));
        }

        String name = ((StrVal) nameVal).string();

        Field field = jv.staticType().getField(name);
        if (Modifier.isStatic(field.getModifiers())) {
            String msg = String.format(Locale.ROOT,
                    "%s: required instance field name, but %s.%s is static field",
                    desc,
                    jv.staticType().getName(),
                    name);
            return c.raise(msg);
        }

        Object fieldVal = field.get(jv.objectReference());
        return vm.java.of(fieldVal, field.getType());
    }

    // }}}1
    // Java_val.set_field {{{1

    /**
     * Implementation of Java_val.set_field.
     */
    private HostResult setFieldFun(
            HostContext c, JavaVal jv, Val nameVal, Val content
            ) throws ReflectiveOperationException {
        String desc = "Java_val.set_field(Field_name Content)";
        if (jv.objectReference() == null) {
            return c.raise(String.format(Locale.ROOT,
                    "%s: required non-null java_val as \\recv, but got null",
                    desc));
        }

        if (! (nameVal instanceof StrVal)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Field_name must be a str, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(nameVal)));
        }
        String name = ((StrVal) nameVal).string();

        if (! (content instanceof JavaVal)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Content must be a java_val, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(content)));
        }
        Object contentObj = ((JavaVal) content).objectReference();

        Field field = jv.staticType().getField(name);
        if (Modifier.isStatic(field.getModifiers())) {
            String msg = String.format(Locale.ROOT,
                    "%s: required instance field name, but %s.%s is static field",
                    desc,
                    jv.staticType().getName(), name);
            return c.raise(msg);
        }

        field.set(jv.objectReference(), contentObj);
        return vm.nada;
    }

    // }}}1
    // Java_val.get_static {{{1

    /**
     * Implementation of Java_val.get_static.
     */
    private HostResult getStaticFun(
            String desc, HostContext c, JavaVal jv, Val nameVal
            ) throws ReflectiveOperationException {
        if (jv.objectReference() == null) {
            return c.raise(String.format(Locale.ROOT,
                        "%s: required class as \\recv, but got null",
                        desc));
        }

        if (! (jv.objectReference() instanceof Class)) {
            Class<?> dynamicClass = jv.objectReference().getClass();
            String msg = String.format(Locale.ROOT,
                    "%s: required class as \\recv, but got %s: %s",
                    desc,
                    dynamicClass.getName(),
                    jv.objectReference());
            return c.raise(msg);
        }
        Class<?> klass = (Class<?>) jv.objectReference();

        if (! (nameVal instanceof StrVal)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Field_name must be a str, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(nameVal)));
        }
        String name = ((StrVal) nameVal).string();

        Field field = klass.getField(name);
        if (! Modifier.isStatic(field.getModifiers())) {
            String msg = String.format(Locale.ROOT,
                    "%s: required static field name, but %s.%s is instance field",
                    desc,
                    klass.getName(),
                    name);
            return c.raise(msg);
        }

        Object fieldVal = field.get(null);
        return vm.java.of(fieldVal, field.getType());
    }

    // }}}1
    // Java_val.set_static {{{1

    /**
     * Implementation of Java_val.set_static.
     */
    private HostResult setStaticFun(
            HostContext c, JavaVal jv, Val nameVal, Val content
            ) throws ReflectiveOperationException {
        Class<?> klass = asClass(jv);
        String desc = "Java_val.set_static(Field_name Content)";
        if (klass == null) {
            return c.call(vm.graph.raiseFormat(
                        "{}: required class java_val for \\recv, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(jv)));
        }

        if (! (nameVal instanceof StrVal)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Field_name must be a str, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(nameVal)));
        }
        String name = ((StrVal) nameVal).string();

        Field field = klass.getField(name);
        if (! Modifier.isStatic(field.getModifiers())) {
            String msg = String.format(Locale.ROOT,
                    "%s: required static field name, but %s.%s is instance field",
                    desc,
                    klass.getName(),
                    name);
            return c.raise(msg);
        }

        if (! (content instanceof JavaVal)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Content must be a java_val, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(content)));
        }
        Object contentObj = ((JavaVal) content).objectReference();

        field.set(null, contentObj);
        return vm.nada;
    }

    // }}}1
    // Java_val.call_method, call_static, new {{{1

    /**
     * Implementation of Java_val.call_method.
     */
    private HostResult callMethodFun(
            CallContext c, JavaVal recv) throws Throwable {
        String desc = "Java_val.call_method(Method_name Args ...[$config])";
        if (! (c.arg(0) instanceof StrVal)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Method_name must be a str, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(c.arg(0))));
        }
        String name = ((StrVal) c.arg(0)).string();

        Val argsVecVal = c.arg(1);
        if (! (argsVecVal instanceof VecVal)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Args must be a vec, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(argsVecVal)));
        }

        FunVal config;
        if (c.argCount() != 3) {
            config = null;
        } else if (! (c.arg(2) instanceof FunVal)) {
            return c.call(vm.graph.raiseFormat("{}: $config must be fun, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(c.arg(2))));
        } else {
            config = (FunVal) c.arg(2);
        }

        List<Val> argList = ((VecVal) argsVecVal).toList();
        int size = argList.size();
        Class<?>[] argTypes = new Class[size];
        Object[] args = new Object[size];
        Optional<GraphNode> argsError = retrieveArgs(desc, argList, size, argTypes, args);

        if (argsError.isPresent()) {
            return c.call(argsError.get());
        }

        Method method;
        try {
            method = recv.staticType().getMethod(name, argTypes);
        } catch (NoSuchMethodException nsme) {
            return c.raise(String.format(Locale.ROOT,
                        "%s: method not found: %s",
                        desc,
                        nsme.getMessage()));
        }
        if (Modifier.isStatic(method.getModifiers())) {
            return c.raise(String.format(Locale.ROOT,
                        "%s: method %s is a static method",
                        desc,
                        name));
        }
        return withConts(c, config, (cc, successCont, errorCont) -> {
            return invokeMethod(cc, method, recv.objectReference(), args, successCont, errorCont);
        });
    }

    /**
     * Implementation of Java_val.call_static.
     */
    private HostResult callStaticFun(
            CallContext c, JavaVal klassVal
            ) throws Throwable {
        Object klassObj = klassVal.objectReference();
        String desc = "Java_val.call_static(Method_name Args ...[$config])";
        if (! (klassObj instanceof Class)) {
            String msg = String.format(Locale.ROOT,
                    "%s: required class as \\recv, but got %s",
                    desc,
                    objectDesc(klassObj));
            return c.raise(msg);
        }
        Class<?> klass = (Class<?>) klassObj;
        if (! (c.arg(0) instanceof StrVal)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Method_name must be a str, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(c.arg(0))));
        }
        String name = ((StrVal) c.arg(0)).string();

        Val argsVecVal = c.arg(1);
        if (! (argsVecVal instanceof VecVal)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Args must be a vec, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(argsVecVal)));
        }

        FunVal config;
        if (c.argCount() != 3) {
            config = null;
        } else if (! (c.arg(2) instanceof FunVal)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: $config must be fun, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(c.arg(2))));
        } else {
            config = (FunVal) c.arg(2);
        }

        List<Val> argList = ((VecVal) argsVecVal).toList();
        int size = argList.size();
        Class<?>[] argTypes = new Class[size];
        Object[] args = new Object[size];
        Optional<GraphNode> argsError = retrieveArgs(desc, argList, size, argTypes, args);

        if (argsError.isPresent()) {
            return c.call(argsError.get());
        }
        Method method;
        try {
            method = klass.getMethod(name, argTypes);
        } catch (NoSuchMethodException nsme) {
            return c.raise(String.format(Locale.ROOT,
                        "%s: method not found: %s",
                        desc,
                        nsme.getMessage()));
        }
        if (! Modifier.isStatic(method.getModifiers())) {
            return c.raise(String.format(Locale.ROOT,
                        "%s: method %s is not a static method",
                        desc,
                        name));
        }
        return withConts(c, config, (cc, successCont, errorCont) -> {
            return invokeMethod(cc, method, null, args, successCont, errorCont);
        });
    }

    /**
     * Implementation of Java_val.new.
     */
    private HostResult newFun(CallContext c, JavaVal klassVal) throws Throwable {
        String desc = "Java_val.new(Args ...[$config])";
        Object klassObj = klassVal.objectReference();
        if (! (klassObj instanceof Class)) {
            return c.raise(String.format(Locale.ROOT,
                        "%s: required class as \\recv, but got %s",
                        desc,
                        objectDesc(klassObj)));
        }
        Class<?> klass = (Class<?>) klassObj;

        Val argsVecVal = c.arg(0);
        if (! (argsVecVal instanceof VecVal)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Args must be a vec, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(argsVecVal)));
        }
        List<Val> argList = ((VecVal) argsVecVal).toList();

        @Nullable
        FunVal config;
        if (c.argCount() == 1) {
            config = null;
        } else if (c.arg(1) instanceof FunVal) {
            config = (FunVal) c.arg(1);
        } else {
            return c.call(vm.graph.raiseFormat("{}: $config must be a fun, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(c.arg(1))));
        }

        int size = argList.size();
        Class<?>[] argTypes = new Class[size];
        Object[] args = new Object[size];
        Optional<GraphNode> argsError = retrieveArgs(desc, argList, size, argTypes, args);

        if (argsError.isPresent()) {
            return c.call(argsError.get());
        }
        Constructor ctor;
        try {
            ctor = klass.getConstructor(argTypes);
        } catch (NoSuchMethodException nsme) {
            return c.raise(String.format(Locale.ROOT,
                        "Java_val.new(A0 A1 ,,,): constructor not found: %s", nsme.getMessage()));
        }
        return withConts(c, config, (cc, successCont, errorCont) -> {
            return invokeConstructor(cc, ctor, args, successCont, errorCont);
        });
    }

    /**
     * Get args from the vec into the arrays.
     * Returns an non-null graph node on an error.
     */
    private Optional<GraphNode> retrieveArgs(
            String prefix, List<Val> argList, int size, Class<?>[] argTypes, Object[] args) {
        for (int i = 0; i < size; ++ i) {
            Val arg = argList.get(i);
            if (! (arg instanceof JavaVal)) {
                return Optional.of(vm.graph.raiseFormat(
                            "{}: required java_val for Args#{}, but got {}",
                            vm.graph.of(vm.str.of(prefix)),
                            vm.graph.of(vm.num.of(i)),
                            vm.graph.repr(arg)));
            }
            argTypes[i] = ((JavaVal) arg).staticType();
            args[i] = ((JavaVal) arg).objectReference();
        }
        return Optional.empty();
    }

    /**
     * Invokes the constructor.
     */
    private HostResult invokeConstructor(
            CallContext c,
            Constructor ctor,
            Object[] args,
            @Nullable FunVal successCont,
            @Nullable FunVal errorCont
            ) {
        Object result;
        try {
            result = ctor.newInstance(args);
        } catch (ReflectiveOperationException roe) {
            Throwable th = unwrapReflectionException(roe);
            return errorCont != null
                ? c.call(errorCont).args(vm.java.of(th, Throwable.class))
                : c.raise(th);
        }
        JavaVal resultJavaVal = vm.java.of(result, ctor.getDeclaringClass());
        return successCont != null
            ? c.call(successCont).args(resultJavaVal)
            : resultJavaVal;
    }

    /**
     * Invokes the method.
     */
    private HostResult invokeMethod(
            HostContext c,
            Method method,
            Object recv,
            Object[] args,
            @Nullable FunVal successCont,
            @Nullable FunVal errorCont) {
        Class<?> returnType = method.getReturnType();
        Object result;
        try {
            result = method.invoke(recv, args);
        } catch (ReflectiveOperationException roe) {
            Throwable th = unwrapReflectionException(roe);
            return errorCont != null
                ? c.call(errorCont).args(vm.java.of(th, Throwable.class))
                : c.raise(th);
        }
        if (successCont == null) {
            return returnType.equals(void.class)
                ? vm.nada
                : vm.java.of(result, returnType);
        } else if (returnType.equals(void.class)) {
            return c.call(successCont).args();
        } else {
            Val resultVal = vm.java.of(result, returnType);
            return c.call(successCont).args(resultVal);
        }
    }

    /**
     * Invokes invoke with success and error conts.
     */
    private HostResult withConts(
            CallContext c,
            @Nullable FunVal config,
            ThrowingFunction3<CallContext, FunVal, FunVal, HostResult> invoke
            ) throws Throwable {
        if (config == null) {
            return invoke.apply(c, null, null);
        }
        FunVal cont = vm.fun.make().take(2).action(cc -> contToInvoke(cc, invoke));
        return c.call("kink/_java/CALL_AUX", readConfHandle)
            .args(config, this.invDefSuccessCont, this.invDefErrorCont, cont);
    }

    /**
     * Continues to invoke, checking errors.
     */
    HostResult contToInvoke(
            CallContext c,
            ThrowingFunction3<CallContext, FunVal, FunVal, HostResult> invoke
            ) throws Throwable {
        Val successContVal = c.arg(0);
        if (! (successContVal instanceof FunVal)) {
            return c.call(vm.graph.raiseFormat(
                        "(cont-to-invoke($success_cont $error_cont)):"
                        + " $success_cont must be a fun, but got {}",
                        vm.graph.repr(successContVal)));
        }

        Val errorContVal = c.arg(1);
        if (! (errorContVal instanceof FunVal)) {
            return c.call(vm.graph.raiseFormat(
                        "(cont-to-invoke($success_cont $error_cont)):"
                        + " $error_cont must be a fun, but got {}",
                        vm.graph.repr(errorContVal)));
        }

        return invoke.apply(c, (FunVal) successContVal, (FunVal) errorContVal);
    }

    /**
     * Implementation of the default success cont of invocation.
     */
    HostResult invDefSuccessCont(CallContext c) {
        return c.argCount() == 0
            ? vm.nada
            : c.arg(0);
    }

    /**
     * Implementation of the default error cont of invocation.
     */
    HostResult invDefErrorCont(CallContext c) {
        Val exVal = c.arg(0);
        Supplier<HostResult> raise = () -> c.call(vm.graph.raiseFormat(
                    "(default-error-cont(Exc_java_val)):"
                    + " Exc_java_val must be a java_val of Throwable, but got {}",
                    vm.graph.repr(exVal)));

        if (! (exVal instanceof JavaVal)) {
            return raise.get();
        }

        Object thObj = ((JavaVal) exVal).objectReference();
        if (! (thObj instanceof Throwable)) {
            return raise.get();
        }

        return c.raise((Throwable) thObj);
    }

    /**
     * Unwrap causing exception in the invocation target if any.
     */
    static Throwable unwrapReflectionException(ReflectiveOperationException roe) {
        if (roe instanceof InvocationTargetException) {
            Throwable cause = roe.getCause();
            if (cause != null) {
                return cause;
            }
        }
        return roe;
    }

    // }}}1
    // Java_val.repr {{{1

    /**
     * Implementation of Java_val.repr.
     */
    private HostResult reprFun(HostContext c, JavaVal java) {
        return vm.str.of(String.format(Locale.ROOT,
                    "(java_val %s as %s)", java.objectReference(), java.staticType().getName()));
    }

    // }}}1

    /**
     * Continues to onOk if val is a java_val of klass,
     * or to onError.
     */
    private HostResult forValOf(Class<?> klass, Val val,
                                ThrowingFunction0<HostResult> onError,
                                ThrowingFunction1<Object, HostResult> onOk) throws Throwable {
        if (! (val instanceof JavaVal)) {
            return onError.apply();
        }

        Object obj = ((JavaVal) val).objectReference();
        if (! isTypable(obj, klass)) {
            return onError.apply();
        }

        return onOk.apply(obj);
    }

    /**
     * Retrieves a class from the klassVal which is expected to be a class java_val.
     */
    private Class<?> asClass(Val klassVal) {
        if (! (klassVal instanceof JavaVal)) {
            return null;
        }

        JavaVal klassJavaVal = (JavaVal) klassVal;
        Object klassObj = klassJavaVal.objectReference();
        if (! (klassObj instanceof Class)) {
            return null;
        }

        return (Class<?>) klassObj;
    }

    /**
     * Makes a varargs method fun.
     */
    private FunVal method(
            String prefix,
            ThrowingFunction2<CallContext, JavaVal, HostResult> action) {
        return method(prefix, UnaryOperator.identity(), action);
    }

    /**
     * Makes a method fun arity of which is [min, max].
     */
    private FunVal methodMinMax(
            String prefix,
            int min, int max,
            ThrowingFunction2<CallContext, JavaVal, HostResult> action) {
        return method(prefix, b -> b.takeMinMax(min, max), action);
    }

    /**
     * Makes a fun.
     */
    private FunVal method(
            String prefix,
            UnaryOperator<HostFunBuilder> configFun,
            ThrowingFunction2<CallContext, JavaVal, HostResult> action) {
        return configFun.apply(vm.fun.make(prefix)).action(c -> {
            if (! (c.recv() instanceof JavaVal)) {
                // TODO use raiseFormat
                String msg = String.format(Locale.ROOT,
                        "%s: required java_val as \\recv, but was not", prefix);
                return c.raise(msg);
            }
            return action.apply(c, (JavaVal) c.recv());
        });
    }

    /**
     * Makes a nullary method fun.
     */
    private FunVal method0(
            String prefix,
            ThrowingFunction2<CallContext, JavaVal, HostResult> action) {
        return vm.fun.make(prefix).take(0).action(c -> {
            if (! (c.recv() instanceof JavaVal)) {
                String msg = String.format(Locale.ROOT,
                        "%s: required java_val as \\recv, but was not", prefix);
                return c.raise(msg);
            }
            return action.apply(c, (JavaVal) c.recv());
        });
    }

    /**
     * Adds an unary method fun.
     */
    private void addMethod1(
            Map<Integer, Val> mapping,
            String recvDesc,
            String name,
            String arg0Desc,
            ThrowingFunction4<String, CallContext, JavaVal, Val, HostResult> action) {
        String desc = String.format(Locale.ROOT, "%s.%s(%s)", recvDesc, name, arg0Desc);
        FunVal fun = vm.fun.make(desc).take(1).action(c -> {
            Val recv = c.recv();
            if (! (recv instanceof JavaVal recvJavaVal)) {
                return c.call(vm.graph.raiseFormat("{}: {} must be java val, but got {}",
                            vm.graph.of(vm.str.of(desc)),
                            vm.graph.of(vm.str.of(recvDesc)),
                            vm.graph.repr(recv)));
            }
            return action.apply(desc, c, recvJavaVal, c.arg(0));
        });
        mapping.put(vm.sym.handleFor(name), fun);
    }

    /**
     * Makes binary method fun.
     */
    private FunVal method2(
            String prefix,
            ThrowingFunction4<CallContext, JavaVal, Val, Val, HostResult> action) {
        return vm.fun.make(prefix).take(2).action(c -> {
            if (! (c.recv() instanceof JavaVal)) {
                String msg = String.format(Locale.ROOT,
                        "%s: required java_val as \\recv, but was not", prefix);
                return c.raise(msg);
            }
            return action.apply(c, (JavaVal) c.recv(), c.arg(0), c.arg(1));
        });
    }

    /**
     * Returns an error message if the klassObj is not a proper class as array components.
     */
    private Optional<String> checkErrorForArrayComponentClass(String prefix, Object klassObj) {
        if (! (klassObj instanceof Class)) {
            return Optional.of(String.format(Locale.ROOT,
                    "%s: requires class as \\recv, but got %s",
                    prefix, objectDesc(klassObj)));
        }
        Class<?> klass = (Class<?>) klassObj;

        if (klass.equals(void.class)) {
            return Optional.of(String.format(Locale.ROOT,
                        "%s: no array class for void", prefix));
        }

        int dimension = arrayDimension(klass);
        if (dimension >= 255) {
            return Optional.of(String.format(Locale.ROOT,
                    "%s: too big array dimension: %d", prefix, dimension + 1));
        }

        return Optional.empty();
    }

    /**
     * Returns an error message if the obj is not an array.
     */
    private Optional<String> checkErrorForArray(String prefix, Object obj) {
        if (obj == null || ! obj.getClass().isArray()) {
            return Optional.of(String.format(Locale.ROOT,
                        "%s: \\recv is not array, but was %s",
                        prefix, objectDesc(obj)));
        } else {
            return Optional.empty();
        }
    }

    /**
     * Returns the array dimension of the class.
     */
    private static int arrayDimension(Class<?> klass) {
        int dimension = 0;
        while (klass.isArray()) {
            ++ dimension;
            klass = klass.getComponentType();
        }
        return dimension;
    }

    /**
     * Returns a description of the object reference.
     */
    private String objectDesc(@Nullable Object obj) {
        return obj == null
            ? "null"
            : String.format(Locale.ROOT, "%s: %s", obj.getClass().getName(), obj);
    }

    /**
     * Returns true if the object is typable as {@code staticType}.
     *
     * @param object the object to test.
     * @param staticType the type to test.
     * @return {@code true} if the object is typable as {@code staticType}.
     */
    public boolean isTypable(@Nullable Object object, Class<?> staticType) {
        return object == null && ! staticType.isPrimitive()
            || staticType.isInstance(object)
            || object instanceof Boolean && staticType.equals(boolean.class)
            || object instanceof Character && staticType.equals(char.class)
            || object instanceof Byte && staticType.equals(byte.class)
            || object instanceof Short && staticType.equals(short.class)
            || object instanceof Integer && staticType.equals(int.class)
            || object instanceof Long && staticType.equals(long.class)
            || object instanceof Float && staticType.equals(float.class)
            || object instanceof Double && staticType.equals(double.class);
    }

}

// vim: et sw=4 sts=4 fdm=marker
