package org.kink_lang.kink;

import java.util.Locale;

import javax.annotation.Nullable;

/**
 * A java val.
 *
 * <p>A java val consists of an object reference, and a static type.
 * the object reference is always typable as the static type.
 * Namely,</p>
 *
 * <ul>
 *   <li>the object is null and the static type is not a primitive type,</li>
 *   <li>the object is a boxed value and the static type is the corresponding primitive type,</li>
 *   <li>or the object is not null and an instance of the static type.</li>
 * </ul>
 */
public class JavaVal extends Val {

    /** The nullable java object reference. */
    @Nullable
    private final Object objectReference;

    /** The static type of the java object. */
    private final Class<?> staticType;

    /**
     * Constructs a java val.
     */
    JavaVal(Vm vm, @Nullable Object objectReference, Class<?> staticType) {
        super(vm, null);
        if (! vm.java.isTypable(objectReference, staticType)) {
            String msg = String.format(Locale.ROOT,
                    "objectReference (%s) must be able to typed as staticType %s",
                    objectReference, staticType.getName());
            throw new IllegalArgumentException(msg);
        }
        this.objectReference = objectReference;
        this.staticType = staticType;
    }

    /**
     * Returns the java object reference.
     *
     * @return the java object reference.
     */
    @Nullable
    public Object objectReference() {
        return this.objectReference;
    }

    /**
     * Returns the static type of the java object reference.
     *
     * @return the static type of the java object reference.
     */
    public Class<?> staticType() {
        return this.staticType;
    }

    @Override
    SharedVars sharedVars() {
        return vm.java.sharedVars;
    }

    @Override
    public String toString() {
        return String.format(Locale.ROOT,
                "JavaVal(%s as %s)", objectReference, staticType.getName());
    }

    @Override
    public int hashCode() {
        return 13
            + 11 * this.vm.hashCode()
            + 41 * System.identityHashCode(this.objectReference)
            + 113 * this.staticType.hashCode();
    }

    /**
     * Returns {@code true} if the argument is equal to {@code this} java val.
     *
     * <p>Namely, if</p>
     *
     * <ul>
     *   <li>they are members of the same vm,</li>
     *   <li>the object references are identical (via Java == operator),</li>
     *   <li>and the static types are equal.</li>
     * </ul>
     *
     * @param arg the object to test.
     * @return {@code true} if the argument is equal to {@code this} java val.
     */
    @Override
    public boolean equals(Object arg) {
        return arg == this
            || arg instanceof JavaVal argJava
            && this.vm.equals(argJava.vm)
            && this.objectReference == argJava.objectReference
            && this.staticType.equals(argJava.staticType);
    }

}

// vim: et sw=4 sts=4 fdm=marker
