package org.kink_lang.kink;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;

import javax.annotation.Nullable;

import org.kink_lang.kink.internal.function.ThrowingFunction2;
import org.kink_lang.kink.internal.function.ThrowingFunction3;
import org.kink_lang.kink.internal.function.ThrowingFunction4;
import org.kink_lang.kink.hostfun.HostFunReaction;
import org.kink_lang.kink.hostfun.HostResult;
import org.kink_lang.kink.hostfun.graph.GraphNode;
import org.kink_lang.kink.hostfun.HostContext;
import org.kink_lang.kink.hostfun.CallContext;
import org.kink_lang.kink.internal.intrinsicsupport.FunsHolder;
import org.kink_lang.kink.internal.vec.VecInternal;
import org.kink_lang.kink.internal.contract.Preconds;
import org.kink_lang.kink.internal.num.NumOperations;

/**
 * The helper of vec vals.
 */
public class VecHelper {

    /** The vm. */
    private final Vm vm;

    /** The empty vecInternal. */
    private VecInternal empty;

    /** The sym handle of op_store. */
    private int storeHandle;

    /** The sym handle of pass_arg. */
    private int passArgHandle;

    /** The sym handle of pass_nothing. */
    private int passNothingHandle;

    /** The sym handle of pass_rest. */
    private int passRestHandle;

    /** The sym handle of op_eq. */
    private int opEqHandle;

    /** The sym handle of op_lt. */
    private int opLtHandle;

    /** The sym handle of repr_vec. */
    private int reprVecHandle;

    /** The sym handle of each. */
    private int eachHandle;

    /** The sym handle of sort. */
    private int sortHandle;

    /** The sym handle of all?. */
    private int allP;

    /** The sym handle of any?. */
    private int anyP;

    /** The sym handle of have??. */
    private int haveP;

    /** The fun {(:X :Y) X &lt; Y }. */
    private FunVal defaultPrecedesFun;

    /** Trait of vecs. */
    SharedVars sharedVars;

    /**
     * Constructs the helper.
     */
    VecHelper(Vm vm) {
        this.vm = vm;
    }

    /**
     * Initializes the helper.
     */
    void init() {
        this.empty = VecInternal.of(vm, new Val[0], 0, 0);
        this.storeHandle = vm.sym.handleFor("op_store");
        this.passArgHandle = vm.sym.handleFor("pass_arg");
        this.passNothingHandle = vm.sym.handleFor("pass_nothing");
        this.passRestHandle = vm.sym.handleFor("pass_rest");
        this.opEqHandle = vm.sym.handleFor("op_eq");
        this.opLtHandle = vm.sym.handleFor("op_lt");
        this.reprVecHandle = vm.sym.handleFor("repr_vec");
        this.eachHandle = vm.sym.handleFor("each");
        this.sortHandle = vm.sym.handleFor("sort");
        this.allP = vm.sym.handleFor("all?");
        this.anyP = vm.sym.handleFor("any?");
        this.haveP = vm.sym.handleFor("have?");

        this.defaultPrecedesFun = vm.fun.make("(compare)")
            .take(2).action(c -> c.call(c.arg(0), opLtHandle).args(c.arg(1)));

        Map<Integer, Val> vars = new HashMap<>();
        vars.put(vm.sym.handleFor("size"), method0("Vec.size",
                    (c, vec) -> vm.num.of(vec.vecInternal().size())));
        vars.put(vm.sym.handleFor("empty?"), method0("Vec.empty?",
                    (c, vec) -> vm.bool.of(vec.vecInternal().size() == 0)));
        vars.put(vm.sym.handleFor("get"), method1("Vec.get(Ind)", this::getMethod));
        vars.put(vm.sym.handleFor("front"), method0("Vec.front", this::frontMethod));
        vars.put(vm.sym.handleFor("back"), method0("Vec.back", this::backMethod));
        vars.put(vm.sym.handleFor("set"), method2("Vec.set(Ind Elem)", this::setMethod));
        vars.put(vm.sym.handleFor("dup"), method0("Vec.dup", this::dupMethod));
        vars.put(vm.sym.handleFor("rev"), method0("Vec.rev", this::revMethod));
        vars.put(vm.sym.handleFor("take_front"), takeDropMethod("Vec.take_front(N)", 0, 0, 1, 0));
        vars.put(vm.sym.handleFor("take_back"), takeDropMethod("Vec.take_back(N)", -1, 1, 0, 1));
        vars.put(vm.sym.handleFor("drop_front"), takeDropMethod("Vec.drop_front(N)", 1, 0, 0, 1));
        vars.put(vm.sym.handleFor("drop_back"), takeDropMethod("Vec.drop_back(N)", 0, 0, -1, 1));
        vars.put(vm.sym.handleFor("slice"),
                method2("Vec.slice(From_pos To_pos)", this::sliceMethod));
        vars.put(vm.sym.handleFor("clear"), method0("Vec.clear", this::clearMethod));
        vars.put(vm.sym.handleFor("clear_slice"),
                method2("Vec.clear_slice(From_pos To_pos)", this::clearSliceMethod));
        vars.put(vm.sym.handleFor("push_front"),
                method1("Vec.push_front(Elem)", this::pushFront));
        vars.put(vm.sym.handleFor("push_back"),
                method1("Vec.push_back(Elem)", this::pushBack));
        vars.put(vm.sym.handleFor("push_at"),
                method2("Vec.push_at(Ind Elem)", this::pushAt));
        vars.put(vm.sym.handleFor("push_each_front"),
                method1("Vec.push_each_front(Eacher)", this::pushEachFront));
        vars.put(vm.sym.handleFor("push_each_back"),
                method1("Vec.push_each_back(Eacher)", this::pushEachBack));
        vars.put(vm.sym.handleFor("push_each_at"),
                method2("Vec.push_each_at(Ind Eacher)", this::pushEachAt));
        vars.put(vm.sym.handleFor("pop_front"), method0("Vec.pop_front", this::popFrontMethod));
        vars.put(vm.sym.handleFor("pop_back"), method0("Vec.pop_back", this::popBackMethod));
        vars.put(vm.sym.handleFor("pop_at"), method1("Vec.pop_at(Ind)", this::popAtMethod));
        vars.put(vm.sym.handleFor("concat"), method0("Vec.concat", this::concatMethod));
        vars.put(vm.sym.handleFor("chunk"), method1("Vec.chunk(Chunk_size)", this::chunkMethod));
        vars.put(vm.sym.handleFor("just"), method0("Vec.just", this::justMethod));
        vars.put(vm.sym.handleFor("just_or"), method1("Vec.just_or", this::justOrMethod));
        vars.put(vm.sym.handleFor("with_just_or"),
                method2("Vec.with_just_or($on_just $on_empty)", this::withJustOrMethod));
        vars.put(vm.sym.handleFor("each"), method1("Vec.each($on_elem)", this::eachMethod));
        vars.put(vm.sym.handleFor("map"), method1("Vec.map($trans)", this::mapMethod));
        vars.put(vm.sym.handleFor("concat_map"),
                method1("Vec.concat_map($trans_to_vec)", this::concatMapMethod));
        vars.put(vm.sym.handleFor("filter"), method1("Vec.filter($match?)", this::filterMethod));
        vars.put(vm.sym.handleFor("count"), method1("Vec.count($match?)", this::countMethod));
        vars.put(vm.sym.handleFor("fold"), method2("Vec.fold(Init $combine)", this::foldMethod));
        vars.put(vm.sym.handleFor("reduce"), method1("Vec.reduce", this::reduceMethod));
        vars.put(vm.sym.handleFor("scan"), method2("Vec.scan(Init $combine)", this::scanMethod));
        vars.put(vm.sym.handleFor("scan_inside"),
                method1("Vec.scan_inside($combine)", this::scanInsideMethod));
        vars.put(vm.sym.handleFor("take_while"),
                method1("Vec.take_while($match?)", this::takeWhileMethod));
        vars.put(vm.sym.handleFor("drop_while"),
                method1("Vec.drop_while($match?)", this::dropWhileMethod));
        vars.put(vm.sym.handleFor("all?"),
                method1("Vec.all?($match?)", this::allPMethod));
        vars.put(vm.sym.handleFor("any?"),
                method1("Vec.any?($match?)", this::anyPMethod));
        vars.put(vm.sym.handleFor("have?"), method1("Vec.have?", this::havePMethod));
        vars.put(vm.sym.handleFor("have_all?"),
                method1("Vec.have_all?(For_aller)", this::haveAllPMethod));
        vars.put(vm.sym.handleFor("have_any?"),
                method1("Vec.have_any??(For_anyer)", this::haveAnyPMethod));
        vars.put(vm.sym.handleFor("search"),
                    method2("Vec.search(Min_ind $accept?)", this::searchMethod));
        vars.put(vm.sym.handleFor("sort"),
                vm.fun.make("Vec.sort($precede?)").takeMinMax(0, 1).action(this::sortMethod));
        vars.put(vm.sym.handleFor("iter"), method0("Vec.iter", this::iterMethod));
        vars.put(vm.sym.handleFor("iter_from"),
                method1("Vec.iter_from(Min_ind)", this::iterFromMethod));
        vars.put(vm.sym.handleFor("op_add"), binOp("Vec.op_add", this::opAddMethod));
        vars.put(storeHandle, method1("Vec.op_store(Rhs)", this::opStoreMethod));
        vars.put(vm.sym.handleFor("op_mul"), method1("Vec.op_mul(Count)", this::opMulMethod));
        vars.put(opEqHandle, binOp("Vec1.op_eq(Vec2)", this::opEqMethod));
        vars.put(vm.sym.handleFor("op_lt"), binOp("Vec1.op_lt(Vec2)", this::opLtMethod));
        vars.put(vm.sym.handleFor("repr"), method0("Vec.repr", this::reprMethod));
        this.sharedVars = vm.sharedVars.of(vars);
    }

    // Vec.get {{{1

    /**
     * Implementation of Vec.get.
     */
    private HostResult getMethod(HostContext c, VecVal vec, Val indVal) {
        VecInternal vecInternal = vec.vecInternal();
        int size = vecInternal.size();
        int ind = NumOperations.getElemIndex(indVal, size);
        if (ind < 0) {
            return c.call(vm.graph.raiseFormat(
                        "Vec.get(Ind): required int num in [0, {}) for Ind, but got {}",
                        vm.graph.of(vm.num.of(size)),
                        vm.graph.repr(indVal)));
        }
        return vecInternal.get(ind);
    }

    // }}}1
    // Vec.front {{{1

    /**
     * Implementation of Vec.front.
     */
    private HostResult frontMethod(HostContext c, VecVal vec) {
        VecInternal vecInternal = vec.vecInternal();
        int size = vecInternal.size();
        if (size == 0) {
            return c.raise("Vec.front: vec is empty");
        }
        return vecInternal.get(0);
    }

    // }}}1
    // Vec.back {{{1

    /**
     * Implementation of Vec.back.
     */
    private HostResult backMethod(HostContext c, VecVal vec) {
        VecInternal vecInternal = vec.vecInternal();
        int size = vecInternal.size();
        if (size == 0) {
            return c.raise("Vec.back: vec is empty");
        }
        return vecInternal.get(size - 1);
    }

    // }}}1
    // Vec.set {{{1

    /**
     * Implementation of Vec.set.
     */
    private HostResult setMethod(HostContext c, VecVal vec, Val indVal, Val elem) {
        VecInternal vecInternal = vec.vecInternal();
        int size = vecInternal.size();
        int ind = NumOperations.getElemIndex(indVal, size);
        if (ind < 0) {
            return c.call(vm.graph.raiseFormat(
                        "Vec.set(Ind Elem): required int num in [0, {}) for Ind, but got {}",
                        vm.graph.of(vm.num.of(size)),
                        vm.graph.repr(indVal)));
        }
        vec.setVecInternal(vecInternal.set(ind, elem));
        return vm.nada;
    }

    // }}}1
    // Vec.dup {{{1

    /**
     * Implementation of Vec.dup.
     */
    private HostResult dupMethod(HostContext c, VecVal vec) {
        VecInternal vecInternal = vec.vecInternal();
        return new VecVal(vm, vecInternal.copy());
    }

    // }}}1
    // Vec.rev {{{1

    /**
     * Implementation of Vec.rev.
     */
    private HostResult revMethod(HostContext c, VecVal vec) {
        VecInternal src = vec.vecInternal();
        int size = src.size();
        VecInternal dest = VecInternal.ofCapa(vm, size);
        for (int i = 0; i < size; ++ i) {
            dest = dest.append(src.get(size - i - 1));
        }
        return new VecVal(vm, dest);
    }

    // }}}1
    // Vec.take_front, take_back, drop_front, drop_back {{{1

    /**
     * Makes take_front/take_back/drop_front/drop_back method funs.
     */
    private FunVal takeDropMethod(
            String prefix,
            int fromNumCoef, int fromSizeCoef, int toNumCoef, int toSizeCoef) {
        return vm.fun.make(prefix).take(1).action(c -> {
            Val recv = c.recv();
            if (! (recv instanceof VecVal)) {
                return c.call(vm.graph.raiseFormat("{}: required vec for \\recv, but got {}",
                            vm.graph.of(vm.str.of(prefix)),
                            vm.graph.repr(recv)));
            }
            VecInternal vecInternal = ((VecVal) recv).vecInternal();
            int size = vecInternal.size();

            Val numVal = c.arg(0);
            int num = NumOperations.getPosIndex(numVal, size);
            if (num < 0) {
                return c.call(vm.graph.raiseFormat(
                            "{}: N must be an int num in [0, {}], but got {}",
                            vm.graph.of(vm.str.of(prefix)),
                            vm.graph.of(vm.num.of(size)),
                            vm.graph.repr(numVal)));
            }
            int from = fromNumCoef * num + fromSizeCoef * size;
            int to = toNumCoef * num + toSizeCoef * size;
            return new VecVal(vm, vecInternal.copyRange(from, to));
        });
    }

    // }}}1
    // Vec.slice {{{1

    /**
     * Implementation of Vec.slice.
     */
    private HostResult sliceMethod(
            HostContext c, VecVal vec, Val fromVal, Val toVal) {
        VecInternal vecInternal = vec.vecInternal();
        int size = vecInternal.size();
        GraphNode error = checkRange("Vec.slice(From_pos To_pos)", size, fromVal, toVal);
        if (error != null) {
            return c.call(error);
        }

        BigDecimal from = ((NumVal) fromVal).bigDecimal();
        BigDecimal to = ((NumVal) toVal).bigDecimal();
        return new VecVal(vm, vecInternal.copyRange(from.intValue(), to.intValue()));
    }

    // }}}1
    // Vec.clear {{{1

    /**
     * Implementation of Vec.clear.
     */
    private HostResult clearMethod(HostContext c, VecVal vec) {
        vec.setVecInternal(this.empty);
        return vm.nada;
    }

    // }}}1
    // Vec.clear_slice {{{1

    /**
     * Implementation of Vec.clear_slice.
     */
    private HostResult clearSliceMethod(
            HostContext c, VecVal vec, Val fromVal, Val toVal) {
        VecInternal vecInternal = vec.vecInternal();
        int size = vecInternal.size();
        GraphNode error = checkRange("Vec.clear_slice(From_pos To_pos)", size, fromVal, toVal);
        if (error != null) {
            return c.call(error);
        }

        BigDecimal from = ((NumVal) fromVal).bigDecimal();
        BigDecimal to = ((NumVal) toVal).bigDecimal();
        vec.setVecInternal(vecInternal.removeRange(from.intValue(), to.intValue()));
        return vm.nada;
    }

    // }}}1
    // Vec.push_front(Elem) {{{1

    /**
     * Implementation of Vec.push_front.
     */
    private HostResult pushFront(HostContext c, VecVal vec, Val elem) {
        VecInternal vecInternal = vec.vecInternal();
        vec.setVecInternal(vecInternal.insert(0, elem));
        return vm.nada;
    }

    // }}}1
    // Vec.push_back(Elem) {{{1

    /**
     * Implementation of Vec.push_back.
     */
    private HostResult pushBack(HostContext c, VecVal vec, Val elem) {
        VecInternal vecInternal = vec.vecInternal();
        vec.setVecInternal(vecInternal.append(elem));
        return vm.nada;
    }

    // }}}1
    // Vec.push_at(Ind Elem) {{{1

    /**
     * Implementation of Vec.push_at.
     */
    private HostResult pushAt(
            HostContext c, VecVal vec, Val indVal, Val elem) {
        VecInternal vecInternal = vec.vecInternal();
        int size = vecInternal.size();
        int ind = NumOperations.getPosIndex(indVal, size);
        if (ind < 0) {
            return c.call(vm.graph.raiseFormat(
                        "Vec.push_at(Ind Elem): required int num in [0, {}] for Ind, but got {}",
                        vm.graph.of(vm.num.of(size)),
                        vm.graph.repr(indVal)));
        }
        vecInternal = vecInternal.insert(ind, elem);
        vec.setVecInternal(vecInternal);
        return vm.nada;
    }

    // }}}1
    // Vec.push_each_xxx {{{1

    /**
     * Implementation of Vec.push_each_front.
     */
    private HostResult pushEachFront(HostContext c, VecVal vec, Val eacherVal) {
        if (eacherVal instanceof VecVal) {
            VecInternal vecInternal = vec.vecInternal();
            VecInternal eacher = ((VecVal) eacherVal).vecInternal();
            vec.setVecInternal(vecInternal.insertAll(0, eacher));
            return vm.nada;
        } else if (eacherVal.hasVar(eachHandle)) {
            return c.call(eacherVal, eachHandle).args(pushEachAtConsume(vec, 0));
        } else {
            return c.call(vm.graph.raiseFormat(
                        "Vec.push_each_front(Eacher): required Eacher supporting .each, but got {}",
                        vm.graph.repr(eacherVal)));
        }
    }

    /**
     * Implementation of Vec.push_each_back.
     */
    private HostResult pushEachBack(HostContext c, VecVal vec, Val eacherVal) {
        if (eacherVal instanceof VecVal) {
            VecInternal vecInternal = vec.vecInternal();
            int size = vecInternal.size();
            VecInternal eacher = ((VecVal) eacherVal).vecInternal();
            vec.setVecInternal(vecInternal.insertAll(size, eacher));
            return vm.nada;
        } else if (eacherVal.hasVar(eachHandle)) {
            return c.call(eacherVal, eachHandle).args(pushEachBackConsume(vec));
        } else {
            return c.call(vm.graph.raiseFormat(
                        "Vec.push_each_back(Eacher): required Eacher supporting .each, but got {}",
                        vm.graph.repr(eacherVal)));
        }
    }

    /**
     * Called by Adder.each via Vec.push_each_back.
     */
    private FunVal pushEachBackConsume(VecVal vec) {
        return vm.fun.make("Vec.push_each_back-call").take(1).action(c -> {
            VecInternal vecInternal = vec.vecInternal();
            vec.setVecInternal(vecInternal.append(c.arg(0)));
            return vm.nada;
        });
    }

    /**
     * Implementation of Vec.push_each_at.
     */
    private HostResult pushEachAt(
            HostContext c, VecVal vec, Val indVal, Val adderVal) {
        VecInternal vecInternal = vec.vecInternal();
        int size = vecInternal.size();
        int ind = NumOperations.getPosIndex(indVal, size);
        if (ind < 0) {
            return c.call(vm.graph.raiseFormat(
                        "Vec.push_each_at(Ind Eacher):"
                        + " required int num in [0, {}] for Ind, but got {}",
                        vm.graph.of(vm.num.of(size)),
                        vm.graph.repr(indVal)));
        }

        if (adderVal instanceof VecVal) {
            VecInternal adder = ((VecVal) adderVal).vecInternal();
            vec.setVecInternal(vecInternal.insertAll(ind, adder));
            return vm.nada;
        } else if (adderVal.hasVar(eachHandle)) {
            return c.call(adderVal, eachHandle).args(pushEachAtConsume(vec, ind));
        } else {
            return c.call(vm.graph.raiseFormat(
                        "Vec.push_each_at(Ind Eacher):"
                        + " required Eacher supporting .each, but got {}",
                        vm.graph.repr(adderVal)));
        }
    }

    /**
     * Called by Adder.each via Vec.push_each_at and .push_each_front.
     */
    private FunVal pushEachAtConsume(VecVal vec, int initialInd) {
        AtomicInteger indAtomic = new AtomicInteger(initialInd);
        return vm.fun.make("Vec.push_each_at-consume").take(1).action(c -> {
            int ind = indAtomic.getAndIncrement();
            VecInternal vecInternal = vec.vecInternal();
            int size = vecInternal.size();
            if (ind > size) {
                return c.raise("Vec.push_each_at-consume: vec was modified during loop");
            }
            vec.setVecInternal(vecInternal.insert(ind, c.arg(0)));
            return vm.nada;
        });
    }

    // }}}1
    // Vec.pop_front {{{1

    /**
     * Implementation of Vec.pop_front.
     */
    private HostResult popFrontMethod(HostContext c, VecVal vec) {
        VecInternal vecInternal = vec.vecInternal();
        int size = vecInternal.size();
        if (size == 0) {
            return c.raise("Vec.pop_front: could not pop from empty vec");
        }
        Val result = vecInternal.get(0);
        vec.setVecInternal(vecInternal.remove(0));
        return result;
    }

    // }}}1
    // Vec.pop_back {{{1

    /**
     * Implementation of Vec.pop_back.
     */
    private HostResult popBackMethod(HostContext c, VecVal vec) {
        VecInternal vecInternal = vec.vecInternal();
        int size = vecInternal.size();
        if (size == 0) {
            return c.raise("Vec.pop_back: could not pop from empty vec");
        }
        Val result = vecInternal.get(size - 1);
        vec.setVecInternal(vecInternal.remove(size - 1));
        return result;
    }

    // }}}1
    // Vec.pop_at {{{1

    /**
     * Implementation of Vec.pop_at.
     */
    private HostResult popAtMethod(HostContext c, VecVal vec, Val indVal) {
        VecInternal vecInternal = vec.vecInternal();
        int size = vecInternal.size();
        int ind = NumOperations.getElemIndex(indVal, size);
        if (ind < 0) {
            return c.call(vm.graph.raiseFormat(
                        "Vec.pop_at(Ind): Ind msut be an int num in [0, {}], but got {}",
                        vm.graph.of(vm.num.of(size)),
                        vm.graph.repr(indVal)));
        }
        Val result = vecInternal.get(ind);
        vec.setVecInternal(vecInternal.remove(ind));
        return result;
    }

    // }}}1
    // Vec.concat {{{1

    /**
     * Implementation of Vec.concat.
     */
    private HostResult concatMethod(HostContext c, VecVal vecs) {
        VecInternal src = vecs.vecInternal();
        int srcSize = src.size();
        VecInternal result = this.empty;
        for (int i = 0; i < srcSize; ++ i) {
            Val elem = src.get(i);
            if (! (elem instanceof VecVal)) {
                return c.call(vm.graph.raiseFormat(
                            "Vec.concat:"
                            + " required vec of vecs for \\recv, but got {} as the #{} elem",
                            vm.graph.repr(elem),
                            vm.graph.of(vm.num.of(i))));
            }
            VecVal vec = (VecVal) elem;
            result = result.insertAll(result.size(), vec.vecInternal());
        }
        return new VecVal(vm, result);
    }

    // }}}1
    // Vec.chunk(Chunk_size) {{{1

    /**
     * Implementation of Vec.chunk(Chunk_size).
     */
    private HostResult chunkMethod(HostContext c, VecVal vec, Val chunkSizeVal) {
        VecInternal vecInternal = vec.vecInternal();
        int vecSize = vecInternal.size();
        if (! (chunkSizeVal instanceof NumVal)) {
            return c.call(vm.graph.raiseFormat(
                        "Vec.chunk(Chunk_size): required int num for Chunk_size, but got {}",
                        vm.graph.repr(chunkSizeVal)));
        }
        BigDecimal chunkSizeDec = ((NumVal) chunkSizeVal).bigDecimal();
        if (vecInternal.size() == 0) {
            boolean sizeIsPositiveInt = chunkSizeDec.signum() > 0
                && chunkSizeDec.remainder(BigDecimal.ONE).signum() == 0;
            if (! sizeIsPositiveInt) {
                return c.call(vm.graph.raiseFormat(
                            "Vec.chunk(Chunk_size):"
                            + " required positive int num for Chunk_size, but got {}",
                            vm.graph.repr(chunkSizeVal)));
            }
            return vm.vec.of();
        }

        int chunkSize = NumOperations.getPosIndex(chunkSizeDec, vecSize);
        if (chunkSize < 1 || vecSize % chunkSize != 0) {
            return c.call(vm.graph.raiseFormat(
                        "Vec.chunk(Chunk_size):"
                        + " required positive divisor of {} for Chunk_size, but got {}",
                        vm.graph.of(vm.num.of(vecSize)),
                        vm.graph.repr(chunkSizeVal)));
        }
        int numChunks = vecSize / chunkSize;
        VecInternal result = VecInternal.ofCapa(vm, numChunks);
        for (int i = 0; i < numChunks; ++ i) {
            int from = i * chunkSize;
            int to = from + chunkSize;
            VecInternal chunk = vecInternal.copyRange(from, to);
            result = result.append(new VecVal(vm, chunk));
        }
        return new VecVal(vm, result);
    }

    // }}}1
    // Vec.just {{{1

    /**
     * Implementation of Vec.just.
     */
    private HostResult justMethod(HostContext c, VecVal vec) {
        VecInternal vecInternal = vec.vecInternal();
        int size = vecInternal.size();
        if (size != 1) {
            String msg = String.format(Locale.ROOT,
                    "Vec.just: required singleton vec as \\recv, but size was %d", size);
            return c.raise(msg);
        }
        return vecInternal.get(0);
    }

    // }}}1
    // Vec.just_or {{{1

    /**
     * Implementation of Vec.just_or.
     */
    private HostResult justOrMethod(HostContext c, VecVal vec, Val onEmpty) {
        VecInternal vecInternal = vec.vecInternal();
        int size = vecInternal.size();
        switch (size) {
            case 0: {
                return c.call((FunVal) onEmpty);
            }
            case 1: {
                return vecInternal.get(0);
            }
            default: {
                String msg = String.format(Locale.ROOT,
                        "Vec.just_or: required empty or singleton vec as \\recv, but size was %d",
                        size);
                return c.raise(msg);
            }
        }
    }

    // }}}1
    // Vec.with_just_or {{{1

    /**
     * Implementation of Vec.with_just_or.
     */
    private HostResult withJustOrMethod(HostContext c, VecVal vec, Val onJustVal, Val onEmptyVal) {
        String desc = "Vec.with_just_or($on_just $on_empty)";
        VecInternal vecInternal = vec.vecInternal();
        int size = vecInternal.size();
        if (size > 1) {
            return c.call(vm.graph.raiseFormat(
                        "{}: required 0 or 1 element for Vec, but it contains {} elements",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.of(vm.num.of(size))));
        }

        if (! (onJustVal instanceof FunVal)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: required fun for $on_just, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(onJustVal)));
        }
        if (! (onEmptyVal instanceof FunVal)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: required fun for $on_empty, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(onEmptyVal)));
        }

        return size == 0
            ? c.call((FunVal) onEmptyVal)
            : c.call((FunVal) onJustVal).args(vecInternal.get(0));
    }

    // }}}1
    // Vec.each {{{1

    /**
     * Implementation of Vec.each($on_elem).
     */
    private HostResult eachMethod(HostContext c, VecVal vec, Val onElem) {
        if (! (onElem instanceof FunVal)) {
            return c.call(vm.graph.raiseFormat(
                        "Vec.each($on_elem): $on_elem must be a fun, but got {}",
                        vm.graph.repr(onElem)));
        }
        VecInternal vecInternal = vec.vecInternal();
        int size = vecInternal.size();
        return size ==0
            ? vm.nada
            : eachLoop(c, vecInternal, 0, size, (FunVal) onElem);
    }

    /**
     * Iteration of Vec.each.
     */
    private HostResult eachLoop(
            HostContext c, VecInternal vecInternal, int ind, int size, FunVal onElem) {
        Val elem = vecInternal.get(ind);
        int nextInd = ind + 1;
        if (nextInd < size) {
            return c.call(onElem).args(elem)
                .on((cc, ignored) -> eachLoop(cc, vecInternal, nextInd, size, onElem));
        } else {
            return c.call(onElem).args(elem);
        }
    }

    // }}}1
    // Vec.map {{{1

    /**
     * Implementation of Vec.map.
     */
    private HostResult mapMethod(HostContext c, VecVal vec, Val trans) {
        if (! (trans instanceof FunVal)) {
            return c.call(vm.graph.raiseFormat("Vec.map($trans): $trans must be a fun, but got {}",
                        vm.graph.repr(trans)));
        }
        VecInternal src = vec.vecInternal();
        int size = src.size();
        VecInternal dest = VecInternal.ofCapa(vm, size);
        return mapLoop(c, src, dest, 0, (FunVal) trans);
    }

    /**
     * Iteration of Vec.map.
     */
    private HostResult mapLoop(
            HostContext c, VecInternal src, VecInternal dest, int index, FunVal trans) {
        if (index >= src.size()) {
            return new VecVal(vm, dest);
        }
        return c.call(trans).args(src.get(index)).on((cc, result) -> mapLoop(
                    cc, src, dest.append(result), index + 1, trans));
    }

    // }}}1
    // Vec.concat_map {{{1

    /**
     * Implementation of Vec.concat_map.
     */
    private HostResult concatMapMethod(HostContext c, VecVal vec, Val trans) {
        if (! (trans instanceof FunVal)) {
            return c.call(vm.graph.raiseFormat(
                        "Vec.concat_map($trans_to_vec): $trans_to_vec must be a fun, but got {}",
                        vm.graph.repr(trans)));
        }
        VecInternal src = vec.vecInternal();
        VecInternal dest = this.empty;
        return concatMapLoop(c, src, dest, 0, (FunVal) trans);
    }

    /**
     * Iteration of Vec.concat_map.
     */
    private HostResult concatMapLoop(
            HostContext c,
            VecInternal src, VecInternal dest,
            int srcIndex,
            FunVal trans) {
        if (srcIndex >= src.size()) {
            return new VecVal(vm, dest);
        }
        return c.call(trans).args(src.get(srcIndex)).on((cc, resultVal) -> {
            if (! (resultVal instanceof VecVal)) {
                return cc.call(vm.graph.raiseFormat(
                            "Vec.concat_map: expected vec for fun result, but got {}",
                            vm.graph.repr(resultVal)));
            }
            VecVal result = (VecVal) resultVal;
            VecInternal newDest = dest.insertAll(dest.size(), result.vecInternal());
            return concatMapLoop(cc, src, newDest, srcIndex + 1, trans);
        });
    }

    // }}}1
    // Vec.filter {{{1

    /**
     * Implementation of Vec.filter.
     */
    private HostResult filterMethod(HostContext c, VecVal vec, Val pred) {
        if (! (pred instanceof FunVal)) {
            return c.call(vm.graph.raiseFormat(
                        "Vec.filter($match?): $match? msut be a fun, but got {}",
                        vm.graph.repr(pred)));
        }
        VecInternal src = vec.vecInternal();
        VecInternal dest = this.empty;
        return filterLoop(c, src, dest, 0, (FunVal) pred);
    }

    /**
     * Iteration of Vec.filter.
     */
    private HostResult filterLoop(
            HostContext c, VecInternal src, VecInternal dest, int srcIndex, FunVal pred) {
        if (srcIndex >= src.size()) {
            return new VecVal(vm, dest);
        }
        Val elem = src.get(srcIndex);
        return c.call(pred).args(elem).on((cc, bool) -> {
            if (! vm.bool.isBool(bool)) {
                return cc.call(vm.graph.raiseFormat(
                            "Vec.filter: expected bool for fun result, but got {}",
                            vm.graph.repr(bool)));
            }
            VecInternal newDest = bool == vm.bool.trueVal
                ? dest.append(elem)
                : dest;
            return filterLoop(cc, src, newDest, srcIndex + 1, pred);
        });
    }

    // }}}1
    // Vec.count {{{1

    /**
     * Implementation of Vec.count.
     */
    private HostResult countMethod(HostContext c, VecVal vec, Val pred) {
        if (! (pred instanceof FunVal)) {
            return c.call(vm.graph.raiseFormat(
                        "Vec.count($match?): $match? must be a fun, but got {}",
                        vm.graph.repr(pred)));
        }
        VecInternal vecInternal = vec.vecInternal();
        return countLoop(c, vecInternal, 0, 0, (FunVal) pred);
    }

    /**
     * Iteration of Vec.count.
     */
    private HostResult countLoop(
            HostContext c, VecInternal vecInternal, int index, int num, FunVal pred) {
        if (index >= vecInternal.size()) {
            return vm.num.of(num);
        }
        return c.call(pred).args(vecInternal.get(index)).on((cc, bool) -> {
            if (! vm.bool.isBool(bool)) {
                return cc.call(vm.graph.raiseFormat(
                            "Vec.count: expected bool for fun result, but got {}",
                            vm.graph.repr(bool)));
            }
            int newNum = num + (bool == vm.bool.trueVal ? 1 : 0);
            return countLoop(cc, vecInternal, index + 1, newNum, pred);
        });
    }

    // }}}1
    // Vec.fold {{{1

    /**
     * Implementation of Vec.fold.
     */
    private HostResult foldMethod(HostContext c, VecVal vec, Val init, Val combine) {
        if (! (combine instanceof FunVal)) {
            return c.call(vm.graph.raiseFormat(
                        "Vec.fold(Init $combine): $combine must be a fun, but got {}",
                        vm.graph.repr(combine)));
        }
        VecInternal vecInternal = vec.vecInternal();
        return foldLoop(c, vecInternal, 0, init, (FunVal) combine);
    }

    /**
     * Iteration of Vec.fold.
     */
    private HostResult foldLoop(
            HostContext c,
            VecInternal vecInternal,
            int index,
            Val accum,
            FunVal combine) {
        int size = vecInternal.size();
        if (index >= size) {
            return accum;
        }

        Val elem = vecInternal.get(index);
        int nextIndex = index + 1;
        if (nextIndex < size) {
            return c.call(combine).args(accum, elem)
                .on((cc, newAccum) -> foldLoop(cc, vecInternal, nextIndex, newAccum, combine));
        } else {
            return c.call(combine).args(accum, elem);
        }
    }

    // }}}1
    // Vec.reduce {{{1

    /**
     * Implementation of Vec.reduce.
     */
    private HostResult reduceMethod(HostContext c, VecVal vec, Val combine) {
        if (! (combine instanceof FunVal)) {
            return c.call(vm.graph.raiseFormat(
                        "Vec.reduce($combine): $combine must be a fun, but got {}",
                        vm.graph.repr(combine)));
        }
        VecInternal vecInternal = vec.vecInternal();
        int size = vecInternal.size();
        if (size == 0) {
            return vm.vec.of();
        } else {
            Val first = vecInternal.get(0);
            return reduceLoop(c, vecInternal, 1, first, (FunVal) combine);
        }
    }

    /**
     * Iteration of Vec.reduce.
     */
    private HostResult reduceLoop(
            HostContext c,
            VecInternal vecInternal,
            int index,
            Val accum,
            FunVal combine) {
        if (index >= vecInternal.size()) {
            return vm.vec.of(accum);
        }
        return c.call(combine).args(accum, vecInternal.get(index)).on((cc, newAccum) -> {
            return reduceLoop(cc, vecInternal, index + 1, newAccum, combine);
        });
    }

    // }}}1
    // Vec.scan, Vec.scan_inside {{{1

    /**
     * Implementation of Vec.scan.
     */
    private HostResult scanMethod(HostContext c, VecVal vec, Val init, Val combine) {
        if (! (combine instanceof FunVal)) {
            return c.call(vm.graph.raiseFormat(
                        "Vec.scan(Init $combine): $combine must be a fun, but got {}",
                        vm.graph.repr(combine)));
        }
        VecInternal src = vec.vecInternal();
        int srcSize = src.size();
        VecInternal dest = VecInternal.ofCapa(vm, srcSize + 1);
        dest = dest.insert(0, init);
        return scanLoop(c, src, dest, 0, init, (FunVal) combine);
    }

    /**
     * Implementation of Vec.scan_inside.
     */
    private HostResult scanInsideMethod(HostContext c, VecVal vec, Val combine) {
        if (! (combine instanceof FunVal)) {
            return c.call(vm.graph.raiseFormat(
                        "Vec.scan_inside($combine): $combine must be a fun, but got {}",
                        vm.graph.repr(combine)));
        }
        VecInternal src = vec.vecInternal();
        int srcSize = src.size();
        if (srcSize == 0) {
            return vm.vec.of();
        }
        VecInternal dest = VecInternal.ofCapa(vm, srcSize);
        Val first = src.get(0);
        dest = dest.insert(0, first);
        return scanLoop(c, src, dest, 1, first, (FunVal) combine);
    }

    /**
     * Iteration of Vec.scan.
     */
    private HostResult scanLoop(
            HostContext c,
            VecInternal src, VecInternal dest,
            int srcIndex, Val accum, FunVal combine) {
        if (srcIndex >= src.size()) {
            return new VecVal(vm, dest);
        }
        return c.call(combine).args(accum, src.get(srcIndex)).on((cc, newAccum) -> {
            VecInternal newDest = dest.append(newAccum);
            return scanLoop(cc, src, newDest, srcIndex + 1, newAccum, combine);
        });
    }

    // }}}1
    // Vec.take_while {{{1

    /**
     * Implementation of Vec.take_while.
     */
    private HostResult takeWhileMethod(HostContext c, VecVal vec, Val pred) {
        if (! (pred instanceof FunVal)) {
            return c.call(vm.graph.raiseFormat(
                        "Vec.take_while($match?): $match? must be a fun, but got {}",
                        vm.graph.repr(pred)));
        }
        VecInternal vecInternal = vec.vecInternal();
        return takeWhileLoop(c, vecInternal, 0, (FunVal) pred);
    }

    /**
     * Iteration of Vec.take_while.
     */
    private HostResult takeWhileLoop(HostContext c, VecInternal src, int srcIndex, FunVal pred) {
        int srcSize = src.size();
        if (srcIndex >= srcSize) {
            return new VecVal(vm, src.copy());
        }

        Val elem = src.get(srcIndex);
        return c.call(pred).args(elem).on((cc, result) -> {
            if (! vm.bool.isBool(result)) {
                return cc.call(vm.graph.raiseFormat(
                            "Vec.take_while($match?):"
                            + " expected bool for the result of $match?, but got {}",
                            vm.graph.repr(result)));
            }
            if (result == vm.bool.falseVal) {
                return new VecVal(vm, src.copyRange(0, srcIndex));
            }
            return takeWhileLoop(cc, src, srcIndex + 1, pred);
        });
    }

    // }}}1
    // Vec.drop_while {{{1

    /**
     * Implementation of Vec.drop_while.
     */
    private HostResult dropWhileMethod(HostContext c, VecVal vec, Val pred) {
        if (! (pred instanceof FunVal)) {
            return c.call(vm.graph.raiseFormat(
                        "Vec.drop_while($match?): $match? must be a fun, but got {}",
                        vm.graph.repr(pred)));
        }
        VecInternal vecInternal = vec.vecInternal();
        return dropWhileLoop(c, vecInternal, 0, vecInternal.size(), (FunVal) pred);
    }

    /**
     * Iteration of Vec.drop_while.
     */
    private HostResult dropWhileLoop(
            HostContext c, VecInternal src, int srcIndex, int srcSize, FunVal pred) {
        if (srcIndex >= srcSize) {
            return vm.vec.of();
        }

        Val elem = src.get(srcIndex);
        return c.call(pred).args(elem).on((cc, result) -> {
            if (! vm.bool.isBool(result)) {
                return cc.call(vm.graph.raiseFormat(
                            "Vec.drop_while($match?):"
                            + " expected bool for the result of $match?, but got {}",
                            vm.graph.repr(result)));
            }
            if (result == vm.bool.falseVal) {
                return new VecVal(vm, src.copyRange(srcIndex, srcSize));
            }
            return dropWhileLoop(cc, src, srcIndex + 1, srcSize, pred);
        });
    }

    // }}}1

    // Vec.all? {{{1

    /**
     * Implementation of Vec.all?
     */
    private HostResult allPMethod(HostContext c, VecVal vec, Val pred) {
        if (! (pred instanceof FunVal)) {
            return c.call(vm.graph.raiseFormat(
                        "Vec.all?($match?): $match? must be a fun, but got {}",
                        vm.graph.repr(pred)));
        }
        VecInternal vecInternal = vec.vecInternal();
        return allPLoop(c, vecInternal, 0, (FunVal) pred);
    }

    /**
     * Iteration of Vec.all?.
     */
    private HostResult allPLoop(HostContext c, VecInternal vecInternal, int index, FunVal pred) {
        int size = vecInternal.size();
        if (index >= size) {
            return vm.bool.trueVal;
        }

        Val elem = vecInternal.get(index);
        int nextIndex = index + 1;
        if (nextIndex < size) {
            return c.call(pred).args(elem).on((cc, result) -> {
                if (! vm.bool.isBool(result)) {
                    return cc.call(vm.graph.raiseFormat(
                                "Vec.all?($match?):"
                                + " expected bool for the result of $match?, but got {}",
                                vm.graph.repr(result)));
                }
                if (result == vm.bool.falseVal) {
                    return vm.bool.falseVal;
                }
                return allPLoop(cc, vecInternal, nextIndex, pred);
            });
        } else {
            return c.call(pred).args(elem);
        }
    }

    // }}}1

    // Vec.any? {{{1

    /**
     * Implementation of Vec.any?.
     */
    private HostResult anyPMethod(HostContext c, VecVal vec, Val pred) {
        if (! (pred instanceof FunVal)) {
            return c.call(vm.graph.raiseFormat(
                        "Vec.any?($match?): $match? must be a fun, but got {}",
                        vm.graph.repr(pred)));
        }
        VecInternal vecInternal = vec.vecInternal();
        return forAnyPLoop(c, vecInternal, 0, (FunVal) pred);
    }

    /**
     * Iteration of Vec.any?.
     */
    private HostResult forAnyPLoop(HostContext c, VecInternal vecInternal, int index, FunVal pred) {
        int size = vecInternal.size();
        if (index >= size) {
            return vm.bool.falseVal;
        }

        Val elem = vecInternal.get(index);
        int nextIndex = index + 1;
        if (nextIndex < size) {
            return c.call(pred).args(elem).on((cc, result) -> {
                if (! vm.bool.isBool(result)) {
                    return cc.call(vm.graph.raiseFormat(
                                "Vec.any?($match?):"
                                + " expected bool for the result of $match?, but got {}",
                                vm.graph.repr(result)));
                }
                if (result == vm.bool.trueVal) {
                    return vm.bool.trueVal;
                }
                return forAnyPLoop(cc, vecInternal, nextIndex, pred);
            });
        } else {
            return c.call(pred).args(elem);
        }
    }

    // }}}1

    // Vec.have? {{{1

    /**
     * Implementation of Vec.have?.
     */
    private HostResult havePMethod(HostContext c, VecVal vec, Val target) {
        VecInternal vecInternal = vec.vecInternal();
        return havePLoop(c, vecInternal, 0, target);
    }

    /**
     * Iteration of Vec.have?.
     */
    private HostResult havePLoop(HostContext c, VecInternal vecInternal, int index, Val target) {
        int size = vecInternal.size();
        if (index >= size) {
            return vm.bool.falseVal;
        }

        Val elem = vecInternal.get(index);
        int nextIndex = index + 1;
        if (nextIndex < size) {
            return c.call(elem, this.opEqHandle).args(target).on((cc, result) -> {
                if (! vm.bool.isBool(result)) {
                    return cc.call(vm.graph.raiseFormat(
                                "Vec.have?:"
                                + " expected bool for result of .op_eq of elem #{}, but got {}",
                                vm.graph.of(vm.num.of(index)),
                                vm.graph.repr(result)));
                }
                if (result == vm.bool.trueVal) {
                    return vm.bool.trueVal;
                }
                return havePLoop(cc, vecInternal, nextIndex, target);
            });
        } else {
            return c.call(elem, this.opEqHandle).args(target);
        }
    }

    // }}}1
    // Vec.have_all? {{{1

    /**
     * Implementation of Vec.have_all?.
     */
    private HostResult haveAllPMethod(HostContext c, VecVal vec, Val arg) {
        return c.call(arg, allP).args(
                vm.fun.make("Vec.have_all?#accept_elem?")
                .take(1).action(cc -> cc.call(vec, haveP).args(cc.arg(0))));
    }

    // }}}1
    // Vec.have_any? {{{1

    /**
     * Implementation of Vec.have_any?.
     */
    private HostResult haveAnyPMethod(HostContext c, VecVal vec, Val arg) {
        return c.call(arg, anyP).args(
                vm.fun.make("Vec.have_any?#accept_elem?")
                .take(1).action(cc -> cc.call(vec, haveP).args(cc.arg(0))));
    }

    // }}}1
    // Vec.search(Min_ind $accept?) {{{1

    /**
     * Implementation of Vec.search.
     */
    private HostResult searchMethod(HostContext c, VecVal vec, Val minIndVal, Val acceptsVal) {
        VecInternal vecInternal = vec.vecInternal();
        int size = vecInternal.size();
        int minInd = NumOperations.getPosIndex(minIndVal, size);
        if (minInd < 0) {
            return c.call(vm.graph.raiseFormat(
                        "Vec.search(Min_ind $accept?):"
                        + " required int num in [0, {}] for Min_ind but got {}",
                        vm.graph.of(vm.num.of(size)),
                        vm.graph.repr(minIndVal)));
        }
        if (! (acceptsVal instanceof FunVal)) {
            return c.call(vm.graph.raiseFormat(
                        "Vec.search(Min_ind $accept?): required fun for $accept?, but got {}",
                        vm.graph.repr(acceptsVal)));
        }
        FunVal accepts = (FunVal) acceptsVal;
        return searchMethodLoop(c, vecInternal, minInd, accepts, size);
    }

    /**
     * Loop of Vec.search.
     */
    private HostResult searchMethodLoop(
            HostContext c, VecInternal vecInternal, int ind, FunVal accepts, int size) {
        if (ind >= size) {
            return vm.vec.of();
        }

        Val elem = vecInternal.get(ind);
        return c.call(accepts).args(elem).on((cc, r) -> {
            if (! vm.bool.isBool(r)) {
                return cc.call(vm.graph.raiseFormat(
                            "Vec.search(Min_ind $accept?):"
                            + " expected bool returned from $accept?, but got {}",
                            vm.graph.repr(r)));
            }
            if (r.equals(vm.bool.trueVal)) {
                return vm.vec.of(vm.num.of(ind));
            }
            return searchMethodLoop(cc, vecInternal, ind + 1, accepts, size);
        });
    }

    // }}}1
    // Vec.sort($precede?) {{{1

    /**
     * Implementation of Vec.sort.
     */
    private HostResult sortMethod(CallContext c) {
        Val precedes = c.argCount() == 0
            ? this.defaultPrecedesFun
            : c.arg(0);
        return c.call("kink/_vec/SORT_VEC", sortHandle).args(c.recv(), precedes);
    }

    // }}}1
    // Vec.iter and .iter_from {{{1

    /**
     * Implementation of Vec.iter.
     */
    private HostResult iterMethod(HostContext c, VecVal vec) {
        VecInternal vecInternal = vec.vecInternal();
        int size = vecInternal.size();
        FunVal ifun = forwardIfun("Vec.iter.ifun", vecInternal, 0, size);
        return c.call("kink/iter/ITER", vm.sym.handleFor("new")).args(ifun);
    }

    /**
     * Implementation of Vec.iter_from.
     */
    private HostResult iterFromMethod(HostContext c, VecVal vec, Val minIndVal) {
        VecInternal vecInternal = vec.vecInternal();
        int size = vecInternal.size();
        int minInd = NumOperations.getPosIndex(minIndVal, size);
        if (minInd < 0) {
            return c.call(vm.graph.raiseFormat(
                        "Vec.iter_from(Min_ind): required int in [0, {}] for Min_ind, but got {}",
                        vm.graph.of(vm.num.of(size)),
                        vm.graph.repr(minIndVal)));
        }
        FunVal ifun = forwardIfun("Vec.iter_from.ifun", vecInternal, minInd, size);
        return c.call("kink/iter/ITER", vm.sym.handleFor("new")).args(ifun);
    }

    /**
     * Ifun for Vec.iter.
     */
    private FunVal forwardIfun(String prefix, VecInternal vecInternal, int ind, int size) {
        return vm.fun.make(prefix).take(2).action(c -> {
            Val proc = c.arg(0);
            if (! (proc instanceof FunVal)) {
                return c.call(vm.graph.raiseFormat("{}: required fun for $proc, but got {}",
                            vm.graph.of(vm.str.of(prefix)),
                            vm.graph.repr(proc)));
            }

            Val fin = c.arg(1);
            if (! (fin instanceof FunVal)) {
                return c.call(vm.graph.raiseFormat("{}: required fun for $fin, but got {}",
                            vm.graph.of(vm.str.of(prefix)),
                            vm.graph.repr(fin)));
            }
            if (ind < size) {
                Val head = vecInternal.get(ind);
                FunVal tail = forwardIfun(prefix, vecInternal, ind + 1, size);
                return c.call((FunVal) proc).args(head, tail);
            } else {
                return c.call((FunVal) fin);
            }
        });
    }

    // }}}1
    // Vec.op_add {{{1

    /**
     * Implementation of Vec.op_add.
     */
    private HostResult opAddMethod(HostContext c, VecVal left, VecVal right) {
        VecInternal result = left.vecInternal().copy();
        int leftSize = result.size();
        result = result.insertAll(leftSize, right.vecInternal());
        return new VecVal(vm, result);
    }

    // }}}1
    // Vec.op_store {{{1

    /**
     * Implementation of Vec.op_store.
     */
    private HostResult opStoreMethod(HostContext c, VecVal lhs, Val arg) {
        if (! (arg instanceof VecVal rhs)) {
            return c.call(FunsHolder.raiseNotVecRhs(vm, arg));
        }

        VecInternal lhsVi = lhs.vecInternal();
        VecInternal rhsVi = rhs.vecInternal();
        int lhsSize = lhsVi.size();
        int rhsSize = rhsVi.size();

        int mandatoryCount = getMandatoryParamsCount(lhsSize, lhsVi);
        int optCount = getOptParamsCount(lhsSize, lhsVi, mandatoryCount);

        int restInd = mandatoryCount + optCount;
        @Nullable Val restParam = restInd == lhsSize ? null : lhsVi.get(restInd);
        if (restInd < lhsSize - 1 || restInd == lhsSize - 1 && ! restParam.hasVar(passRestHandle)) {
            return c.call(vm.graph.raiseFormat("Vec.op_store(Rhs): unexpected lhs param: {}",
                        vm.graph.repr(restParam)));
        }

        if (rhsSize < mandatoryCount || restParam == null && rhsSize > mandatoryCount + optCount) {
            FunVal fun = FunsHolder.wrongNumberOfArgs(vm, mandatoryCount, vm.graph.repr(lhs), rhs);
            return c.call(fun);
        }

        return opStoreMandatoryLoop(c, 0,
                lhsVi,
                rhsSize, rhsVi,
                mandatoryCount, optCount, restParam);
    }

    /**
     * Returns the count of mandatory params.
     */
    private int getMandatoryParamsCount(int lhsSize, VecInternal lhsVi) {
        for (int i = 0; i < lhsSize; ++ i) {
            Val param = lhsVi.get(i);
            if (! (param instanceof VarrefVal || param.hasVar(storeHandle))) {
                return i;
            }
        }
        return lhsSize;
    }

    /**
     * Returns the count of opt params.
     */
    private int getOptParamsCount(int lhsSize, VecInternal lhsVi, int offset) {
        for (int i = 0; offset + i < lhsSize; ++ i) {
            Val param = lhsVi.get(offset + i);
            if (! param.hasVar(passArgHandle)) {
                return i;
            }
        }
        return lhsSize - offset;
    }

    /**
     * Loop for mandatory args.
     */
    private HostResult opStoreMandatoryLoop(
            HostContext c,
            int ind,
            VecInternal lhs,
            int rhsSize,
            VecInternal rhs,
            int mandatoryCount, int optCount, @Nullable Val restParam) {
        while (ind < mandatoryCount) {
            Val param = lhs.get(ind);
            Val rval = rhs.get(ind);

            if (! (param instanceof VarrefVal)) {
                int nextInd = ind + 1;
                return c.call(param, storeHandle).args(rval)
                    .on((cc, r) -> opStoreMandatoryLoop(
                                cc, nextInd, lhs, rhsSize, rhs,
                                mandatoryCount, optCount, restParam));
            }

            VarrefVal varref = (VarrefVal) param;
            Val owner = varref.owner();
            int handle = varref.symHandle();
            owner.setVar(handle, rval);
            ++ ind;
        }
        return opStoreOptLoop(
                c, ind, lhs, rhsSize, rhs, mandatoryCount, optCount, restParam);
    }


    /**
     * Loop for optional args.
     */
    private HostResult opStoreOptLoop(
            HostContext c, int ind, VecInternal lhs, int rhsSize, VecInternal rhs,
            int mandatoryCount, int optCount, @Nullable Val restParam) {
        if (ind == mandatoryCount + optCount) {
            return opStoreRest(c, ind, rhsSize, rhs, restParam);
        }

        Val param = lhs.get(ind);
        HostFunReaction next = (cc, r) -> opStoreOptLoop(
                cc, ind + 1, lhs, rhsSize, rhs, mandatoryCount, optCount, restParam);
        return (ind < rhsSize
                ? c.call(param, passArgHandle).args(rhs.get(ind))
                : c.call(param, passNothingHandle)
                ).on(next);
    }

    /**
     * Handles the rest args.
     */
    private HostResult opStoreRest(
            HostContext c, int ind, int rhsSize, VecInternal rhs, @Nullable Val restParam) {
        if (restParam == null) {
            return vm.nada;
        }
        VecInternal restArgs = rhs.copyRange(Math.min(ind, rhsSize), rhsSize);
        VecVal restArgsVec = new VecVal(vm, restArgs);
        return c.call(restParam, passRestHandle).args(restArgsVec);
    }

    // }}}1
    // Vec.op_mul(Count) {{{1

    /**
     * Implementation of Vec.op_mul(Count).
     */
    private HostResult opMulMethod(HostContext c, VecVal vec, Val countVal) {
        VecInternal vecInternal = vec.vecInternal();
        BigInteger count = NumOperations.getExactBigInteger(countVal);
        if (count == null || count.signum() < 0) {
            return c.call(vm.graph.raiseFormat(
                        "Vec.op_mul(Count): required int num >=0 for Count, but got {}",
                        vm.graph.repr(countVal)));
        }

        int size = vecInternal.size();
        if (size == 0) {
            return vm.vec.of();
        }

        // Size of vecInternal may be changed concurrently,
        // thus the resultSizeEstimate is only an estimate.
        BigInteger resultSizeEstimate = count.multiply(BigInteger.valueOf(size));
        if (resultSizeEstimate.compareTo(BigInteger.valueOf(Integer.MAX_VALUE)) > 0) {
            return c.call(vm.graph.raiseFormat(
                        "Vec.op_mul(Count): too long result: size {} * Count {} is {}",
                        vm.graph.of(vm.num.of(size)),
                        vm.graph.of(countVal),
                        vm.graph.of(vm.num.of(resultSizeEstimate))));
        }

        int countInt = count.intValueExact();
        VecInternal dest = VecInternal.ofCapa(vm, size * countInt);
        for (int i = 0; i < countInt; ++ i) {
            dest = dest.insertAll(dest.size(), vecInternal);
        }
        return new VecVal(vm, dest);
    }

    // }}}1
    // Vec == Another_vec and Vec != Another_vec {{{1

    /**
     * Implementation of Vec.op_eq.
     */
    private HostResult opEqMethod(HostContext c, VecVal left, VecVal right) {
        VecInternal lv = left.vecInternal();
        VecInternal rv = right.vecInternal();
        int size = lv.size();
        if (rv.size() != size) {
            return vm.bool.falseVal;
        }
        if (size == 0) {
            return vm.bool.trueVal;
        }
        return eqLoop(c, lv, rv, 0, size);
    }

    /**
     * Loop of Vec.op_eq.
     */
    private HostResult eqLoop(
            HostContext c, VecInternal lv, VecInternal rv, int ind, int size) {
        Val lelem = lv.get(ind);
        Val relem = rv.get(ind);
        if (ind + 1 >= size) {
            return c.call(lelem, opEqHandle).args(relem);
        }
        return c.call(lelem, opEqHandle).args(relem).on((cc, r) -> {
            if (! vm.bool.isBool(r)) {
                return cc.call(vm.graph.raiseFormat(
                            "Vec1.op_eq(Vec2): expected bool result from Elem.op_eq, but got {}",
                            vm.graph.repr(r)));
            }
            if (r == vm.bool.falseVal) {
                return vm.bool.of(false);
            }
            return eqLoop(cc, lv, rv, ind + 1, size);
        });
    }

    // }}}1
    // Vec < Another_vec Vec <= Another_vec Vec >= Another_vec Vec > Another_vec {{{1

    /**
     * Implementation of Vec.op_lt.
     */
    private HostResult opLtMethod(HostContext c, VecVal left, VecVal right) {
        VecInternal lv = left.vecInternal();
        VecInternal rv = right.vecInternal();
        int lsize = lv.size();
        int rsize = rv.size();
        return lsize == 0 ? vm.bool.of(rsize != 0)
            : rsize == 0 ? vm.bool.falseVal
            : lsize == rsize ? compareSameSize(c, lv, rv, 0, lsize)
            : compareDifferentSize(c, lv, rv, 0, Math.min(lsize, rsize), lsize < rsize);
    }

    /**
     * Loop of op_lt when the size of the two vecs are different.
     */
    private HostResult compareDifferentSize(
            HostContext c,
            VecInternal lv, VecInternal rv,
            int ind,
            int size,
            boolean atEnd) {
        if (ind == size) {
            return vm.bool.of(atEnd);
        }
        Val lelem = lv.get(ind);
        Val relem = rv.get(ind);
        return c.call(lelem, opLtHandle).args(relem).on((cc, accept) -> {
            if (! vm.bool.isBool(accept)) {
                return cc.call(vm.graph.raiseFormat(
                            "Vec1.op_lt(Vec2): expected bool result from .op_lt, but got {}",
                            vm.graph.repr(accept)));
            }
            if (accept.equals(vm.bool.trueVal)) {
                return vm.bool.trueVal;
            }
            return cc.call(relem, opLtHandle).args(lelem).on((c3, reject) -> {
                if (! vm.bool.isBool(reject)) {
                    return c3.call(vm.graph.raiseFormat(
                                "Vec1.op_lt(Vec2): expected bool result from .op_lt, but got {}",
                                vm.graph.repr(reject)));
                }
                if (reject.equals(vm.bool.trueVal)) {
                    return vm.bool.falseVal;
                }
                return compareDifferentSize(c3, lv, rv, ind + 1, size, atEnd);
            });
        });
    }

    /**
     * Loop of op_lt when the size of the two vecs are the same.
     */
    private HostResult compareSameSize(
            HostContext c,
            VecInternal lv, VecInternal rv,
            int ind,
            int size) {
        Val lelem = lv.get(ind);
        Val relem = rv.get(ind);
        if (ind + 1 == size) {
            return c.call(lelem, opLtHandle).args(relem);
        }
        return c.call(lelem, opLtHandle).args(relem).on((cc, accept) -> {
            if (! vm.bool.isBool(accept)) {
                return cc.call(vm.graph.raiseFormat(
                            "Vec1.op_lt(Vec2): expected bool result from .op_lt, but got {}",
                            vm.graph.repr(accept)));
            }
            if (accept.equals(vm.bool.trueVal)) {
                return vm.bool.trueVal;
            }
            return cc.call(relem, opLtHandle).args(lelem).on((c3, reject) -> {
                if (! vm.bool.isBool(reject)) {
                    return c3.call(vm.graph.raiseFormat(
                                "Vec1.op_lt(Vec2): expected bool result from .op_lt, but got {}",
                                vm.graph.repr(reject)));
                }
                if (reject.equals(vm.bool.trueVal)) {
                    return vm.bool.falseVal;
                }
                return compareSameSize(c3, lv, rv, ind + 1, size);
            });
        });
    }

    // }}}1
    // Vec.repr {{{1

    /**
     * Implementation of Vec.repr.
     */
    private HostResult reprMethod(HostContext c, VecVal vec) {
        return c.call("kink/_vec/REPR_VEC", reprVecHandle).args(vec);
    }

    // }}}1

    /**
     * Checks that the pair of fromVal and toVal is a valid range
     * in the sequence with {@code size},
     * and returns a node which yields an error message, or null.
     */
    @Nullable
    private GraphNode checkRange(String prefix, int size, Val fromVal, Val toVal) {
        if (! (fromVal instanceof NumVal)) {
            return vm.graph.raiseFormat("{}: From_pos must be a num, but got {}",
                    vm.graph.of(vm.str.of(prefix)),
                    vm.graph.repr(fromVal));
        }
        BigDecimal fromDecimal = ((NumVal) fromVal).bigDecimal();

        if (! (toVal instanceof NumVal)) {
            return vm.graph.raiseFormat("{}: To_pos must be a num, but got {}",
                    vm.graph.of(vm.str.of(prefix)),
                    vm.graph.repr(toVal));
        }
        BigDecimal toDecimal = ((NumVal) toVal).bigDecimal();

        if (! NumOperations.isRangePair(fromDecimal, toDecimal, size)) {
            return vm.graph.raiseFormat("{}: required range pair in [0, {}], but got {} and {}",
                    vm.graph.of(vm.str.of(prefix)),
                    vm.graph.of(vm.num.of(size)),
                    vm.graph.repr(fromVal),
                    vm.graph.repr(toVal));
        }

        return null;
    }

    /**
     * Makes a nullary method fun.
     */
    private FunVal method0(
            String prefix, ThrowingFunction2<CallContext, VecVal, HostResult> action) {
        return vm.fun.make(prefix).take(0).action(c -> {
            Val recv = c.recv();
            if (! (recv instanceof VecVal)) {
                return c.call(vm.graph.raiseFormat("{}: required vec for \\recv, but got {}",
                            vm.graph.of(vm.str.of(prefix)),
                            vm.graph.repr(recv)));
            }
            return action.apply(c, (VecVal) recv);
        });
    }

    /**
     * Makes an unary method fun.
     */
    private FunVal method1(
            String prefix, ThrowingFunction3<CallContext, VecVal, Val, HostResult> action) {
        return vm.fun.make(prefix).take(1).action(c -> {
            Val recv = c.recv();
            if (! (recv instanceof VecVal)) {
                return c.call(vm.graph.raiseFormat("{}: required vec for \\recv, but got {}",
                            vm.graph.of(vm.str.of(prefix)),
                            vm.graph.repr(recv)));
            }
            return action.apply(c, (VecVal) recv, c.arg(0));
        });
    }

    /**
     * Makes a binary method fun.
     */
    private FunVal method2(
            String prefix,
            ThrowingFunction4<CallContext, VecVal, Val, Val, HostResult> action) {
        return vm.fun.make(prefix).take(2).action(c -> {
            Val recv = c.recv();
            if (! (recv instanceof VecVal)) {
                return c.call(vm.graph.raiseFormat("{}: required vec for \\recv, but got {}",
                            vm.graph.of(vm.str.of(prefix)),
                            vm.graph.repr(recv)));
            }
            return action.apply(c, (VecVal) recv, c.arg(0), c.arg(1));
        });
    }

    /**
     * Makes an binary operator method fun.
     */
    private FunVal binOp(
            String prefix, ThrowingFunction3<CallContext, VecVal, VecVal, HostResult> action) {
        return method1(prefix, (c, vec, arg) -> {
            if (! (arg instanceof VecVal)) {
                return c.call(vm.graph.raiseFormat("{}: the arg must be a vec, but got {}",
                            vm.graph.of(vm.str.of(prefix)),
                            vm.graph.repr(arg)));
            }
            return action.apply(c, vec, (VecVal) arg);
        });
    }

    // factory methods {{{1

    /**
     * Returns a vec containing the elems.
     *
     * @param elems the elems of the vec.
     * @return a vec.
     */
    public VecVal of(List<? extends Val> elems) {
        return elems.isEmpty() ? of() : new VecVal(vm, VecInternal.of(vm, elems));
    }

    /**
     * Returns a new vec from the range of the array.
     *
     * @param vals the array of vals.
     * @param from the from index of the range (inclusive).
     * @param to the to index of the range (exclusive).
     * @return a new vec.
     */
    public VecVal of(Val[] vals, int from, int to) {
        Preconds.checkRange(from, to, vals.length);
        return from == to ? of() : new VecVal(vm, VecInternal.of(vm, vals, from, to));
    }

    /**
     * Returns a new empty vec.
     *
     * @return a new empty vec.
     */
    public VecVal of() {
        return new VecVal(vm, this.empty);
    }

    /**
     * Returns a new vec.
     *
     * @param e0 the #0 elem.
     * @return a new vec.
     */
    public VecVal of(Val e0) {
        return new VecVal(vm, VecInternal.of(vm, e0));
    }

    /**
     * Returns a new vec.
     *
     * @param e0 the #0 elem.
     * @param e1 the #1 elem.
     * @return a new vec.
     */
    public VecVal of(Val e0, Val e1) {
        return new VecVal(vm, VecInternal.of(vm, e0, e1));
    }

    /**
     * Returns a new vec.
     *
     * @param e0 the #0 elem.
     * @param e1 the #1 elem.
     * @param e2 the #2 elem.
     * @return a new vec.
     */
    public VecVal of(Val e0, Val e1, Val e2) {
        return new VecVal(vm, VecInternal.of(vm, e0, e1, e2));
    }

    /**
     * Returns a new vec.
     *
     * @param e0 the #0 elem.
     * @param e1 the #1 elem.
     * @param e2 the #2 elem.
     * @param e3 the #3 elem.
     * @return a new vec.
     */
    public VecVal of(Val e0, Val e1, Val e2, Val e3) {
        return new VecVal(vm, VecInternal.of(vm, e0, e1, e2, e3));
    }

    /**
     * Returns a new vec.
     *
     * @param elems the elems.
     * @return a new vec.
     */
    public VecVal of(Val... elems) {
        return new VecVal(vm, VecInternal.of(vm, elems, 0, elems.length));
    }

    // }}}1

}

// vim: et sw=4 sts=4 fdm=marker
