package org.kink_lang.kink;

import java.util.Locale;
import java.util.Map;
import java.util.function.Function;

import org.kink_lang.kink.hostfun.HostFunAction;
import org.kink_lang.kink.hostfun.graph.GraphFacade;
import org.kink_lang.kink.hostfun.graph.impl.GraphFacadeImpl;
import org.kink_lang.kink.internal.compile.tempval.IntVal;
import org.kink_lang.kink.internal.mod.java.JavaMod;
import org.kink_lang.kink.internal.mod.java.JavaProxyMod;
import org.kink_lang.kink.internal.mod.java.ThrowerMod;
import org.kink_lang.kink.internal.mod.stopwatch.StopwatchMod;
import org.kink_lang.kink.internal.ovis.OwnVarIndexes;
import org.kink_lang.kink.internal.sym.SymRegistryImpl;

/**
 * A Kink vm.
 */
public class Vm {

    /** The nada val. */
    public final Val nada;

    /** The helper for bool vals. */
    public final BoolHelper bool;

    /** The helper for env vals. */
    public final BindingHelper binding;

    /** The helper for num vals. */
    public final NumHelper num;

    /** The helper for str vals. */
    public final StrHelper str;

    /** The helper for bin vals. */
    public final BinHelper bin;

    /** The helper of funs. */
    public final FunHelper fun;

    /** The helper for varref vals. */
    public final VarrefHelper varref;

    /** The helper for kontTag vals. */
    final KontTagHelper kontTag;

    /** The helper for exception vals. */
    public final ExceptionHelper exception;

    /** The helper for vec vals. */
    public final VecHelper vec;

    /** The helper for trace vals. */
    public final TraceHelper trace;

    /** The helper for location vals. */
    public final LocationHelper location;

    /** The helper for java vals. */
    public final JavaHelper java;

    /** The registry of syms. */
    public final SymRegistry sym;

    /** The registry of modules. */
    final ModRegistry mod;

    /** The facade for the Execution Graph DSL. */
    public final GraphFacade graph;

    /** The holder of int vals. */
    final IntVal.Holder ints;

    /** The factory of SharedVars.*/
    public final SharedVarsFactory sharedVars;

    /** The component registry. */
    public final ComponentRegistry component;

    /** The empty ovis. */
    final OwnVarIndexes emptyOvis = OwnVarIndexes.buildEmpty();

    /** Kont tag for try/break/finally. */
    final KontTagVal escapeKontTag;

    /** Default implementation of "repr" method. */
    final FunVal defaultRepr;

    /** Vars for nada val. */
    private final SharedVars nadaVars;

    /**
     * Constructs a vm.
     */
    private Vm() {
        this.nada = new Val(this, null) {
            @Override public String toString() { return "nada"; }
            @Override
            SharedVars sharedVars() {
                return Vm.this.nadaVars;
            }
        };
        this.bool = new BoolHelper(this);
        this.binding = new BindingHelper(this);
        this.num = new NumHelper(this);
        this.str = new StrHelper(this);
        this.bin = new BinHelper(this);
        this.fun = new FunHelper(this);
        this.varref = new VarrefHelper(this);
        this.vec = new VecHelper(this);
        this.kontTag = new KontTagHelper(this);
        this.exception = new ExceptionHelper(this);
        this.trace = new TraceHelper(this);
        this.location = new LocationHelper(this);
        this.java = new JavaHelper(this);
        this.sym = new SymRegistryImpl();
        this.mod = new ModRegistry(this);
        this.graph = new GraphFacadeImpl(this);
        this.ints = new IntVal.Holder(this);
        this.component = new ComponentRegistry(this);

        this.defaultRepr = this.fun.make("Val.repr")
            .take(0).action(c -> this.str.of(String.format(Locale.ROOT,
                            "(val val_id=%d)", c.recv().identity())));
        // vm.sharedVars must be initialized after vm.defaultRepr because it depends on defaultRepr
        this.sharedVars = new SharedVarsFactory(this);

        this.escapeKontTag = new KontTagVal(this);
        this.nadaVars = this.sharedVars.of(Map.of(
                    this.sym.handleFor("repr"), this.fun.constant(this.str.of("nada"))));

        this.bool.init();
        this.binding.init();
        this.num.init();
        this.str.init();
        this.bin.init();
        this.fun.init();
        this.varref.init();
        this.kontTag.init();
        this.exception.init();
        this.vec.init();
        this.trace.init();
        this.location.init();
        this.java.init();
        ((GraphFacadeImpl) this.graph).init();
        this.ints.init();

        this.component.register(IntVal.Holder.class, this.ints);

        this.mod.register("kink/KONT_TAG", () -> new KontTagMod(this).makeMod());
        this.mod.register("kink/_INTERNAL", () -> new InternalMod(this).makeMod());
        this.mod.register(JavaMod.MOD_NAME, () -> JavaMod.makeMod(this));
        this.mod.register(JavaProxyMod.MOD_NAME, () -> JavaProxyMod.makeMod(this));
        this.mod.register(ThrowerMod.MOD_NAME, () -> ThrowerMod.makeMod(this));
        this.mod.register(StopwatchMod.MOD_NAME, () -> StopwatchMod.makeMod(this));
    }

    /**
     * Returns a new vm.
     *
     * @return a new vm.
     */
    public static Vm newVm() {
        return new Vm();
    }

    /**
     * Returns a new val.
     *
     * @return a new val.
     */
    public final Val newVal() {
        return new Val(this);
    }

    /**
     * Returns a new val with the shared vars.
     *
     * @param sharedVars shared vars as the initial variable mapping.
     * @return a new val.
     */
    public final Val newVal(SharedVars sharedVars) {
        return new Val(this, sharedVars);
    }

    /**
     * Runs a new execution stack which starts from the bootstrap action.
     *
     * <p>The execution starts from {@code bootstrapAction}.</p>
     *
     * <p>If the execution succeeds with a result,
     * the method invokes {@code onReturned} with the result val.
     * If the execution fails,
     * the method invokes {@code onRaised} with the exception.</p>
     *
     * <p>Example:</p>
     *
     * <pre>
     * FunVal fun = ...;
     * boolean isSuccess = vm.run(
     *      c -&gt; c.call(fun).args(x, y),
     *      result -&gt; { System.out.printf("OK!: %s%n", result); return true; },
     *      exc -&gt; { System.out.printf("Bad: %s", exc); return false; }
     * );
     * </pre>
     *
     * @param bootstrapAction the action which is called to start the execution.
     * @param onReturned called when the execution succeeds with a result.
     * @param onRaised called when the execution fails with an exception.
     * @param <T> the result type of {@code onReturned} and {@code onRaised}.
     * @return the result of {@code onReturned} or {@code onRaised}.
     */
    public <T> T run(
            HostFunAction bootstrapAction,
            Function<? super Val, ? extends T> onReturned,
            Function<? super ExceptionVal, ? extends T> onRaised) {
        FunVal bootFun = this.fun.make("(bootstrap)").action(bootstrapAction);
        StackMachine stackMachine = new StackMachine(this);
        StackMachine.Outcome outcome = stackMachine.run(bootFun);
        return outcome instanceof StackMachine.Outcome.Returned ret
            ? onReturned.apply(ret.val())
            : onRaised.apply(((StackMachine.Outcome.Raised) outcome).exception());
    }

}

// vim: et sw=4 sts=4 fdm=marker
