package org.kink_lang.kink.hostfun.graph.impl;

import org.kink_lang.kink.*;
import org.kink_lang.kink.internal.function.ThrowingFunction3;
import org.kink_lang.kink.hostfun.HostFunReaction;
import org.kink_lang.kink.hostfun.HostResult;
import org.kink_lang.kink.hostfun.graph.CallGraphNodeToRecv;
import org.kink_lang.kink.hostfun.graph.CallGraphNodeToArgs;
import org.kink_lang.kink.hostfun.graph.GraphNode;
import org.kink_lang.kink.hostfun.HostContext;

/**
 * Call graph with the form X.foo(,,,) or X.foo[Y](,,,).
 */
class MethodCallGraph implements CallGraphNodeToRecv {

    /** The vm. */
    private final Vm vm;

    /** The owner of the fun. */
    private final GraphNode owner;

    /** The sym handle of the fun. */
    private final int symHandle;

    /** Emits the recv of the calling. */
    private final ThrowingFunction3<HostContext, Val, HostFunReaction, HostResult> recvEmitter;

    /** The args of the fun. */
    private final GraphNode[] args;

    /**
     * Makes a call graph in the form ownerRecv.symHandle(...args).
     */
    static MethodCallGraph withOwnerRecv(
            Vm vm, GraphNode ownerRecv, int symHandle, GraphNode[] args) {
        return new MethodCallGraph(
                vm, ownerRecv, symHandle, (c, o, handler) -> handler.reaction(c, o), args);
    }

    /**
     * Makes a call graph in the form owner.symHandle[recv](...args).
     */
    static MethodCallGraph withDistinctRecv(
            Vm vm, GraphNode owner, int symHandle, GraphNode recv, GraphNode[] args) {
        return new MethodCallGraph(
                vm, owner, symHandle, (c, o, handler) -> c.call(recv).on(handler), args);
    }

    /**
     * Constructs a call graph.
     */
    private MethodCallGraph(Vm vm, GraphNode owner, int symHandle
            , ThrowingFunction3<HostContext, Val, HostFunReaction, HostResult> recvEmitter
            , GraphNode[] args) {
        this.vm = vm;
        this.owner = owner;
        this.recvEmitter = recvEmitter;
        this.symHandle = symHandle;
        this.args = args;
    }

    @Override
    public HostResult evaluateIn(HostContext c) {
        return c.call(this.owner).on((cc, ownerVal) -> {
            return recvEmitter.apply(cc, ownerVal, (c3, recvVal) -> {
                return new ArgsCollector(vm, args).collectArgs(c3, (c4, argVals) -> {
                    return c4.call(ownerVal, symHandle).recv(recvVal).args(argVals);
                });
            });
        });
    }

    @Override
    public CallGraphNodeToArgs recv(GraphNode recv) {
        return withDistinctRecv(vm, owner, symHandle, recv, args);
    }

    @Override
    public GraphNode args(GraphNode... args) {
        return new MethodCallGraph(this.vm, this.owner, this.symHandle, this.recvEmitter, args);
    }

}

// vim: et sw=4 sts=4 fdm=marker
