/**
 * Interfaces used to make
 * <a href="../../../../../../manual/language/execution.html#host-system-functions">
 * host system functions</a>, or host funs,
 * which are Kink functions defined in Java code.
 *
 * <p>Host funs can be built using {@linkplain HostFunBuilder},
 * through {@linkplain org.kink_lang.kink.FunHelper#make() vm.fun.make()}
 * or {@linkplain org.kink_lang.kink.FunHelper#make(String) vm.fun.make(desc)}.</p>
 *
 * <p>The following example makes a host fun which compares two nums
 * then returns the smaller one.</p>
 *
 * <pre>
 *  FunVal min = vm.fun.make("min(x y)").take(2).action(c -&gt; {
 *      Val x = c.arg(0);
 *      Val y = c.arg(1);
 *      if (! (x instanceof NumVal &amp;&amp; y instanceof NumVal)) {
 *          return c.raise("max(x y): required two nums, but were not");
 *      }
 *      BigDecimal xn = ((NumVal) x).bigDecimal();
 *      BigDecimal yn = ((NumVal) y).bigDecimal();
 *      return xn.compareTo(yn) &lt;= 0 ? x : y;
 *  });
 * </pre>
 *
 * <p>If the number of arguments passed to the fun is not two,
 * the fun raises a Kink exception, because
 * {@link HostFunBuilder#take(int) take(2)} is specified.
 * After the arity checking, the control is delegated to the lambda given to
 * {@link HostFunBuilder#action(HostFunAction) action} method.</p>
 *
 * <p>The following is a quick guidance to compose a host fun:</p>
 *
 * <h2>Take a recv and args</h2>
 *
 * <p>You can get the recv passed to the fun call using {@link CallContext#recv()}.</p>
 *
 * <p>You can get the number of args using {@link CallContext#argCount()},
 * and get an arg using {@link CallContext#arg(int)}.</p>
 *
 * <h2>Return a val</h2>
 *
 * <p>If you want the fun to return a val, just return the val as the result of the lambda,
 * as in the example program above.</p>
 *
 * <h2>Raise an exception</h2>
 *
 * <p>If you want to raise an exception, retrun the result of
 * {@link HostContext#raise(String)} or {@link HostContext#raise(Throwable)}
 * as in the example program above.</p>
 *
 * <h2>Call a fun</h2>
 *
 * <p>You can call a fun as both a tail call and a non-tail call.</p>
 *
 * <p>If you want to call a fun as a tail call,
 * use flowish API starting from
 * {@linkplain HostContext}.call methods without terminating by
 * {@linkplain CallFlowToOn#on(HostFunReaction)}.</p>
 *
 * <p>Example:</p>
 *
 * <pre>
 *  int exitHandle = vm.sym.handleFor("exit");
 *  FunVal abort = vm.fun.make().action(c -&gt; {
 *      return c.call("kink/PROCESS", exitHandle).args(vm.num.of(1));
 *  });
 * </pre>
 *
 * <p>If you want to call a fun as a non-tail call, use flowish API starting from
 * {@linkplain HostContext}.call methods terminating by
 * {@linkplain CallFlowToOn#on(HostFunReaction)}.</p>
 *
 * <p>Example:</p>
 *
 * <pre>
 *  int opLtHandle = vm.sym.handleFor("op_lt");
 *  FunVal min = vm.fun.make("min(x y)").take(2).action(c -&gt; {
 *      Val x = c.arg(0);
 *      Val y = c.arg(1);
 *      return c.call(y, opLtHandle).args(x).on((cc, yIsSmaller) -&gt; {
 *          // yIsSmaller is the result of y.op_lt(x)
 *          return yIsSmaller.equals(vm.bool.trueVal) ? y
 *              : yIsSmaller.equals(vm.bool.falseVal) ? x
 *              : cc.raise("min(x y): got non-bool result from &lt; operator");
 *      });
 *  });
 * </pre>
 *
 * <p>{@link CallContext} given to the {@code action} lambda ({@code c} in the program above)
 * is valid until the lambda returns the result.
 * Thus, in the {@code on} lambda, {@link HostContext} given to the {@code on} lambda
 * ({@code cc} in the program above) must be used instead of {@code c}.</p>
 *
 * <p>Note: the Kink runtime manages its own runtime.
 * Thus, an action fun returns a HostResult to the Kink runtime to let it call a Kink fun,
 * instead of calling the Kink fun on the Java stack frame of the {@code action} lambda itself.
 * This kind of technique is called
 * “<a  href="http://gcc.gnu.org/onlinedocs/gcc/Nested-Functions.html">trampoline</a>.”</p>
 *
 * <h2>Call a graph</h2>
 *
 * <p>Trampoline code described in the previous section can become messy easily,
 * when calling multiple functions.
 * In this case, you can use {@link org.kink_lang.kink.hostfun.graph Graph DSL}.
 * Graph DSL abstracts out a chain of {@link CallFlowToOn#on(ThrowingFunction2)}.</p>
 *
 * <p>Example:</p>
 *
 * <pre>
 *  FunVal min = vm.fun.make("min(x y)").take(2).action(c -&gt; {
 *      Val x = c.arg(0);
 *      Val y = c.arg(1);
 *      if (! (x instanceof NumVal &amp;&amp; y instanceof NumVal)) {
 *          // call a graph node
 *          return c.call(vm.graph.raiseFormat(
 *              "min(x y): required two nums, but got {} and {}",
 *              vm.graph.repr(x),
 *              vm.graph.repr(y)));
 *      }
 *      BigDecimal xn = ((NumVal) x).bigDecimal();
 *      BigDecimal yn = ((NumVal) y).bigDecimal();
 *      return xn.compareTo(yn) &lt;= 0 ? x : y;
 *  });
 * </pre>
 */
package org.kink_lang.kink.hostfun;

// vim: et sw=4 sts=4 fdm=marker
