package org.kink_lang.kink.internal.compile.bootstrap;

import java.lang.invoke.CallSite;
import java.lang.invoke.ConstantCallSite;
import java.lang.invoke.MethodHandle;
import java.lang.invoke.MethodHandles;
import java.lang.invoke.MethodHandles.Lookup;
import java.lang.invoke.MethodType;
import java.math.BigDecimal;

import org.kink_lang.kink.Val;
import org.kink_lang.kink.Vm;
import org.kink_lang.kink.internal.intrinsicsupport.PreloadedFuns;
import org.kink_lang.kink.internal.compile.tempval.IntVal;

/**
 * Bootstrapper of constant values.
 */
public final class ConstBootstrapper {

    /**
     * Not instantiated.
     */
    private ConstBootstrapper() {
    }

    /**
     * Bootstrap BigDecimal constant.
     * Avoiding condy because of JDK-8280473.
     *
     * @param caller the lookup of the caller.
     * @param name not used.
     * @param type ()BigDecimal.
     * @param decimalStr the source of the decimal.
     * @return the constant call site of the BigDecimal constant.
     */
    public static CallSite bootstrapBigDecimal(
            Lookup caller,
            String name,
            MethodType type,
            String decimalStr) {
        BigDecimal decimal = getBigDecimal(decimalStr);
        return new ConstantCallSite(MethodHandles.constant(BigDecimal.class, decimal));
    }

    /**
     * Gets a BigDecimal number.
     */
    static BigDecimal getBigDecimal(String decimalStr) {
        BigDecimal decimal = new BigDecimal(decimalStr);
        if (decimal.scale() == 0) {
            try {
                return BigDecimal.valueOf(decimal.longValueExact());
            } catch (ArithmeticException ex) {
                return decimal;
            }
        } else {
            return decimal;
        }
    }

    /**
     * Gets nada.
     * Avoiding condy because of JDK-8280473.
     *
     * @param caller the lookup of the caller.
     * @param name unused.
     * @param type ()Val.
     * @return a constant call site of nada.
     * @throws Throwable any throwable.
     */
    public static CallSite bootstrapNada(
            Lookup caller,
            String name,
            MethodType type) throws Throwable {
        return new ConstantCallSite(MethodHandles.constant(Val.class, getVm(caller).nada));
    }

    /**
     * Bootstarp true.
     * Avoiding condy because of JDK-8280473.
     *
     * @param caller the lookup of the caller.
     * @param name unused.
     * @param type ()Val.
     * @return a constant call site of true.
     * @throws Throwable any throwable.
     */
    public static CallSite bootstrapTrue(
            Lookup caller,
            String name,
            MethodType type) throws Throwable {
        var trueVal = getVm(caller).bool.trueVal;
        var mh = MethodHandles.constant(Val.class, trueVal);
        return new ConstantCallSite(mh);
    }

    /**
     * Bootstarp false.
     * Avoiding condy because of JDK-8280473.
     *
     * @param caller the lookup of the caller.
     * @param name unused.
     * @param type ()Val.
     * @return a constant call site of false.
     * @throws Throwable any throwable.
     */
    public static CallSite bootstrapFalse(
            Lookup caller,
            String name,
            MethodType type) throws Throwable {
        var falseVal = getVm(caller).bool.falseVal;
        var mh = MethodHandles.constant(Val.class, falseVal);
        return new ConstantCallSite(mh);
    }

    /**
     * Bootstarp a str.
     * Avoiding condy because of JDK-8280473.
     *
     * @param caller the lookup of the caller.
     * @param name unused.
     * @param type ()Val.
     * @param string the string.
     * @return a constant call site of the str.
     * @throws Throwable any throwable.
     */
    public static CallSite bootstrapStr(
            Lookup caller,
            String name,
            MethodType type,
            String string) throws Throwable {
        var strVal = StrRegistry.strVal(getVm(caller), string);
        var mh = MethodHandles.constant(Val.class, strVal);
        return new ConstantCallSite(mh);
    }

    /**
     * Bootstarp a num.
     * Avoiding condy because of JDK-8280473.
     *
     * @param caller the lookup of the caller.
     * @param name unused.
     * @param type ()Val.
     * @param decimalStr the string representation of BigDecimal.
     * @return a constant call site of the num val.
     * @throws Throwable any throwable.
     */
    public static CallSite bootstrapNum(
            Lookup caller,
            String name,
            MethodType type,
            String decimalStr) throws Throwable {
        var numVal = NumRegistry.numVal(getVm(caller), new BigDecimal(decimalStr));
        var mh = MethodHandles.constant(Val.class, numVal);
        return new ConstantCallSite(mh);
    }

    /**
     * Bootstarp an temporary int val.
     * Avoiding condy because of JDK-8280473.
     *
     * @param caller the lookup of the caller.
     * @param name unused.
     * @param type ()Val.
     * @param num the int num.
     * @return a constant call site of the temporary int val.
     * @throws Throwable any throwable.
     */
    public static CallSite bootstrapInt(
            Lookup caller,
            String name,
            MethodType type,
            int num) throws Throwable {
        var intVal = IntVal.of(getVm(caller), num);
        var mh = MethodHandles.constant(Val.class, intVal);
        return new ConstantCallSite(mh);
    }

    /**
     * Bootstarp a preloaded fun.
     * Avoiding condy because of JDK-8280473.
     *
     * @param caller the lookup of the caller.
     * @param name unused.
     * @param type ()Val.
     * @param sym the sym of the fun var.
     * @return a constant call site of the fun val.
     * @throws Throwable any throwable.
     */
    public static CallSite bootstrapPreloaded(
            Lookup caller,
            String name,
            MethodType type,
            String sym) throws Throwable {
        var funVal = PreloadedFuns.preloadedFun(getVm(caller), sym);
        var mh = MethodHandles.constant(Val.class, funVal);
        return new ConstantCallSite(mh);
    }

    /**
     * Gets the vm.
     */
    private static Vm getVm(Lookup caller) throws Throwable {
        MethodHandle vmMh = caller.findStaticGetter(caller.lookupClass(), "vmStatic", Vm.class);
        return (Vm) vmMh.invoke();
    }

}

// vim: et sw=4 sts=4 fdm=marker
