package org.kink_lang.kink.internal.compile.classgen;

import java.lang.invoke.MethodHandle;
import java.lang.invoke.MethodHandles.Lookup;
import java.util.List;
import java.util.Map;

import org.kink_lang.kink.FunVal;
import org.kink_lang.kink.Vm;
import org.kink_lang.kink.internal.callstack.Trace;
import org.kink_lang.kink.internal.compile.javaclassir.JavaClassIr;
import org.kink_lang.kink.internal.compile.javaclassir.JcirFactory;
import org.kink_lang.kink.internal.control.Control;

/**
 * Generates a fun class from IR.
 */
public class ClassGenerator {

    /** Logger for the class. */
    private static final System.Logger LOGGER = System.getLogger(ClassGenerator.class.getName());

    /** The vm. */
    private final Vm vm;

    /** Lookup of org.kink_lang.kink package. */
    private final Lookup packageLookup;

    /**
     * Constructs a generator.
     *
     * @param vm the vm.
     * @param packageLookup the lookup with privilege of org.kink_lang.kink package.
     */
    public ClassGenerator(Vm vm, Lookup packageLookup) {
        this.vm = vm;
        this.packageLookup = packageLookup;
    }

    /**
     * Generates a fun class, initializing static fields.
     *
     * @param jcir the IR of the fun.
     * @return the lookup of the generated class.
     * @throws Exception reflection exception.
     */
    public Lookup generate(JavaClassIr jcir) throws Exception {
        LOGGER.log(System.Logger.Level.DEBUG, "generating class for {0}", jcir.desc());
        byte[] byteCode = new BytecodeGenerator().generate(jcir);
        Lookup lookup = this.packageLookup.defineHiddenClass(byteCode, false);
        @SuppressWarnings("unchecked")
        Class<? extends FunVal> klass = (Class<? extends FunVal>) lookup.lookupClass();
        LOGGER.log(System.Logger.Level.DEBUG,
                "generated class for {0}: {1}", jcir.desc(), klass.getName());
        initTraceMap(lookup, klass, jcir.traces());
        initVmStatic(lookup, klass);
        initChildJcirFactories(lookup, klass, jcir.childJcirFactories());
        return lookup;
    }

    /**
     * Initializes the value of traceMap field.
     */
    private void initTraceMap(
            Lookup lookup,
            Class<? extends FunVal> klass,
            Map<Integer, Trace> traces) {
        Control.runWrappingThrowable(() -> {
            MethodHandle mh = lookup.findStaticSetter(klass, "traceMap", Map.class);
            mh.invoke(traces);
            return null;
        });
    }

    /**
     * Initializes vmStatic field.
     */
    private void initVmStatic(Lookup lookup, Class<? extends FunVal> klass) {
        Control.runWrappingThrowable(() -> {
            MethodHandle mh = lookup.findStaticSetter(klass, "vmStatic", Vm.class);
            mh.invoke(this.vm);
            return null;
        });
    }

    /**
     * Initializes childJcirFactories field.
     */
    private void initChildJcirFactories(
            Lookup lookup,
            Class<? extends FunVal> klass,
            List<JcirFactory> childJcirFactories) {
        Control.runWrappingThrowable(() -> {
            MethodHandle mh = lookup.findStaticSetter(klass, "childJcirFactories", List.class);
            mh.invoke(childJcirFactories);
            return null;
        });
    }

}

// vim: et sw=4 sts=4 fdm=marker
