package org.kink_lang.kink.internal.compile.javaclassir;

import java.util.List;

import org.objectweb.asm.Handle;
import org.objectweb.asm.Type;
import org.objectweb.asm.commons.GeneratorAdapter;
import org.objectweb.asm.commons.Method;

import java.lang.invoke.MethodType;

/**
 * Thin wrapping records of GeneratorAdapter methods.
 */
public sealed interface Insn {

    /**
     * loadthis insn.
     */
    record LoadThis() implements Insn {
        @Override
        public void generateBytecodeSnippet(BytecodeGenState state) {
            state.ga().loadThis();
        }
    }

    /**
     * load-arg insn.
     *
     * @param index the index of the arg.
     */
    record LoadArg(int index) implements Insn {
        @Override
        public void generateBytecodeSnippet(BytecodeGenState state) {
            state.ga().loadArg(index());
        }
    }

    /**
     * store-arg insn.
     *
     * @param index the index of the arg.
     */
    record StoreArg(int index) implements Insn {
        @Override
        public void generateBytecodeSnippet(BytecodeGenState state) {
            state.ga().storeArg(index());
        }
    }

    /**
     * push-int insn.
     *
     * @param value to push.
     */
    record PushInt(int value) implements Insn {
        @Override
        public void generateBytecodeSnippet(BytecodeGenState state) {
            state.ga().push(value());
        }
    }

    /**
     * push-string or null insn.
     *
     * @param value string or null.
     */
    record PushString(String value) implements Insn {
        @Override
        public void generateBytecodeSnippet(BytecodeGenState state) {
            state.ga().push(value());
        }
    }

    /**
     * add-int insn.
     */
    record AddInt() implements Insn {
        @Override
        public void generateBytecodeSnippet(BytecodeGenState state) {
            state.ga().math(GeneratorAdapter.ADD, Type.getType(int.class));
        }
    }

    /**
     * subtract-int insn.
     */
    record SubInt() implements Insn {
        @Override
        public void generateBytecodeSnippet(BytecodeGenState state) {
            state.ga().math(GeneratorAdapter.SUB, Type.getType(int.class));
        }
    }

    /**
     * Get-field insn.
     *
     * @param owner the owner type.
     * @param name the name of the field.
     * @param type the field type.
     */
    record GetField(Type owner, String name, Type type) implements Insn {
        @Override
        public void generateBytecodeSnippet(BytecodeGenState state) {
            state.ga().getField(owner(), name(), type());
        }
    }

    /**
     * Get-static-field insn.
     *
     * @param owner the owner type.
     * @param name the name of the field.
     * @param type the field type.
     */
    record GetStatic(Type owner, String name, Type type) implements Insn {
        @Override
        public void generateBytecodeSnippet(BytecodeGenState state) {
            state.ga().getStatic(owner(), name(), type());
        }
    }

    /**
     * Invoke-static insn.
     *
     * @param owner the owner of the method
     * @param method the method.
     */
    record InvokeStatic(Type owner, Method method) implements Insn {
        @Override
        public void generateBytecodeSnippet(BytecodeGenState state) {
            state.ga().invokeStatic(owner(), method());
        }
    }

    /**
     * Invoke-virtual insn.
     *
     * @param owner the owner of the method
     * @param method the method.
     */
    record InvokeVirtual(Type owner, Method method) implements Insn {
        @Override
        public void generateBytecodeSnippet(BytecodeGenState state) {
            state.ga().invokeVirtual(owner(), method());
        }
    }

    /**
     * Invoke-special insn.
     *
     * @param owner the owner of the method
     * @param method the method.
     */
    record InvokeConstructor(Type owner, Method method) implements Insn {
        @Override
        public void generateBytecodeSnippet(BytecodeGenState state) {
            state.ga().invokeConstructor(owner(), method());
        }
    }

    /**
     * New-array insn.
     *
     * @param type the element type
     */
    record NewArray(Type type) implements Insn {
        @Override
        public void generateBytecodeSnippet(BytecodeGenState state) {
            state.ga().newArray(type());
        }
    }

    /**
     * New-instance insn.
     *
     * @param type the type of the new object.
     */
    record NewInstance(Type type) implements Insn {
        @Override
        public void generateBytecodeSnippet(BytecodeGenState state) {
            state.ga().newInstance(type());
        }
    }

    /**
     * Dup insn.
     */
    record Dup() implements Insn {
        @Override
        public void generateBytecodeSnippet(BytecodeGenState state) {
            state.ga().dup();
        }
    }

    /**
     * Invoke-dynamic insn.
     *
     * @param descriptorString the descriptor of the insn type.
     * @param bootstrapHandle the handle of the bootstrap method.
     * @param bootstrapArgs the static args for the bootstrap method.
     */
    record InvokeDynamic(
            String descriptorString,
            Handle bootstrapHandle,
            List<Object> bootstrapArgs) implements Insn {

        /**
         * Constructs an insn.
         */
        public InvokeDynamic {
            bootstrapArgs = List.copyOf(bootstrapArgs);
        }

        /**
         * Constructs an insn.
         *
         * @param descriptor the descriptor of the insn type.
         * @param bootstrapHandle the handle of the bootstrap method.
         * @param bootstrapArgs the static args for the bootstrap method.
         */
        public InvokeDynamic(
                MethodType descriptor,
                Handle bootstrapHandle,
                List<Object> bootstrapArgs) {
            this(descriptor.descriptorString(), bootstrapHandle, bootstrapArgs);
        }

        @Override
        public void generateBytecodeSnippet(BytecodeGenState state) {
            state.ga().invokeDynamic(
                    bootstrapHandle().getName(),
                    descriptorString(),
                    bootstrapHandle(),
                    bootstrapArgs().stream().toArray(Object[]::new));
        }
    }

    /**
     * Return insn.
     */
    record ReturnValue() implements Insn {
        @Override
        public void generateBytecodeSnippet(BytecodeGenState state) {
            state.ga().returnValue();
        }
    }

    /**
     * Throw insn.
     *
     * @param type the type of the exception class.
     * @param message the exception message.
     */
    record Throw(Type type, String message) implements Insn {
        @Override
        public void generateBytecodeSnippet(BytecodeGenState state) {
            state.ga().throwException(type(), message());
        }
    }

    /**
     * Instanceof insn.
     *
     * @param type the type to test.
     */
    record InstanceOf(Type type) implements Insn {
        @Override
        public void generateBytecodeSnippet(BytecodeGenState state) {
            state.ga().instanceOf(type());
        }
    }

    /**
     * Check-cast insn.
     *
     * @param type the type to cast the value to.
     */
    record CheckCast(Type type) implements Insn {
        @Override
        public void generateBytecodeSnippet(BytecodeGenState state) {
            state.ga().checkCast(type());
        }
    }

    /**
     * Goto insn.
     *
     * @param labelKey the label key of the destination.
     */
    record GoTo(String labelKey) implements Insn {
        @Override
        public void generateBytecodeSnippet(BytecodeGenState state) {
            state.ga().goTo(state.labelFor(labelKey()));
        }
    }

    /**
     * If-eq insn.
     *
     * @param type the type of the values.
     * @param labelKey the label key of the destination.
     */
    record IfEq(Type type, String labelKey) implements Insn {
        @Override
        public void generateBytecodeSnippet(BytecodeGenState state) {
            state.ga().ifCmp(type(), GeneratorAdapter.EQ, state.labelFor(labelKey()));
        }
    }

    /**
     * If-eq insn for int.
     *
     * @param labelKey the label key of the destination.
     */
    record IfEqInt(String labelKey) implements Insn {
        @Override
        public void generateBytecodeSnippet(BytecodeGenState state) {
            state.ga().ifICmp(GeneratorAdapter.EQ, state.labelFor(labelKey()));
        }
    }

    /**
     * If-lt insn for int.
     *
     * @param labelKey the label key of the destination.
     */
    record IfLtInt(String labelKey) implements Insn {
        @Override
        public void generateBytecodeSnippet(BytecodeGenState state) {
            state.ga().ifICmp(GeneratorAdapter.LT, state.labelFor(labelKey()));
        }
    }

    /**
     * If-gt insn for int.
     *
     * @param labelKey the label key of the destination.
     */
    record IfGtInt(String labelKey) implements Insn {
        @Override
        public void generateBytecodeSnippet(BytecodeGenState state) {
            state.ga().ifICmp(GeneratorAdapter.GT, state.labelFor(labelKey()));
        }
    }

    /**
     * If-ge insn for int.
     *
     * @param labelKey the label key of the destination.
     */
    record IfGeInt(String labelKey) implements Insn {
        @Override
        public void generateBytecodeSnippet(BytecodeGenState state) {
            state.ga().ifICmp(GeneratorAdapter.GE, state.labelFor(labelKey()));
        }
    }

    /**
     * If-nonzero insn.
     *
     * @param labelKey the label key of the destination.
     */
    record IfNonZero(String labelKey) implements Insn {
        @Override
        public void generateBytecodeSnippet(BytecodeGenState state) {
            state.ga().ifZCmp(GeneratorAdapter.NE, state.labelFor(labelKey()));
        }
    }

    /**
     * If-nonnull insn.
     *
     * @param labelKey the label key of the destination.
     */
    record IfNonNull(String labelKey) implements Insn {
        @Override
        public void generateBytecodeSnippet(BytecodeGenState state) {
            state.ga().ifNonNull(state.labelFor(labelKey()));
        }
    }

    /**
     * Marks the label.
     *
     * @param labelKey the label key to mark.
     */
    record Mark(String labelKey) implements Insn {
        @Override
        public void generateBytecodeSnippet(BytecodeGenState state) {
            state.ga().mark(state.labelFor(labelKey()));
        }
    }

    /**
     * Table-switch insn.
     *
     * @param switchKey the key of the switch.
     */
    record Switch(String switchKey) implements Insn {
        @Override
        public void generateBytecodeSnippet(BytecodeGenState state) {
            state.generateSwitch(switchKey());
        }
    }

    /**
     * Marks the case label for the table-switch.
     *
     * @param switchKey the key of the switch.
     * @param num the num of the case label.
     */
    record Case(String switchKey, int num) implements Insn {
        @Override
        public void generateBytecodeSnippet(BytecodeGenState state) {
            state.ga().mark(state.caseLabel(switchKey(), num()));
        }
    }

    /**
     * Marks the default label for the table-switch.
     *
     * @param switchKey the key of the switch.
     */
    record Default(String switchKey) implements Insn {
        @Override
        public void generateBytecodeSnippet(BytecodeGenState state) {
            state.ga().mark(state.defaultLabel(switchKey()));
        }
    }

    /**
     * Array-store insn.
     *
     * @param type the type of the elem.
     */
    record ArrayStore(Type type) implements Insn {
        @Override
        public void generateBytecodeSnippet(BytecodeGenState state) {
            state.ga().arrayStore(type());
        }
    }

    /**
     * New-local and store-local insns.
     *
     * @param localKey the key of the local var.
     * @param type the type of the local var.
     */
    record StoreNewLocal(String localKey, Type type) implements Insn {
        @Override
        public void generateBytecodeSnippet(BytecodeGenState state) {
            int local = state.ga().newLocal(type());
            state.registerLocal(localKey(), local);
            state.ga().storeLocal(local);
        }
    }

    /**
     * Load-local insn.
     *
     * @param localKey the key of the local var.
     */
    record LoadLocal(String localKey) implements Insn {
        @Override
        public void generateBytecodeSnippet(BytecodeGenState state) {
            state.ga().loadLocal(state.getLocal(localKey()));
        }
    }

    /**
     * Generates a fragment of bytecode invoking GeneratorAdapter methods.
     *
     * @param state the state of bytecode generation.
     */
    void generateBytecodeSnippet(BytecodeGenState state);

}

// vim: et sw=4 sts=4 fdm=marker
