package org.kink_lang.kink.internal.compile.javaclassir;

import java.lang.invoke.CallSite;
import java.lang.invoke.MethodHandles.Lookup;
import java.lang.invoke.MethodType;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.objectweb.asm.Handle;
import org.objectweb.asm.Opcodes;
import org.objectweb.asm.Type;
import org.objectweb.asm.commons.Method;

import org.kink_lang.kink.FunVal;
import org.kink_lang.kink.NumHelper;
import org.kink_lang.kink.NumVal;
import org.kink_lang.kink.Val;
import org.kink_lang.kink.VarrefHelper;
import org.kink_lang.kink.VarrefVal;
import org.kink_lang.kink.VecHelper;
import org.kink_lang.kink.VecVal;
import org.kink_lang.kink.Vm;
import org.kink_lang.kink.internal.callstack.CallStack;
import org.kink_lang.kink.internal.callstack.Cse;
import org.kink_lang.kink.internal.callstack.FakeCallTraceCse;
import org.kink_lang.kink.internal.callstack.Location;
import org.kink_lang.kink.internal.callstack.ResumeCse;
import org.kink_lang.kink.internal.callstack.Trace;
import org.kink_lang.kink.internal.compile.bootstrap.ConstBootstrapper;
import org.kink_lang.kink.internal.compile.bootstrap.TraceBootstrapper;
import org.kink_lang.kink.internal.compile.tempval.IntVal;
import org.kink_lang.kink.internal.program.itree.*;
import org.kink_lang.kink.internal.vec.VecInternal;

/**
 * Generates insns of doResume.
 */
public class InsnsGenerator {

    /** The vm. */
    private final Vm vm;

    /** The program name. */
    private final String programName;

    /** The program text. */
    private final String programText;

    /** Generates binding. */
    private final BindingGenerator bindingGen;

    /** Generates lvar access insns. */
    private final LvarAccessGenerator lvarAccGen;

    /** Generates letrec insns. */
    private final LetRecGenerator letRecGen;

    /** Generates control insns. */
    private final ControlGenerator controlGen;

    /** The key str supplier. */
    private final KeyStrSupplier keySup;

    /** Generates make-ssa-fun insns. */
    private final MakeFastFunGenerator makeFastFunGen;

    /** The trace accumulator. */
    private final TraceAccumulator traceAccum;

    /** The program counter supplier. */
    private final ProgramCounterSupplier pcSup;

    /** The accumulator of child JavaClassIr factories. */
    private final ChildJcirAccumulator jcirAccum;

    /**
     * Constructs a generator.
     *
     * @param vm the vm.
     * @param programName the program name.
     * @param programText the program text.
     * @param bindingGen generates the binding of the fun call.
     * @param lvarAccGen generates lvar access insns.
     * @param makeFastFunGen generates make-ssa-fun insns.
     * @param letRecGen generates letrec insns.
     * @param controlGen generates control insns.
     * @param keySup the key str supplier.
     * @param traceAccum the trace accumulator.
     * @param pcSup the program counter supplier.
     * @param jcirAccum the accumulator of child JavaClassIr factories.
     */
    public InsnsGenerator(
            Vm vm,
            String programName,
            String programText,
            BindingGenerator bindingGen,
            LvarAccessGenerator lvarAccGen,
            MakeFastFunGenerator makeFastFunGen,
            LetRecGenerator letRecGen,
            ControlGenerator controlGen,
            KeyStrSupplier keySup,
            TraceAccumulator traceAccum,
            ProgramCounterSupplier pcSup,
            ChildJcirAccumulator jcirAccum) {
        this.vm = vm;
        this.programName = programName;
        this.programText = programText;
        this.bindingGen = bindingGen;
        this.lvarAccGen = lvarAccGen;
        this.makeFastFunGen = makeFastFunGen;
        this.letRecGen = letRecGen;
        this.controlGen = controlGen;
        this.keySup = keySup;
        this.traceAccum = traceAccum;
        this.pcSup = pcSup;
        this.jcirAccum = jcirAccum;
    }

    /** Type of StackMachine class. */
    static final Type STACKMACHINE_TYPE = Type.getType("Lorg/kink_lang/kink/StackMachine;");

    /** Type of DataStack class. */
    static final Type DATASTACK_TYPE = Type.getType("Lorg/kink_lang/kink/DataStack;");

    /** Switch key str for programCounter. */
    static final String PROGRAMCOUNTER_KEY = "pc";

    /** Insn of producing nada. */
    static final Insn PRODUCE_NADA = new Insn.InvokeDynamic(
            MethodType.methodType(Val.class),
            new Handle(Opcodes.H_INVOKESTATIC,
                Type.getType(ConstBootstrapper.class).getInternalName(),
                "bootstrapNada",
                MethodType.methodType(
                    CallSite.class,
                    Lookup.class,
                    String.class,
                    MethodType.class).descriptorString(),
                false),
            List.of());

    /** Insn of producing stackMachine arg. */
    static final Insn LOAD_STACKMACHINE = new Insn.LoadArg(0);

    /** Insn of storeing contParam arg. */
    static final Insn STORE_CONTPARAM = new Insn.StoreArg(1);

    /** Insn of producing contParam arg. */
    static final Insn LOAD_CONTPARAM = new Insn.LoadArg(1);

    /** Insn of producing callStack arg. */
    static final Insn LOAD_CALLSTACK = new Insn.LoadArg(3);

    /** Insn of producing dataStack arg. */
    static final Insn LOAD_DATASTACK = new Insn.LoadArg(4);

    /** Insn of producing argCount arg. */
    static final Insn LOAD_ARGCOUNT = new Insn.LoadArg(5);

    /** Insn of invoking DataStack.topOffset. */
    static final Insn INVOKE_TOP_OFFSET = new Insn.InvokeVirtual(
            InsnsGenerator.DATASTACK_TYPE,
            new Method("topOffset", Type.INT_TYPE, new Type[0]));

    /** Insn of invoking DataStack.push. */
    static final Insn INVOKE_PUSH_TO_DATASTACK = new Insn.InvokeVirtual(
            InsnsGenerator.DATASTACK_TYPE,
            new Method("push",
                Type.VOID_TYPE,
                new Type[] { Type.getType(Val.class) }));

    /** Insn of invoking DataStack.pop. */
    static final Insn INVOKE_POP_FROM_DATASTACK = new Insn.InvokeVirtual(
            InsnsGenerator.DATASTACK_TYPE,
            new Method("pop", Type.getType(Val.class), new Type[0]));

    /** Insn of invoking DataStack.atOffset. */
    static final Insn INVOKE_AT_OFFSET = new Insn.InvokeVirtual(
            DATASTACK_TYPE,
            new Method("atOffset", Type.getType(Val.class),
                new Type[] { Type.INT_TYPE }));

    /** Insn of invoking DataStack.setAtOffset. */
    static final Insn INVOKE_SET_AT_OFFSET = new Insn.InvokeVirtual(
            InsnsGenerator.DATASTACK_TYPE,
            new Method("setAtOffset",
                Type.VOID_TYPE,
                new Type[] { Type.getType(int.class), Type.getType(Val.class) }));

    /** Insn of invoking DataStack.recv. */
    static final Insn INVOKE_RECV = new Insn.InvokeVirtual(
            InsnsGenerator.DATASTACK_TYPE,
            new Method("recv", Type.getType(Val.class), new Type[0]));

    /** Insn of invoking DataStack.arg. */
    static final Insn INVOKE_ARG = new Insn.InvokeVirtual(
            InsnsGenerator.DATASTACK_TYPE,
            new Method("arg",
                Type.getType(Val.class),
                new Type[] { Type.INT_TYPE }));

    /** Insn of invoking DataStack.argVec. */
    static final Insn INVOKE_ARG_VEC = new Insn.InvokeVirtual(
            InsnsGenerator.DATASTACK_TYPE,
            new Method("argVec",
                Type.getType(VecVal.class),
                new Type[] { Type.INT_TYPE }));

    /** Insn of invoking CallStack.pushCse. */
    static final Insn INVOKE_PUSH_CSE = new Insn.InvokeVirtual(
            Type.getType(CallStack.class),
            new Method("pushCse",
                Type.VOID_TYPE,
                new Type[] {
                    Type.getType(Cse.class),
                    Type.INT_TYPE,
                    Type.INT_TYPE,
                    Type.INT_TYPE }));

    /** Insn of invoking IntVal.of. */
    static final Insn INVOKE_INT_VAL_OF = new Insn.InvokeStatic(
            Type.getType(IntVal.class),
            new Method("of",
                Type.getType(IntVal.class),
                new Type[] { Type.getType(Vm.class), Type.INT_TYPE }));

    /** Insn of invoking DataStack.removeFromOffset. */
    static final Insn INVOKE_REMOVE_FROM_OFFSET = new Insn.InvokeVirtual(
            InsnsGenerator.DATASTACK_TYPE,
            new Method("removeFromOffset", Type.VOID_TYPE,
                new Type[] { Type.INT_TYPE }));

    /** Insn of invoking StackMachine.transitionToCall. */
    static final Insn INVOKE_TRANSITION_TO_CALL = new Insn.InvokeVirtual(
            InsnsGenerator.STACKMACHINE_TYPE,
            new Method("transitionToCall", Type.VOID_TYPE,
                new Type[] { Type.getType(FunVal.class) }));

    /** Insn of invoking StackMachine.transitionToResult. */
    static final Insn INVOKE_TRANSITION_TO_RESULT = new Insn.InvokeVirtual(
            InsnsGenerator.STACKMACHINE_TYPE,
            new Method("transitionToResult",
                Type.VOID_TYPE,
                new Type[] { Type.getType(Val.class) }));

    /** Insn of invoking StackMachine.callFun. */
    static final Insn INVOKE_CALL_FUN = new Insn.InvokeVirtual(
            InsnsGenerator.STACKMACHINE_TYPE,
            new Method("callFun",
                Type.VOID_TYPE,
                new Type[] { Type.getType(ResumeCse.class), Type.INT_TYPE, Type.INT_TYPE }));

    /** Insn of invoking StackMachine.raiseWrongNumberOfArgs. */
    static final Insn INVOKE_RAISE_WRONG_NUMBER_OF_ARGS = new Insn.InvokeVirtual(
            InsnsGenerator.STACKMACHINE_TYPE,
            new Method("raiseWrongNumberOfArgs", Type.VOID_TYPE,
                new Type[] {
                    Type.INT_TYPE,
                    Type.getType(String.class),
                    Type.getType(VecVal.class) }));

    /** Insn of invoking StackMachine.raiseNotVecRhs. */
    static final Insn INVOKE_RAISE_NOT_VEC_RHS = new Insn.InvokeVirtual(
            InsnsGenerator.STACKMACHINE_TYPE,
            new Method("raiseNotVecRhs", Type.VOID_TYPE,
                new Type[] { Type.getType(Val.class) }));

    /** Insn of invoking StackMachine.raiseNotFun. */
    static final Insn INVOKE_RAISE_NOT_FUN = new Insn.InvokeVirtual(
            InsnsGenerator.STACKMACHINE_TYPE,
            new Method("raiseNotFun", Type.VOID_TYPE,
                new Type[] {
                    Type.getType(Val.class),
                    Type.getType(String.class),
                    Type.getType(Trace.class) }));

    /** Insn of invoking VecVal.vecInternal. */
    static final Insn INVOKE_VEC_INTERNAL = new Insn.InvokeVirtual(
            Type.getType(VecVal.class),
            new Method("vecInternal", Type.getType(VecInternal.class), new Type[0]));

    /** Insn of invoking VecInternal.get. */
    static final Insn INVOKE_VEC_INTERNAL_GET = new Insn.InvokeVirtual(
            Type.getType(VecInternal.class),
            new Method("get", Type.getType(Val.class),
                new Type[] { Type.INT_TYPE }));

    /** Insn of invoking VecInternal.size. */
    static final Insn INVOKE_VEC_INTERNAL_SIZE = new Insn.InvokeVirtual(
            Type.getType(VecInternal.class),
            new Method("size", Type.INT_TYPE, new Type[0]));

    /** Insn of invoking VecHelper.of(). */
    static final Insn INVOKE_VEC_OF_EMPTY = new Insn.InvokeVirtual(
            Type.getType(VecHelper.class),
            new Method("of", Type.getType(VecVal.class), new Type[0]));

    /** Insn of invoking VecHelper.of(singleElem). */
    static final Insn INVOKE_VEC_OF_SINGLE = new Insn.InvokeVirtual(
            Type.getType(VecHelper.class),
            new Method("of",
                Type.getType(VecVal.class),
                new Type[] { Type.getType(Val.class) }));

    /** Insn of calling VecVal(vm, vecInternal). */
    static final Insn INVOKE_VEC_CONSTRUCTOR = new Insn.InvokeConstructor(
            Type.getType(VecVal.class),
            new Method("<init>", Type.VOID_TYPE,
                new Type[] { Type.getType(Vm.class),
                    Type.getType(VecInternal.class) }));

    /** Insn of loading vm. */
    static final List<Insn> LOAD_VM = List.of(
            new Insn.LoadThis(),
            new Insn.GetField(JavaClassIr.TYPE_BASE, "vm", Type.getType(Vm.class)));

    /** Insn of loading vm.vec. */
    static final Insn LOAD_VEC_HELPER = new Insn.GetField(
            Type.getType(Vm.class), "vec", Type.getType(VecHelper.class));

    /** Method handle of TraceBootstrapper.bootstrapTrace. */
    static final Handle BOOTSTRAP_TRACE_HANDLE = new Handle(
            Opcodes.H_INVOKESTATIC,
            Type.getType(TraceBootstrapper.class).getInternalName(),
            "bootstrapTrace",
            MethodType.methodType(
                CallSite.class,
                Lookup.class,
                String.class,
                MethodType.class,
                int.class).descriptorString(),
            false);

    /** Method handle of TraceBootstrapper.bootstrapFakeCallCse. */
    static final Handle BOOTSTRAP_FAKE_CALL_CSE = new Handle(
            Opcodes.H_INVOKESTATIC,
            Type.getType(TraceBootstrapper.class).getInternalName(),
            "bootstrapFakeCallCse",
            MethodType.methodType(
                CallSite.class,
                Lookup.class,
                String.class,
                MethodType.class,
                int.class).descriptorString(),
            false);

    /** Method handle of ConstBootstrapper.bootstrapInt. */
    static final Handle BOOTSTRAP_INT_HANDLE = new Handle(
            Opcodes.H_INVOKESTATIC,
            Type.getType(ConstBootstrapper.class).getInternalName(),
            "bootstrapInt",
            MethodType.methodType(
                CallSite.class,
                Lookup.class,
                String.class,
                MethodType.class,
                int.class).descriptorString(),
            false);

    /** Method handle of ConstBootstrapper.bootstrapStr. */
    static final Handle BOOTSTRAP_STR_HANDLE = new Handle(
            Opcodes.H_INVOKESTATIC,
            Type.getType(ConstBootstrapper.class).getInternalName(),
            "bootstrapStr",
            MethodType.methodType(
                CallSite.class,
                Lookup.class,
                String.class,
                MethodType.class,
                String.class).descriptorString(),
            false);

    /** Method handle of ConstBootstrapper.bootstrapNum. */
    static final Handle BOOTSTRAP_NUM_HANDLE = new Handle(
            Opcodes.H_INVOKESTATIC,
            Type.getType(ConstBootstrapper.class).getInternalName(),
            "bootstrapNum",
            MethodType.methodType(
                CallSite.class,
                Lookup.class,
                String.class,
                MethodType.class,
                String.class).descriptorString(),
            false);

    /** Handle of ConstBootstrapper.bootstrapBigDecimal. */
    static final Handle BOOTSTRAP_BIGDECIMAL_HANDLE = new Handle(Opcodes.H_INVOKESTATIC,
            Type.getType(ConstBootstrapper.class).getInternalName(),
            "bootstrapBigDecimal",
            MethodType.methodType(
                CallSite.class,
                Lookup.class,
                String.class,
                MethodType.class,
                String.class).descriptorString(),
            false);

    /** Method handle of GetVarBootstrapper.bootstrapGetVar. */
    static final Handle BOOTSTRAP_GET_VAR_HANDLE = new Handle(
            Opcodes.H_INVOKESTATIC,
            "org/kink_lang/kink/GetVarBootstrapper",
            "bootstrapGetVar",
            MethodType.methodType(
                CallSite.class,
                Lookup.class,
                String.class,
                MethodType.class,
                int.class).descriptorString(),
            false);

    /** Method handle of GetVarBootstrapper.bootstrapSetVar. */
    static final Handle BOOTSTRAP_SET_VAR_HANDLE = new Handle(
            Opcodes.H_INVOKESTATIC,
            "org/kink_lang/kink/SetVarBootstrapper",
            "bootstrapSetVar",
            MethodType.methodType(
                CallSite.class,
                Lookup.class,
                String.class,
                MethodType.class,
                int.class).descriptorString(),
            false);

    /**
     * Generates insns of doResume.
     *
     * @param itree the itree.
     * @param resultCtx the result context.
     * @return the list of insns.
     */
    public List<Insn> generate(Itree itree, ResultContext resultCtx) {
        var visitor = new Visitor(resultCtx);
        return List.copyOf(itree.accept(visitor));
    }

    /**
     * Pushes the trace.
     */
    private Insn produceTrace(Trace trace) {
        int traceKey = traceAccum.add(trace);
        return new Insn.InvokeDynamic(
                MethodType.methodType(Trace.class),
                BOOTSTRAP_TRACE_HANDLE,
                List.of(traceKey));
    }

    /**
     * contParam = var[ownerKey].getVar(sym).
     */
    private List<Insn> deref(String ownerKey, String sym, int pos) {
        List<Insn> insns = new ArrayList<>();

        // contParam = get-var[symHandle](derefOwner)
        insns.add(new Insn.LoadLocal(ownerKey));
        insns.add(new Insn.InvokeDynamic(
                    MethodType.methodType(Val.class, Val.class),
                    BOOTSTRAP_GET_VAR_HANDLE,
                    List.of(vm.sym.handleFor(sym))));
        insns.add(STORE_CONTPARAM);

        // every val has .repr, so null check is not needed
        if (sym.equals("repr")) {
            return insns;
        }

        // if contParam == null; goto derefNonnull
        insns.add(LOAD_CONTPARAM);
        String derefNonnull = keySup.newKeyStr("deref-nonnull");
        insns.add(new Insn.IfNonNull(derefNonnull));

        // stackMachine.transitionToRaiseNoSuchVar(); return
        insns.add(LOAD_STACKMACHINE);
        insns.add(new Insn.PushString(sym));
        insns.add(new Insn.LoadLocal(ownerKey));
        insns.add(produceTrace(Trace.of(new Location(programName, programText, pos))));
        insns.add(new Insn.InvokeVirtual(
                    STACKMACHINE_TYPE,
                    new Method("transitionToRaiseNoSuchVar",
                        Type.VOID_TYPE,
                        new Type[] {
                            Type.getType(String.class),
                            Type.getType(Val.class),
                            Type.getType(Trace.class) })));
        insns.add(new Insn.ReturnValue());

        insns.add(new Insn.Mark(derefNonnull));
        return insns;
    }

    /**
     * Insns to push fake-call cse.
     */
    static List<Insn> pushFakeCall(int traceKey) {
        return List.of(
                LOAD_CALLSTACK,
                new Insn.InvokeDynamic(
                    MethodType.methodType(FakeCallTraceCse.class),
                    InsnsGenerator.BOOTSTRAP_FAKE_CALL_CSE,
                    List.of(traceKey)),
                new Insn.PushInt(0),
                new Insn.PushInt(0),
                new Insn.PushInt(0),
                INVOKE_PUSH_CSE);
    }

    // elems of vec or args {{{

    /**
     * Loads the current count of elems to the local var.
     */
    private List<Insn> loadCurElemCount(String countKey) {
        // #count = ((IntVal) dataStack.pop()).getInt()
        return List.of(
                LOAD_DATASTACK,
                new Insn.InvokeVirtual(DATASTACK_TYPE,
                    new Method("pop", Type.getType(Val.class), new Type[0])),
                new Insn.CheckCast(Type.getType(IntVal.class)),
                new Insn.InvokeVirtual(Type.getType(IntVal.class),
                    new Method("getInt", Type.INT_TYPE, new Type[0])),
                new Insn.StoreNewLocal(countKey, Type.INT_TYPE));
    }

    /**
     * Spreads the elems to datastack.
     */
    private List<Insn> spreadElems(ItreeElem.Spread spread) {
        // contParam = eval(spread.expr)
        List<Insn> insns = new ArrayList<>(generate(spread.expr(), ResultContext.NON_TAIL));

        // if (contParam instanceof VecVal) goto is-vec
        insns.add(LOAD_CONTPARAM);
        insns.add(new Insn.InstanceOf(Type.getType(VecVal.class)));
        String isVecLabel = keySup.newKeyStr("is-vec");
        insns.add(new Insn.IfNonZero(isVecLabel));

        // stackMachine.transitionToCannotSpread(trace, contParam); return
        insns.add(LOAD_STACKMACHINE);
        Insn traceInsn = produceTrace(
                Trace.of(new Location(programName, programText, spread.pos())));
        insns.add(traceInsn);
        insns.add(LOAD_CONTPARAM);
        insns.add(new Insn.InvokeVirtual(STACKMACHINE_TYPE,
                    new Method("transitionToCannotSpread",
                        Type.VOID_TYPE,
                        new Type[] { Type.getType(Trace.class), Type.getType(Val.class) })));
        insns.add(new Insn.ReturnValue());

        // :is-vec
        // #vi = ((VecVal) contParam).vecInternal()
        insns.add(new Insn.Mark(isVecLabel));
        insns.add(LOAD_CONTPARAM);
        insns.add(new Insn.CheckCast(Type.getType(VecVal.class)));
        insns.add(INVOKE_VEC_INTERNAL);
        String viKey = keySup.newKeyStr("vi");
        insns.add(new Insn.StoreNewLocal(viKey, Type.getType(VecInternal.class)));

        // #spreadSize = #vi.size()
        insns.add(new Insn.LoadLocal(viKey));
        insns.add(INVOKE_VEC_INTERNAL_SIZE);
        String spreadSize = keySup.newKeyStr("spreadSize");
        insns.add(new Insn.StoreNewLocal(spreadSize, Type.INT_TYPE));

        // if (dataStack.ensureCapaSpPlus(dataStackUsageUpperBound() + #spreadSize))
        //   goto ensured
        insns.add(LOAD_DATASTACK);
        insns.add(new Insn.LoadThis());
        insns.add(new Insn.InvokeVirtual(JavaClassIr.TYPE_BASE,
                    new Method("dataStackUsageUpperBound", Type.INT_TYPE, new Type[0])));
        insns.add(new Insn.LoadLocal(spreadSize));
        insns.add(new Insn.AddInt());
        insns.add(new Insn.InvokeVirtual(DATASTACK_TYPE,
                    new Method("ensureCapaSpPlus",
                        Type.BOOLEAN_TYPE,
                        new Type[] { Type.INT_TYPE })));
        String ensuredLabel = keySup.newKeyStr("ensured");
        insns.add(new Insn.IfNonZero(ensuredLabel));

        // stackMachine.transitionToRaiseOn(msg, trace); return
        insns.add(LOAD_STACKMACHINE);
        insns.add(new Insn.PushString("data stack overflow"));
        insns.add(traceInsn);
        insns.add(new Insn.InvokeVirtual(STACKMACHINE_TYPE,
                    new Method("transitionToRaiseOn",
                        Type.VOID_TYPE,
                        new Type[] { Type.getType(String.class), Type.getType(Trace.class) })));
        insns.add(new Insn.ReturnValue());


        // ensured:
        //   #count = ((IntVal) dataStack.pop()).getInt()
        insns.add(new Insn.Mark(ensuredLabel));
        String countKey = keySup.newKeyStr("count");
        insns.addAll(loadCurElemCount(countKey));

        // dataStack.pushAll(#vi #spreadSize)
        insns.add(LOAD_DATASTACK);
        insns.add(new Insn.LoadLocal(viKey));
        insns.add(new Insn.LoadLocal(spreadSize));
        insns.add(new Insn.InvokeVirtual(DATASTACK_TYPE,
                    new Method("pushAll",
                        Type.VOID_TYPE,
                        new Type[] { Type.getType(VecInternal.class), Type.INT_TYPE })));

        // dataStack.push(IntVal.of(vm, #count + #spreadSize))
        insns.add(LOAD_DATASTACK);
        insns.addAll(LOAD_VM);
        insns.add(new Insn.LoadLocal(countKey));
        insns.add(new Insn.LoadLocal(spreadSize));
        insns.add(new Insn.AddInt());
        insns.add(INVOKE_INT_VAL_OF);
        insns.add(INVOKE_PUSH_TO_DATASTACK);

        return insns;
    }

    /**
     * Pushes the single elem to datastack.
     */
    private List<Insn> pushSingle(Itree itree) {
        // contParam = eval(itree)
        List<Insn> insns = new ArrayList<>(generate(itree, ResultContext.NON_TAIL));

        // #count = ((IntVal) dataStack.pop()).getInt()
        String countKey = keySup.newKeyStr("count");
        insns.addAll(loadCurElemCount(countKey));

        // dataStack.push(contParam)
        insns.add(LOAD_DATASTACK);
        insns.add(LOAD_CONTPARAM);
        insns.add(INVOKE_PUSH_TO_DATASTACK);

        // dataStack.push(IntVal.of(vm, #count + 1))
        insns.add(LOAD_DATASTACK);
        insns.addAll(LOAD_VM);
        insns.add(new Insn.LoadLocal(countKey));
        insns.add(new Insn.PushInt(1));
        insns.add(new Insn.AddInt());
        insns.add(INVOKE_INT_VAL_OF);
        insns.add(INVOKE_PUSH_TO_DATASTACK);
        return insns;
    }

    /**
     * Pushes elems to datastack, including spreading.
     * The count of the elems is stored to elemCountKey.
     */
    private List<Insn> pushElemsWithSpread(List<ItreeElem> elems, String elemCountKey) {
        // dataStack.push(int(0))
        List<Insn> insns = new ArrayList<>(List.of(
                    LOAD_DATASTACK,
                    new Insn.InvokeDynamic(
                        MethodType.methodType(Val.class),
                        BOOTSTRAP_INT_HANDLE,
                        List.of(0)),
                    INVOKE_PUSH_TO_DATASTACK));

        for (ItreeElem elem : elems) {
            insns.addAll(elem instanceof ItreeElem.Spread spread
                    ? spreadElems(spread)
                    : pushSingle((Itree) elem));
        }

        // #elemCount = ((IntVal) dataStack.pop()).getInt();
        insns.add(LOAD_DATASTACK);
        insns.add(INVOKE_POP_FROM_DATASTACK);
        insns.add(new Insn.CheckCast(Type.getType(IntVal.class)));
        insns.add(new Insn.InvokeVirtual(
                    Type.getType(IntVal.class),
                    new Method("getInt", Type.INT_TYPE, new Type[0])));
        insns.add(new Insn.StoreNewLocal(elemCountKey, Type.INT_TYPE));

        return insns;
    }

    /**
     * Pushes elems to datastack, without spreading.
     * The count of the elems is stored to elemCountKey.
     */
    private List<Insn> pushElemsAllSingle(List<Itree> elems, String elemCountKey) {
        List<Insn> insns = new ArrayList<>();
        for (Itree elem: elems) {
            insns.addAll(generate(elem, ResultContext.NON_TAIL));
            insns.add(LOAD_DATASTACK);
            insns.add(LOAD_CONTPARAM);
            insns.add(INVOKE_PUSH_TO_DATASTACK);
        }

        // #elemCount = size of elems
        insns.add(new Insn.PushInt(elems.size()));
        insns.add(new Insn.StoreNewLocal(elemCountKey, Type.INT_TYPE));
        return insns;
    }

    /**
     * Pushes elems to datastack.
     * The count of the elems is stored to elemCountKey.
     */
    private List<Insn> pushElems(List<ItreeElem> elems, String elemCountKey) {
        boolean allSingle = elems.stream().allMatch(e -> e instanceof Itree);
        return allSingle
            ? pushElemsAllSingle(elems.stream().map(e -> (Itree) e).toList(), elemCountKey)
            : pushElemsWithSpread(elems, elemCountKey);
    }

    /**
     * Whether elems is a singleton tuple.
     */
    private boolean isSingleton(List<ItreeElem> elems) {
        return elems.size() == 1
            && elems.get(0) instanceof Itree;
    }

    // }}}

    // vec {{{

    /** Insns of {@code contParam = vm.vec.of()}. */
    private static final List<Insn> MAKE_EMPTY_VEC = List.of(
            new Insn.LoadThis(),
            new Insn.GetField(JavaClassIr.TYPE_BASE, "vm", Type.getType(Vm.class)),
            LOAD_VEC_HELPER,
            INVOKE_VEC_OF_EMPTY,
            STORE_CONTPARAM);

    /**
     * Makes a singleton vec and stores it to contParam.
     */
    private List<Insn> makeSingletonVec(Itree elem) {
        List<Insn> insns = new ArrayList<>(generate(elem, ResultContext.NON_TAIL));

        // contParam = vm.vec.of(contParam)
        insns.addAll(LOAD_VM);
        insns.add(LOAD_VEC_HELPER);
        insns.add(LOAD_CONTPARAM);
        insns.add(INVOKE_VEC_OF_SINGLE);
        insns.add(STORE_CONTPARAM);
        return insns;
    }

    /**
     * Makes a vec using datastack and stores it to contParam.
     */
    private List<Insn> makeVecOnDataStack(List<ItreeElem> elems) {
        String elemCountKey = keySup.newKeyStr("elemCount");
        List<Insn> insns = new ArrayList<>(pushElems(elems, elemCountKey));

        // elemsOffset = dataStack.getTopOffset() - elemCount
        insns.add(LOAD_DATASTACK);
        insns.add(INVOKE_TOP_OFFSET);
        insns.add(new Insn.LoadLocal(elemCountKey));
        insns.add(new Insn.SubInt());
        String elemsOffsetKey = keySup.newKeyStr("elemsOffset");
        insns.add(new Insn.StoreNewLocal(elemsOffsetKey, Type.INT_TYPE));

        // contParam = dataStack.makeVecFromOffset(elemsOffset)
        insns.add(LOAD_DATASTACK);
        insns.add(new Insn.LoadLocal(elemsOffsetKey));
        insns.add(new Insn.InvokeVirtual(DATASTACK_TYPE,
                    new Method("makeVecFromOffset",
                        Type.getType(VecVal.class),
                        new Type[] { Type.INT_TYPE })));
        insns.add(STORE_CONTPARAM);

        // dataStack.removeFromOffset(elemsOffset)
        insns.add(LOAD_DATASTACK);
        insns.add(new Insn.LoadLocal(elemsOffsetKey));
        insns.add(INVOKE_REMOVE_FROM_OFFSET);
        return insns;
    }

    // }}}

    /**
     * Visits an itree to generate code for.
     */
    private final class Visitor implements ItreeVisitor<List<Insn>> {

        /** The result context. */
        private final ResultContext resultCtx;

        /**
         * Constructs a visitor.
         */
        Visitor(ResultContext resultCtx) {
            this.resultCtx = resultCtx;
        }

        /**
         * Makes a trace of a call at the place.
         */
        private Trace makeCallTrace(String sym, int pos) {
            int symHandle = vm.sym.handleFor(sym);
            Location location = new Location(programName, programText, pos);
            return resultCtx.onTailOrNot(Trace.of(symHandle, location));
        }

        /**
         * Pushes args and does a call.
         */
        private List<Insn> pushArgsAndCall(
                List<ItreeElem> args,
                Trace trace,
                List<Insn> callFunEpilogue) {
            String argCount = keySup.newKeyStr("argCount");
            List<Insn> insns = new ArrayList<>(pushElems(args, argCount));
            insns.addAll(doCall(trace, callFunEpilogue, argCount));
            return insns;
        }

        /**
         * Does call.
         */
        private List<Insn> doCall(Trace trace, List<Insn> callFunEpilogue, String argCount) {
            if (resultCtx.equals(ResultContext.TAIL)) {
                return List.of(
                        // stackMachine.tailCallFun(trace, #argCount)
                        LOAD_STACKMACHINE,
                        produceTrace(trace),
                        new Insn.LoadLocal(argCount),
                        new Insn.InvokeVirtual(STACKMACHINE_TYPE,
                            new Method("tailCallFun",
                                Type.VOID_TYPE,
                                new Type[] { Type.getType(Trace.class), Type.INT_TYPE })),

                        // return
                        new Insn.ReturnValue());
            } else {
                List<Insn> insns = new ArrayList<>(10);
                // stackMachine.callFun(this, resumePc, #argCount)
                insns.add(LOAD_STACKMACHINE);
                insns.add(new Insn.LoadThis());
                int resumePc = pcSup.newProgramCounter(trace);
                insns.add(new Insn.PushInt(resumePc));
                insns.add(new Insn.LoadLocal(argCount));
                insns.add(INVOKE_CALL_FUN);

                // return
                insns.add(new Insn.ReturnValue());

                // case pc=resumePc
                insns.add(new Insn.Case(PROGRAMCOUNTER_KEY, resumePc));

                // insns specific to callFun
                insns.addAll(callFunEpilogue);
                return insns;
            }
        }

        @Override
        public List<Insn> visit(NadaItree itree) {
            List<Insn> insns = new ArrayList<>(List.of(PRODUCE_NADA, STORE_CONTPARAM));
            insns.addAll(resultCtx.returnOnTail());
            return insns;
        }

        @Override
        public List<Insn> visit(StrItree itree) {
            List<Insn> insns = new ArrayList<>(List.of(
                        new Insn.InvokeDynamic(
                            MethodType.methodType(Val.class),
                            BOOTSTRAP_STR_HANDLE,
                            List.of(itree.str())),
                        STORE_CONTPARAM));
            insns.addAll(resultCtx.returnOnTail());
            return insns;
        }

        @Override
        public List<Insn> visit(NumItree itree) {
            List<Insn> insns = new ArrayList<>(List.of(
                        new Insn.InvokeDynamic(
                            MethodType.methodType(Val.class),
                            BOOTSTRAP_NUM_HANDLE,
                            List.of(itree.num().toString())),
                        STORE_CONTPARAM));
            insns.addAll(resultCtx.returnOnTail());
            return insns;
        }

        @Override
        public List<Insn> visit(RecvItree itree) {
            List<Insn> insns = new ArrayList<>(List.of(
                        LOAD_DATASTACK,
                    INVOKE_RECV,
                        STORE_CONTPARAM));
            insns.addAll(resultCtx.returnOnTail());
            return insns;
        }

        @Override
        public List<Insn> visit(ArgVecItree itree) {
            List<Insn> insns = new ArrayList<>(List.of(
                        LOAD_DATASTACK,
                        LOAD_ARGCOUNT,
                    INVOKE_ARG_VEC,
                        STORE_CONTPARAM));
            insns.addAll(resultCtx.returnOnTail());
            return insns;
        }

        @Override
        public List<Insn> visit(BindingItree itree) {
            List<Insn> insns = new ArrayList<>(bindingGen.generateBinding());
            insns.add(STORE_CONTPARAM);
            insns.addAll(resultCtx.returnOnTail());
            return insns;
        }

        @Override
        public List<Insn> visit(FastFunItree itree) {
            List<Insn> insns = new ArrayList<>(makeFastFunGen.makeFun(itree));
            insns.add(STORE_CONTPARAM);
            insns.addAll(resultCtx.returnOnTail());
            return insns;
        }

        @Override
        public List<Insn> visit(SlowFunItree itree) {
            SlowFunCompiler compiler = new SlowFunCompiler(vm, programName, programText);
            int jcirInd = jcirAccum.add(() -> compiler.compile(itree));
            // make-fun[jcirInd](vm, binding)
            List<Insn> insns = new ArrayList<>(List.of(
                        new Insn.LoadThis(),
                        new Insn.GetField(JavaClassIr.TYPE_BASE, "vm", Type.getType(Vm.class))));
            insns.addAll(bindingGen.generateBinding());
            // FIXME odd; make it a method of InsnsGenerator
            insns.add(MakeFastFunGenerator.invokeMakeFun(1, jcirInd));
            insns.add(STORE_CONTPARAM);
            insns.addAll(resultCtx.returnOnTail());
            return insns;
        }

        @Override
        public List<Insn> visit(LderefItree itree) {
            var location = new Location(programName, programText, itree.pos());
            List<Insn> insns = new ArrayList<>(lvarAccGen.loadLvar(itree.lvar(), location));
            insns.addAll(resultCtx.returnOnTail());
            return insns;
        }

        @Override
        public List<Insn> visit(LstoreItree itree) {
            List<Insn> insns = new ArrayList<>();
            if (itree.rhs() instanceof RecvItree) {
                insns.addAll(lvarAccGen.passRecv(itree.lvar()));
            } else {
                insns.addAll(generate(itree.rhs(), ResultContext.NON_TAIL));
                if (! lvarAccGen.isUnused(itree.lvar())) {
                    insns.addAll(lvarAccGen.storeLvar(itree.lvar()));
                }
            }

            insns.add(PRODUCE_NADA);
            insns.add(STORE_CONTPARAM);
            insns.addAll(resultCtx.returnOnTail());
            return insns;
        }

        @Override
        public List<Insn> visit(DerefItree itree) {
            List<Insn> insns = new ArrayList<>(generate(itree.owner(), ResultContext.NON_TAIL));

            // Val derefOwner = contParam
            insns.add(LOAD_CONTPARAM);
            String ownerVar = keySup.newKeyStr("derefOwner");
            insns.add(new Insn.StoreNewLocal(ownerVar, Type.getType(Val.class)));

            // contParam = derefOwner.getVar()
            insns.addAll(deref(ownerVar, itree.sym(), itree.pos()));

            insns.addAll(resultCtx.returnOnTail());
            return insns;
        }

        @Override
        public List<Insn> visit(StoreItree itree) {
            // contParam = owner
            List<Insn> insns = new ArrayList<>(generate(itree.owner(), ResultContext.NON_TAIL));

            // dataStack.push(contParam)
            insns.add(LOAD_DATASTACK);
            insns.add(LOAD_CONTPARAM);
            insns.add(INVOKE_PUSH_TO_DATASTACK);

            // contParam = rhs
            insns.addAll(generate(itree.rhs(), ResultContext.NON_TAIL));

            // set-var[sym](dataStack.pop(), contParam)
            insns.add(LOAD_DATASTACK);
            insns.add(new Insn.InvokeVirtual(
                        DATASTACK_TYPE,
                        new Method("pop", Type.getType(Val.class), new Type[0])));
            insns.add(LOAD_CONTPARAM);
            insns.add(new Insn.InvokeDynamic(
                        MethodType.methodType(void.class, Val.class, Val.class),
                        BOOTSTRAP_SET_VAR_HANDLE,
                        List.of(vm.sym.handleFor(itree.sym()))));

            // contParam = nada
            insns.add(PRODUCE_NADA);
            insns.add(STORE_CONTPARAM);

            insns.addAll(resultCtx.returnOnTail());
            return insns;
        }

        @Override
        public List<Insn> visit(VarrefItree itree) {
            // contParam = owner
            List<Insn> insns = new ArrayList<>(generate(itree.owner(), ResultContext.NON_TAIL));

            // contParam = vm.varref.of(contParam, symHandle)
            insns.addAll(LOAD_VM);
            insns.add(new Insn.GetField(Type.getType(Vm.class),
                        "varref", Type.getType(VarrefHelper.class)));
            insns.add(LOAD_CONTPARAM);
            insns.add(new Insn.PushInt(vm.sym.handleFor(itree.sym())));
            insns.add(new Insn.InvokeVirtual(Type.getType(VarrefHelper.class),
                        new Method("of", Type.getType(VarrefVal.class),
                            new Type[] { Type.getType(Val.class), Type.INT_TYPE })));
            insns.add(STORE_CONTPARAM);

            insns.addAll(resultCtx.returnOnTail());
            return insns;
        }

        @Override
        public List<Insn> visit(LetRecItree itree) {
            List<Insn> insns = new ArrayList<>(
                    letRecGen.letRec(itree, InsnsGenerator.this::generate));
            insns.add(PRODUCE_NADA);
            insns.add(STORE_CONTPARAM);

            insns.addAll(resultCtx.returnOnTail());
            return insns;
        }

        @Override
        public List<Insn> visit(SeqItree itree) {
            List<Insn> insns = new ArrayList<>();
            itree.getLeadingSteps().stream()
                .map(step -> generate(step, ResultContext.NON_TAIL))
                .forEach(insns::addAll);
            insns.addAll(generate(itree.getLastStep(), this.resultCtx));
            return insns;
        }

        /**
         * Loads rhs.
         *
         * @param callTrace the trace of op_store.
         * @param rhsVec the lvar key str of RHS vec.
         * @param vecInternal the lvar key str of VecInternal of RHS.
         * @param size the lvar key str of size of RHS.
         */
        private List<Insn> loadRhsVecWithSize(
                int callTrace, String rhsVec, String vecInternal, String size) {
            List<Insn> insns = new ArrayList<>(20);

            // if contParam instanceof VecVal; goto #is-vec
            insns.add(LOAD_CONTPARAM);
            insns.add(new Insn.InstanceOf(Type.getType(VecVal.class)));
            String isVec = keySup.newKeyStr("is-vec");
            insns.add(new Insn.IfNonZero(isVec));

            // push trace
            insns.addAll(pushFakeCall(callTrace));

            // stackMachine.raiseNotVecRhs(contParam)
            insns.add(LOAD_STACKMACHINE);
            insns.add(LOAD_CONTPARAM);
            insns.add(INVOKE_RAISE_NOT_VEC_RHS);
            insns.add(new Insn.ReturnValue());

            // #is-vec:
            insns.add(new Insn.Mark(isVec));

            // #rhsVec = (VecVal) contParam
            insns.add(InsnsGenerator.LOAD_CONTPARAM);
            insns.add(new Insn.CheckCast(Type.getType(VecVal.class)));
            insns.add(new Insn.StoreNewLocal(rhsVec, Type.getType(VecVal.class)));

            // #vecInternal = #rhsVec.vecInternal()
            insns.add(new Insn.LoadLocal(rhsVec));
            insns.add(INVOKE_VEC_INTERNAL);
            insns.add(new Insn.StoreNewLocal(vecInternal, Type.getType(VecInternal.class)));

            // #size = #vecInternal.size()
            insns.add(new Insn.LoadLocal(vecInternal));
            insns.add(INVOKE_VEC_INTERNAL_SIZE);
            insns.add(new Insn.StoreNewLocal(size, Type.INT_TYPE));

            return insns;
        }

        /**
         * Pass mandatory args.
         */
        private List<Insn> passMandatoryArgs(List<LocalVar> lvars, String vecInternal) {
            List<Insn> insns = new ArrayList<>();
            for (int mandatoryInd = 0; mandatoryInd < lvars.size(); ++ mandatoryInd) {
                LocalVar lvar = lvars.get(mandatoryInd);

                if (! lvarAccGen.isUnused(lvar)) {
                    // contParam = #vecInternal.get(#mandatoryInd)
                    insns.add(new Insn.LoadLocal(vecInternal));
                    insns.add(new Insn.PushInt(mandatoryInd));
                    insns.add(INVOKE_VEC_INTERNAL_GET);
                    insns.add(STORE_CONTPARAM);

                    // store val
                    insns.addAll(lvarAccGen.storeLvar(lvar));
                }
            }
            return insns;
        }

        @Override
        public List<Insn> visit(OptVecAssignmentItree itree) {
            // contParam = rhs
            List<Insn> insns = new ArrayList<>(generate(itree.rhs(), ResultContext.NON_TAIL));

            // #rhsVec = (VecVal) contParam, #vecInternal = VecInternal, #size = size of rhs
            int callTrace = traceAccum.add(makeCallTrace("op_store", itree.pos()));
            String rhsVec = keySup.newKeyStr("rhsVec");
            String vecInternal = keySup.newKeyStr("vecInternal");
            String size = keySup.newKeyStr("size");
            insns.addAll(loadRhsVecWithSize(callTrace, rhsVec, vecInternal, size));

            // if #size < min or #size > max; goto #wrong-arity
            insns.add(new Insn.LoadLocal(size));
            insns.add(new Insn.PushInt(itree.mandatory().size()));
            String wrongArity = keySup.newKeyStr("wrong-arity");
            insns.add(new Insn.IfLtInt(wrongArity));
            insns.add(new Insn.LoadLocal(size));
            insns.add(new Insn.PushInt(itree.mandatory().size() + itree.opt().size()));
            insns.add(new Insn.IfGtInt(wrongArity));

            // goto #valid-arity:
            String validArity = keySup.newKeyStr("valid-arity");
            insns.add(new Insn.GoTo(validArity));

            // #wrong-arity:
            insns.add(new Insn.Mark(wrongArity));

            // push trace
            insns.addAll(pushFakeCall(callTrace));

            // stackMachine.raiseWrongNumberOfArgs(min, "...", #rhsVec); return
            insns.add(LOAD_STACKMACHINE);
            insns.add(new Insn.PushInt(itree.mandatory().size()));
            insns.add(new Insn.PushString(itree.lhsRepr()));
            insns.add(new Insn.LoadLocal(rhsVec));
            insns.add(INVOKE_RAISE_WRONG_NUMBER_OF_ARGS);
            insns.add(new Insn.ReturnValue());

            // #valid-arity
            insns.add(new Insn.Mark(validArity));

            // pass mandatory args
            insns.addAll(passMandatoryArgs(itree.mandatory(), vecInternal));

            for (int optInd = 0; optInd < itree.opt().size(); ++ optInd) {
                LocalVar lvar = itree.opt().get(optInd);
                int argInd = optInd + itree.mandatory().size();

                if (! lvarAccGen.isUnused(lvar)) {
                    // if #size > argInd; goto #opt-given-{argInd}
                    insns.add(new Insn.LoadLocal(size));
                    insns.add(new Insn.PushInt(argInd));
                    String optGiven = keySup.newKeyStr("opt-given-" + argInd);
                    insns.add(new Insn.IfGtInt(optGiven));

                    // _ = vm.vec.of(); goto #endif-opt-2
                    insns.addAll(LOAD_VM);
                    insns.add(LOAD_VEC_HELPER);
                    insns.add(INVOKE_VEC_OF_EMPTY);
                    String endifOpt = keySup.newKeyStr("endif-opt-" + argInd);
                    insns.add(new Insn.GoTo(endifOpt));

                    // #opt-given-{argInd}:
                    // _ = vm.vec.of(#vecInternal.get(argInd))
                    insns.add(new Insn.Mark(optGiven));
                    insns.addAll(LOAD_VM);
                    insns.add(LOAD_VEC_HELPER);
                    insns.add(new Insn.LoadLocal(vecInternal));
                    insns.add(new Insn.PushInt(argInd));
                    insns.add(INVOKE_VEC_INTERNAL_GET);
                    insns.add(INVOKE_VEC_OF_SINGLE);

                    // #endif-opt-{argInd}:
                    // contParam = _
                    // store lvar
                    insns.add(new Insn.Mark(endifOpt));
                    insns.add(STORE_CONTPARAM);
                    insns.addAll(lvarAccGen.storeLvar(lvar));
                }
            }

            insns.add(PRODUCE_NADA);
            insns.add(STORE_CONTPARAM);
            insns.addAll(resultCtx.returnOnTail());
            return insns;
        }

        @Override
        public List<Insn> visit(OptRestVecAssignmentItree itree) {
            // contParam = rhs
            List<Insn> insns = new ArrayList<>(generate(itree.rhs(), ResultContext.NON_TAIL));

            // #rhsVec = (VecVal) contParam, #vecInternal = VecInternal, #size = size of rhs
            int callTrace = traceAccum.add(makeCallTrace("op_store", itree.pos()));
            String rhsVec = keySup.newKeyStr("rhsVec");
            String vecInternal = keySup.newKeyStr("vecInternal");
            String size = keySup.newKeyStr("size");
            insns.addAll(loadRhsVecWithSize(callTrace, rhsVec, vecInternal, size));

            // if #size >= min; goto #valid-arity
            insns.add(new Insn.LoadLocal(size));
            insns.add(new Insn.PushInt(itree.mandatory().size()));
            String validArity = keySup.newKeyStr("valid-arity");
            insns.add(new Insn.IfGeInt(validArity));

            // push trace
            insns.addAll(pushFakeCall(callTrace));

            // stackMachine.raiseWrongNumberOfArgs(min, "...", #rhsVec); return
            insns.add(LOAD_STACKMACHINE);
            insns.add(new Insn.PushInt(itree.mandatory().size()));
            insns.add(new Insn.PushString(itree.lhsRepr()));
            insns.add(new Insn.LoadLocal(rhsVec));
            insns.add(INVOKE_RAISE_WRONG_NUMBER_OF_ARGS);
            insns.add(new Insn.ReturnValue());

            // #valid-arity:
            insns.add(new Insn.Mark(validArity));

            // pass mandatory args
            insns.addAll(passMandatoryArgs(itree.mandatory(), vecInternal));

            for (int optInd = 0; optInd < itree.opt().size(); ++ optInd) {
                LocalVar lvar = itree.opt().get(optInd);
                int argInd = optInd + itree.mandatory().size();

                if (! lvarAccGen.isUnused(lvar)) {
                    // if #size > argInd; goto #opt-given-{argInd}
                    insns.add(new Insn.LoadLocal(size));
                    insns.add(new Insn.PushInt(argInd));
                    String optGiven = keySup.newKeyStr("opt-given-" + argInd);
                    insns.add(new Insn.IfGtInt(optGiven));

                    // _ = vm.vec.of(); goto #endif-opt-2
                    insns.addAll(LOAD_VM);
                    insns.add(LOAD_VEC_HELPER);
                    insns.add(INVOKE_VEC_OF_EMPTY);
                    String endifOpt = keySup.newKeyStr("endif-opt-" + argInd);
                    insns.add(new Insn.GoTo(endifOpt));

                    // #opt-given-{argInd}:
                    // _ = vm.vec.of(#vecInternal.get(argInd))
                    insns.add(new Insn.Mark(optGiven));
                    insns.addAll(LOAD_VM);
                    insns.add(LOAD_VEC_HELPER);
                    insns.add(new Insn.LoadLocal(vecInternal));
                    insns.add(new Insn.PushInt(argInd));
                    insns.add(INVOKE_VEC_INTERNAL_GET);
                    insns.add(INVOKE_VEC_OF_SINGLE);

                    // #endif-opt-{argInd}:
                    // contParam = _
                    // store lvar
                    insns.add(new Insn.Mark(endifOpt));
                    insns.add(STORE_CONTPARAM);
                    insns.addAll(lvarAccGen.storeLvar(lvar));
                }
            }

            if (! lvarAccGen.isUnused(itree.rest())) {
                // if #size > #mandatory + #opt; goto #rest-given
                insns.add(new Insn.LoadLocal(size));
                insns.add(new Insn.PushInt(itree.mandatory().size() + itree.opt().size()));
                String restGiven = keySup.newKeyStr("rest-given");
                insns.add(new Insn.IfGtInt(restGiven));

                // _ = vm.vec.of(); goto #endif-rest
                insns.addAll(LOAD_VM);
                insns.add(LOAD_VEC_HELPER);
                insns.add(INVOKE_VEC_OF_EMPTY);
                String endifRest = keySup.newKeyStr("endif-rest");
                insns.add(new Insn.GoTo(endifRest));

                // #rest-given:
                // _ = new VecVal(vm, #vecInternal.copyRange(#mandatory + #opt, #size))
                insns.add(new Insn.Mark(restGiven));
                insns.add(new Insn.NewInstance(Type.getType(VecVal.class)));
                insns.add(new Insn.Dup());
                insns.addAll(LOAD_VM);
                insns.add(new Insn.LoadLocal(vecInternal));
                insns.add(new Insn.PushInt(itree.mandatory().size() + itree.opt().size()));
                insns.add(new Insn.LoadLocal(size));
                insns.add(new Insn.InvokeVirtual(Type.getType(VecInternal.class),
                            new Method("copyRange", Type.getType(VecInternal.class),
                                new Type[] { Type.INT_TYPE, Type.INT_TYPE })));
                insns.add(INVOKE_VEC_CONSTRUCTOR);

                // #endif-rest:
                // contParam = _
                // store contParam to rest
                insns.add(new Insn.Mark(endifRest));
                insns.add(STORE_CONTPARAM);
                insns.addAll(lvarAccGen.storeLvar(itree.rest()));
            }

            insns.add(PRODUCE_NADA);
            insns.add(STORE_CONTPARAM);
            insns.addAll(resultCtx.returnOnTail());
            return insns;
        }

        @Override
        public List<Insn> visit(RestVecAssignmentItree itree) {
            // contParam = rhs
            List<Insn> insns = new ArrayList<>(generate(itree.rhs(), ResultContext.NON_TAIL));

            // if contParam instanceof VecVal; goto #is-vec
            insns.add(LOAD_CONTPARAM);
            insns.add(new Insn.InstanceOf(Type.getType(VecVal.class)));
            String isVec = keySup.newKeyStr("is-vec");
            insns.add(new Insn.IfNonZero(isVec));

            // push trace
            insns.addAll(pushFakeCall(traceAccum.add(makeCallTrace("op_store", itree.pos()))));

            // stackMachine.raiseNotVecRhs(contParam)
            insns.add(LOAD_STACKMACHINE);
            insns.add(LOAD_CONTPARAM);
            insns.add(INVOKE_RAISE_NOT_VEC_RHS);
            insns.add(new Insn.ReturnValue());

            // #is-vec:
            insns.add(new Insn.Mark(isVec));

            if (! lvarAccGen.isUnused(itree.lvar())) {
                // contParam = new VecVal(vm, ((VecVal) contParam).vecInternal().copy())
                insns.add(new Insn.NewInstance(Type.getType(VecVal.class)));
                insns.add(new Insn.Dup());
                insns.addAll(LOAD_VM);
                insns.add(LOAD_CONTPARAM);
                insns.add(new Insn.CheckCast(Type.getType(VecVal.class)));
                insns.add(INVOKE_VEC_INTERNAL);
                insns.add(new Insn.InvokeVirtual(Type.getType(VecInternal.class),
                            new Method("copy", Type.getType(VecInternal.class), new Type[0])));
                insns.add(INVOKE_VEC_CONSTRUCTOR);
                insns.add(STORE_CONTPARAM);

                // store contParam to lhs
                insns.addAll(lvarAccGen.storeLvar(itree.lvar()));
            }

            insns.add(PRODUCE_NADA);
            insns.add(STORE_CONTPARAM);
            insns.addAll(resultCtx.returnOnTail());
            return insns;
        }

        @Override
        public List<Insn> visit(NestedVecAssignmentItree itree) {
            // contParam = rhs
            List<Insn> insns = new ArrayList<>(generate(itree.rhs(), ResultContext.NON_TAIL));

            // #rhs = contParam
            insns.add(LOAD_CONTPARAM);
            String rhs = keySup.newKeyStr("rhs");
            insns.add(new Insn.StoreNewLocal(rhs, Type.getType(Val.class)));

            // if #rhs instanceof VecVal; goto #is-vec-rhs
            insns.add(new Insn.LoadLocal(rhs));
            insns.add(new Insn.InstanceOf(Type.getType(VecVal.class)));
            String isVecRhs = keySup.newKeyStr("is-vec-rhs");
            insns.add(new Insn.IfNonZero(isVecRhs));

            // push call trace
            int callTraceKey = traceAccum.add(makeCallTrace("op_store", itree.pos()));
            insns.addAll(pushFakeCall(callTraceKey));

            // stackMachine.raiseNotVecRhs(#rhs); return
            insns.add(LOAD_STACKMACHINE);
            insns.add(new Insn.LoadLocal(rhs));
            insns.add(INVOKE_RAISE_NOT_VEC_RHS);
            insns.add(new Insn.ReturnValue());

            // #is-vec-rhs:
            insns.add(new Insn.Mark(isVecRhs));

            // #rhsVec = (VecVal) #rhs
            insns.add(new Insn.LoadLocal(rhs));
            insns.add(new Insn.CheckCast(Type.getType(VecVal.class)));
            String rhsVec = keySup.newKeyStr("rhsVec");
            insns.add(new Insn.StoreNewLocal(rhsVec, Type.getType(VecVal.class)));

            // #rhsInternal = #rhsVec.vecInternal()
            insns.add(new Insn.LoadLocal(rhsVec));
            insns.add(INVOKE_VEC_INTERNAL);
            String rhsInternal = keySup.newKeyStr("rhsInternal");
            insns.add(new Insn.StoreNewLocal(rhsInternal, Type.getType(VecInternal.class)));

            // if #rhsInternal.size() == expected; goto #right-sized-rhs
            insns.add(new Insn.LoadLocal(rhsInternal));
            insns.add(INVOKE_VEC_INTERNAL_SIZE);
            insns.add(new Insn.PushInt(itree.params().size()));
            String rightSizedRhs = keySup.newKeyStr("right-sized-rhs");
            insns.add(new Insn.IfEqInt(rightSizedRhs));

            // push call trace
            insns.addAll(pushFakeCall(callTraceKey));

            // stackMachine.raiseWrongNumberOfArgs(expected, "...", #rhsVec); return
            insns.add(LOAD_STACKMACHINE);
            insns.add(new Insn.PushInt(itree.params().size()));
            insns.add(new Insn.PushString(itree.lhsRepr()));
            insns.add(new Insn.LoadLocal(rhsVec));
            insns.add(INVOKE_RAISE_WRONG_NUMBER_OF_ARGS);
            insns.add(new Insn.ReturnValue());

            // #right-sized-rhs:
            insns.add(new Insn.Mark(rightSizedRhs));

            for (int i = 0; i < itree.params().size(); ++ i) {
                NestedParam param = itree.params().get(i);
                insns.addAll(param instanceof LocalVar lvar
                        ? storeVecElem(lvar, rhsInternal, i)
                        : passTupleVecElem(
                            (NestedParam.Tuple) param, rhsInternal, i, callTraceKey));
            }

            insns.add(PRODUCE_NADA);
            insns.add(STORE_CONTPARAM);
            insns.addAll(resultCtx.returnOnTail());
            return insns;
        }

        /**
         * Stores vec elem to lvar.
         */
        private List<Insn> storeVecElem(LocalVar lvar, String vecInternal, int i) {
            if (lvarAccGen.isUnused(lvar)) {
                return List.of();
            }

            List<Insn> insns = new ArrayList<>(10);
            // contParam = #vecInternal.get(i)
            insns.add(new Insn.LoadLocal(vecInternal));
            insns.add(new Insn.PushInt(i));
            insns.add(INVOKE_VEC_INTERNAL_GET);
            insns.add(STORE_CONTPARAM);

            // store param
            insns.addAll(lvarAccGen.storeLvar(lvar));
            return insns;
        }

        /**
         * Passes vec elem to tuple.
         */
        private List<Insn> passTupleVecElem(
                NestedParam.Tuple tupleParam, String vecInternal, int i, int callTraceKey) {
            List<Insn> insns = new ArrayList<>();
            // #tupleExpected = #rhsInternal.get(i)
            insns.add(new Insn.LoadLocal(vecInternal));
            insns.add(new Insn.PushInt(i));
            insns.add(INVOKE_VEC_INTERNAL_GET);
            String tupleExpected = keySup.newKeyStr("tupleExpected");
            insns.add(new Insn.StoreNewLocal(tupleExpected, Type.getType(Val.class)));
            insns.addAll(passTuple(tupleParam, tupleExpected, callTraceKey));
            return insns;
        }

        @Override
        public List<Insn> visit(VarrefVecAssignmentItree itree) {
            List<Insn> insns = new ArrayList<>();

            // push owners
            itree.params().stream()
                .filter(p -> p instanceof GenericVar)
                .map(p -> (GenericVar) p)
                .forEach(gvar -> {
                    insns.addAll(generate(gvar.owner(), ResultContext.NON_TAIL));
                    insns.add(LOAD_DATASTACK);
                    insns.add(LOAD_CONTPARAM);
                    insns.add(INVOKE_PUSH_TO_DATASTACK);
                });

            // contParam = rhs
            insns.addAll(generate(itree.rhs(), ResultContext.NON_TAIL));

            // if contParam instanceof VecVal; goto #is-vec
            insns.add(LOAD_CONTPARAM);
            insns.add(new Insn.InstanceOf(Type.getType(VecVal.class)));
            String isVec = keySup.newKeyStr("is-vec");
            insns.add(new Insn.IfNonZero(isVec));

            // push trace
            int callTraceKey = traceAccum.add(makeCallTrace("op_store", itree.pos()));
            insns.addAll(pushFakeCall(callTraceKey));

            // stackMachine.raiseNotVecRhs(contParam); return
            insns.add(LOAD_STACKMACHINE);
            insns.add(LOAD_CONTPARAM);
            insns.add(INVOKE_RAISE_NOT_VEC_RHS);
            insns.add(new Insn.ReturnValue());

            // #is-vec:
            insns.add(new Insn.Mark(isVec));

            // #rhsVec = (VecVal) contParam
            insns.add(LOAD_CONTPARAM);
            insns.add(new Insn.CheckCast(Type.getType(VecVal.class)));
            String rhsVec = keySup.newKeyStr("rhsVec");
            insns.add(new Insn.StoreNewLocal(rhsVec, Type.getType(VecVal.class)));

            // #vecInternal = #rhsVec.vecInternal()
            insns.add(new Insn.LoadLocal(rhsVec));
            insns.add(INVOKE_VEC_INTERNAL);
            String vecInternal = keySup.newKeyStr("vecInternal");
            insns.add(new Insn.StoreNewLocal(vecInternal, Type.getType(VecInternal.class)));

            // if #vecInternal.size() == expected; goto #valid
            insns.add(new Insn.LoadLocal(vecInternal));
            insns.add(INVOKE_VEC_INTERNAL_SIZE);
            insns.add(new Insn.PushInt(itree.params().size()));
            String valid = keySup.newKeyStr("valid");
            insns.add(new Insn.IfEqInt(valid));

            // push trace
            insns.addAll(pushFakeCall(callTraceKey));

            // stackMachine.raiseWrongNumberOfArgs(expected, "...", #rhsVec); return
            insns.add(LOAD_STACKMACHINE);
            insns.add(new Insn.PushInt(itree.params().size()));
            insns.add(new Insn.PushString(itree.lhsRepr()));
            insns.add(new Insn.LoadLocal(rhsVec));
            insns.add(INVOKE_RAISE_WRONG_NUMBER_OF_ARGS);
            insns.add(new Insn.ReturnValue());

            // #valid:
            insns.add(new Insn.Mark(valid));

            // #topOffset = dataStack.getTopOffset()
            insns.add(LOAD_DATASTACK);
            insns.add(INVOKE_TOP_OFFSET);
            String topOffset = keySup.newKeyStr("topOffset");
            insns.add(new Insn.StoreNewLocal(topOffset, Type.INT_TYPE));

            int gvarCount = (int) itree.params().stream()
                .filter(p -> p instanceof GenericVar)
                .count();
            int storedGvars = 0;
            for (int i = 0; i < itree.params().size(); ++ i) {
                VarrefParam param = itree.params().get(i);
                if (param instanceof GenericVar gvar) {
                    // _ = dataStack.getAtOffset(#topOffset - (gvarCount - storedGvars))
                    insns.add(LOAD_DATASTACK);
                    insns.add(new Insn.LoadLocal(topOffset));
                    insns.add(new Insn.PushInt(gvarCount - storedGvars));
                    insns.add(new Insn.SubInt());
                    insns.add(INVOKE_AT_OFFSET);

                    // store[var](_, #vecInternal.get(i))
                    insns.add(new Insn.LoadLocal(vecInternal));
                    insns.add(new Insn.PushInt(i));
                    insns.add(INVOKE_VEC_INTERNAL_GET);
                    insns.add(new Insn.InvokeDynamic(
                                MethodType.methodType(void.class, Val.class, Val.class),
                                BOOTSTRAP_SET_VAR_HANDLE,
                                List.of(vm.sym.handleFor(gvar.name()))));

                    ++ storedGvars;
                } else {
                    LocalVar lvar = (LocalVar) param;
                    if (! lvarAccGen.isUnused(lvar)) {
                        // contParam = #vecInternal.get(i)
                        insns.add(new Insn.LoadLocal(vecInternal));
                        insns.add(new Insn.PushInt(i));
                        insns.add(INVOKE_VEC_INTERNAL_GET);
                        insns.add(STORE_CONTPARAM);

                        // store contParam to lvar
                        insns.addAll(lvarAccGen.storeLvar(lvar));
                    }
                }
            }

            // dataStack.removeFromOffset(#topOffset - gvarCount)
            insns.add(LOAD_DATASTACK);
            insns.add(new Insn.LoadLocal(topOffset));
            insns.add(new Insn.PushInt(gvarCount));
            insns.add(new Insn.SubInt());
            insns.add(INVOKE_REMOVE_FROM_OFFSET);

            insns.add(PRODUCE_NADA);
            insns.add(STORE_CONTPARAM);
            insns.addAll(resultCtx.returnOnTail());
            return insns;
        }

        @Override
        public List<Insn> visit(ArgsPassingItree itree) {
            String validCount = keySup.newKeyStr("args-passing-valid-count");
            int paramCount = itree.lvars().size();
            List<Insn> insns = new ArrayList<>(List.of(
                        LOAD_ARGCOUNT,
                        new Insn.PushInt(paramCount),
                        new Insn.IfEqInt(validCount)));
            insns.addAll(pushFakeCall(traceAccum.add(makeCallTrace("op_store", itree.pos()))));
            insns.addAll(List.of(
                        // stackMachine.raiseWrongNumberOfArgs(count, lhsRepr, #args)
                        LOAD_STACKMACHINE,
                        new Insn.PushInt(paramCount),
                        new Insn.PushString(itree.lhsRepr()),
                        InsnsGenerator.LOAD_DATASTACK,
                        InsnsGenerator.LOAD_ARGCOUNT,
                    INVOKE_ARG_VEC,
                        INVOKE_RAISE_WRONG_NUMBER_OF_ARGS,

                        // return
                        new Insn.ReturnValue(),

                        // valid count
                        new Insn.Mark(validCount)));

            // pass args
            for (int i = 0; i < itree.lvars().size(); ++ i) {
                LocalVar lvar = itree.lvars().get(i);
                insns.addAll(lvarAccGen.passArg(lvar, i));
            }

            insns.add(PRODUCE_NADA);
            insns.add(STORE_CONTPARAM);
            insns.addAll(resultCtx.returnOnTail());
            return insns;
        }

        @Override
        public List<Insn> visit(NestedArgsPassingItree itree) {
            List<Insn> insns = new ArrayList<>();

            // if argCount == expected, goto #valid-count
            insns.add(LOAD_ARGCOUNT);
            insns.add(new Insn.PushInt(itree.params().size()));
            String validCount = keySup.newKeyStr("valid-count");
            insns.add(new Insn.IfEqInt(validCount));

            // callStack.pushCse(callTrace, 0, 0, 0)
            int callTraceKey = traceAccum.add(makeCallTrace("op_store", itree.pos()));
            insns.addAll(pushFakeCall(callTraceKey));

            // stackMachine.raiseWrongNumberOfArgs(expected, "...", args); return
            insns.add(LOAD_STACKMACHINE);
            insns.add(new Insn.PushInt(itree.params().size()));
            insns.add(new Insn.PushString(itree.lhsRepr()));
            insns.add(LOAD_DATASTACK);
            insns.add(LOAD_ARGCOUNT);
            insns.add(INVOKE_ARG_VEC);
            insns.add(INVOKE_RAISE_WRONG_NUMBER_OF_ARGS);
            insns.add(new Insn.ReturnValue());

            // #valid-count:
            insns.add(new Insn.Mark(validCount));

            for (int i = 0; i < itree.params().size(); ++ i) {
                NestedParam param = itree.params().get(i);
                insns.addAll(param instanceof LocalVar lvar
                        ? lvarAccGen.passArg(lvar, i)
                        : passTupleArg((NestedParam.Tuple) param, i, callTraceKey));
            }

            insns.add(PRODUCE_NADA);
            insns.add(STORE_CONTPARAM);
            insns.addAll(resultCtx.returnOnTail());
            return insns;
        }

        /**
         * Passes tuple arg.
         */
        private List<Insn> passTupleArg(NestedParam.Tuple param, int argInd, int outerTraceKey) {
            List<Insn> insns = new ArrayList<>();
            // #tupleExpected = dataStack.getArg(argInd)
            insns.add(LOAD_DATASTACK);
            insns.add(new Insn.PushInt(argInd));
            insns.add(INVOKE_ARG);
            String tupleExpected = keySup.newKeyStr("tupleExpected");
            insns.add(new Insn.StoreNewLocal(tupleExpected, Type.getType(Val.class)));

            insns.addAll(passTuple(param, tupleExpected, outerTraceKey));
            return insns;
        }

        /**
         * Passes tuple.
         */
        private List<Insn> passTuple(
                NestedParam.Tuple param, String tupleExpected, int outerTraceKey) {
            List<Insn> insns = new ArrayList<>();

            // if #tupleExpected instanceof VecVal; goto #is-tuple
            insns.add(new Insn.LoadLocal(tupleExpected));
            insns.add(new Insn.InstanceOf(Type.getType(VecVal.class)));
            String isTuple = keySup.newKeyStr("is-tuple");
            insns.add(new Insn.IfNonZero(isTuple));

            // push outer and inner trace
            insns.addAll(pushFakeCall(outerTraceKey));
            int innerTraceKey = traceAccum.add(Trace.of(vm.sym.handleFor("op_store")));
            insns.addAll(pushFakeCall(innerTraceKey));

            // stackMachine.raiseNotVecRhs(#tupleExpected); return
            insns.add(LOAD_STACKMACHINE);
            insns.add(new Insn.LoadLocal(tupleExpected));
            insns.add(INVOKE_RAISE_NOT_VEC_RHS);
            insns.add(new Insn.ReturnValue());

            // #is-tuple:
            insns.add(new Insn.Mark(isTuple));

            // #tuple = (VecVal) #tupleExpected
            insns.add(new Insn.LoadLocal(tupleExpected));
            insns.add(new Insn.CheckCast(Type.getType(VecVal.class)));
            String tuple = keySup.newKeyStr("tuple");
            insns.add(new Insn.StoreNewLocal(tuple, Type.getType(VecVal.class)));

            // #vecInternal = #tuple.vecInternal()
            insns.add(new Insn.LoadLocal(tuple));
            insns.add(INVOKE_VEC_INTERNAL);
            String vecInternal = keySup.newKeyStr("vecInternal");
            insns.add(new Insn.StoreNewLocal(vecInternal, Type.getType(VecInternal.class)));

            // if #vecInternal.size() == expected; goto #right-sized-tuple
            insns.add(new Insn.LoadLocal(vecInternal));
            insns.add(INVOKE_VEC_INTERNAL_SIZE);
            insns.add(new Insn.PushInt(param.lvars().size()));
            String rightSizedTuple = keySup.newKeyStr("right-sized-tuple");
            insns.add(new Insn.IfEqInt(rightSizedTuple));

            // push outer and inner trace
            insns.addAll(pushFakeCall(outerTraceKey));
            insns.addAll(pushFakeCall(innerTraceKey));

            // stackMachine.raiseWrongNumberOfArgs(expected, "...", tuple); return
            insns.add(LOAD_STACKMACHINE);
            insns.add(new Insn.PushInt(param.lvars().size()));
            insns.add(new Insn.PushString(param.lhsRepr()));
            insns.add(new Insn.LoadLocal(tuple));
            insns.add(INVOKE_RAISE_WRONG_NUMBER_OF_ARGS);
            insns.add(new Insn.ReturnValue());

            // #right-sized-tuple:
            insns.add(new Insn.Mark(rightSizedTuple));

            for (int i = 0; i < param.lvars().size(); ++ i) {
                LocalVar lvar = param.lvars().get(i);
                if (! lvarAccGen.isUnused(lvar)) {
                    // store #vecInternal.get(i)
                    insns.add(new Insn.LoadLocal(vecInternal));
                    insns.add(new Insn.PushInt(i));
                    insns.add(INVOKE_VEC_INTERNAL_GET);
                    insns.add(STORE_CONTPARAM);
                    insns.addAll(lvarAccGen.storeLvar(lvar));
                }
            }

            return insns;
        }

        @Override
        public List<Insn> visit(VecItree itree) {
            List<ItreeElem> elems = itree.elems();
            List<Insn> insns = new ArrayList<>(
                    elems.isEmpty() ? MAKE_EMPTY_VEC
                    : isSingleton(elems) ? makeSingletonVec((Itree) elems.get(0))
                    : makeVecOnDataStack(elems));
            insns.addAll(resultCtx.returnOnTail());
            return insns;
        }

        /**
         * Checks contParam is a FunVal.
         */
        private List<Insn> checkFun(String sym, int pos) {
            List<Insn> insns = new ArrayList<>();

            // if contParam instanceof FunVal; goto #is-fun
            String isFun = keySup.newKeyStr("is-fun");
            insns.add(LOAD_CONTPARAM);
            insns.add(new Insn.InstanceOf(Type.getType(FunVal.class)));
            insns.add(new Insn.IfNonZero(isFun));

            // stackMachine.raiseNotFun(contParam, sym, trace); return
            insns.add(LOAD_STACKMACHINE);
            insns.add(LOAD_CONTPARAM);
            insns.add(new Insn.PushString(sym));
            insns.add(produceTrace(Trace.of(new Location(programName, programText, pos))));
            insns.add(INVOKE_RAISE_NOT_FUN);
            insns.add(new Insn.ReturnValue());

            // #is-fun
            insns.add(new Insn.Mark(isFun));
            return insns;
        }

        @Override
        public List<Insn> visit(McallItree itree) {
            List<Insn> insns = new ArrayList<>(generate(itree.ownerRecv(), ResultContext.NON_TAIL));

            // #recv = contParam
            insns.add(LOAD_CONTPARAM);
            String recv = keySup.newKeyStr("recv");
            insns.add(new Insn.StoreNewLocal(recv, Type.getType(Val.class)));

            // contParam = #recv.getVar()
            insns.addAll(deref(recv, itree.sym(), itree.pos()));

            // check contParam is a fun
            insns.addAll(checkFun(itree.sym(), itree.pos()));

            // dataStack.push(contParam)
            insns.add(LOAD_DATASTACK);
            insns.add(LOAD_CONTPARAM);
            insns.add(INVOKE_PUSH_TO_DATASTACK);

            // dataStack.push(#recv)
            insns.add(LOAD_DATASTACK);
            insns.add(new Insn.LoadLocal(recv));
            insns.add(INVOKE_PUSH_TO_DATASTACK);

            // push args and do a call
            Trace trace = makeCallTrace(itree.sym(), itree.pos());
            insns.addAll(pushArgsAndCall(itree.args(), trace, List.of()));
            return insns;
        }

        @Override
        public List<Insn> visit(SymcallItree itree) {
            // contParam = fun
            List<Insn> insns = new ArrayList<>(generate(itree.fun(), ResultContext.NON_TAIL));

            // check contParam is a fun
            insns.addAll(checkFun(itree.sym(), itree.pos()));

            // dataStack.push(contParam)
            insns.add(LOAD_DATASTACK);
            insns.add(LOAD_CONTPARAM);
            insns.add(INVOKE_PUSH_TO_DATASTACK);

            // contParam = recv
            insns.addAll(generate(itree.recv(), ResultContext.NON_TAIL));

            // dataStack.push(contParam)
            insns.add(LOAD_DATASTACK);
            insns.add(LOAD_CONTPARAM);
            insns.add(INVOKE_PUSH_TO_DATASTACK);

            // push args and do a call
            Trace trace = makeCallTrace(itree.sym(), itree.pos());
            insns.addAll(pushArgsAndCall(itree.args(), trace, List.of()));
            return insns;
        }

        @Override
        public List<Insn> visit(AssignmentItree itree) {
            McallItree mcall = new McallItree(
                    itree.lhs(),
                    "op_store",
                    List.of(itree.rhs()),
                    itree.pos());
            return generate(mcall, this.resultCtx);
        }

        /**
         * Methdo names for bi-arithmetic oeration.
         *
         * @param kinkMethod the name of the kink method.
         * @param javaMethod the name of the java method.
         */
        private record MethodNames(String kinkMethod, String javaMethod) {};

        /** Mapping: op → method names. */
        private static final Map<BiArithmeticItree.Op, MethodNames> BI_ARITHMETIC_MAP = Map.of(
                BiArithmeticItree.Op.ADD, new MethodNames("op_add", "add"),
                BiArithmeticItree.Op.SUB, new MethodNames("op_sub", "subtract"),
                BiArithmeticItree.Op.MUL, new MethodNames("op_mul", "multiply"));

        @Override
        public List<Insn> visit(BiArithmeticItree itree) {
            // contParam = recv
            List<Insn> insns = new ArrayList<>(generate(itree.recv(), ResultContext.NON_TAIL));

            // _ = NumVal.isPlainNum(contParam)
            insns.add(LOAD_CONTPARAM);
            insns.add(new Insn.InvokeStatic(Type.getType(NumVal.class),
                        new Method("isPlainNum", Type.BOOLEAN_TYPE,
                            new Type[] { Type.getType(Val.class) })));

            // if (_) goto #plain-num
            String plainNum = keySup.newKeyStr("plain-num");
            insns.add(new Insn.IfNonZero(plainNum));

            // #leftOperand = contParam
            String leftOperand = keySup.newKeyStr("leftOperand");
            insns.add(LOAD_CONTPARAM);

            // contParam = deref[op](#leftOperand)
            insns.add(new Insn.StoreNewLocal(leftOperand, Type.getType(Val.class)));
            MethodNames mnames = BI_ARITHMETIC_MAP.get(itree.op());
            insns.addAll(deref(leftOperand, mnames.kinkMethod(), itree.pos()));

            // check contParam is a fun
            insns.addAll(checkFun(mnames.kinkMethod(), itree.pos()));

            // push contParam as fun
            insns.add(LOAD_DATASTACK);
            insns.add(LOAD_CONTPARAM);
            insns.add(INVOKE_PUSH_TO_DATASTACK);

            // push #leftOperand as recv
            insns.add(LOAD_DATASTACK);
            insns.add(new Insn.LoadLocal(leftOperand));
            insns.add(INVOKE_PUSH_TO_DATASTACK);

            // push num arg
            insns.add(LOAD_DATASTACK);
            insns.add(new Insn.InvokeDynamic(
                        MethodType.methodType(Val.class),
                        BOOTSTRAP_NUM_HANDLE,
                        List.of(itree.arg().toString())));
            insns.add(INVOKE_PUSH_TO_DATASTACK);

            // call/tailcall fun
            String one = keySup.newKeyStr("one");
            insns.add(new Insn.PushInt(1));
            insns.add(new Insn.StoreNewLocal(one, Type.INT_TYPE));
            Trace trace = makeCallTrace(mnames.kinkMethod(), itree.pos());
            String afterArithmetic = keySup.newKeyStr("after-arithmetic");
            insns.addAll(doCall(trace, List.of(new Insn.GoTo(afterArithmetic)), one));

            // #plain-num:
            insns.add(new Insn.Mark(plainNum));

            // contParam = vm.num.of(((NumVal) contParam).bigDecimal().op(num))
            insns.addAll(LOAD_VM);
            insns.add(new Insn.GetField(
                        Type.getType(Vm.class),
                        "num",
                        Type.getType(NumHelper.class)));
            insns.add(LOAD_CONTPARAM);
            insns.add(new Insn.CheckCast(Type.getType(NumVal.class)));
            insns.add(new Insn.InvokeVirtual(Type.getType(NumVal.class),
                        new Method("bigDecimal", Type.getType(BigDecimal.class), new Type[0])));
            insns.add(new Insn.InvokeDynamic(
                        MethodType.methodType(BigDecimal.class),
                        BOOTSTRAP_BIGDECIMAL_HANDLE,
                        List.of(itree.arg().toString())));
            insns.add(new Insn.InvokeVirtual(Type.getType(BigDecimal.class),
                        new Method(mnames.javaMethod(), Type.getType(BigDecimal.class),
                            new Type[] { Type.getType(BigDecimal.class) })));

            insns.add(new Insn.InvokeVirtual(Type.getType(NumHelper.class),
                        new Method("of", Type.getType(NumVal.class),
                            new Type[] { Type.getType(BigDecimal.class) })));
            insns.add(STORE_CONTPARAM);

            // #after-arithmetic:
            insns.add(new Insn.Mark(afterArithmetic));

            insns.addAll(resultCtx.returnOnTail());
            return insns;
        }

        @Override
        public List<Insn> visit(IfItree itree) {
            return controlGen.preloadedIf(itree, InsnsGenerator.this::generate, this.resultCtx);
        }

        @Override
        public List<Insn> visit(BranchItree itree) {
            return controlGen.branch(itree, InsnsGenerator.this::generate, this.resultCtx);
        }

        @Override
        public List<Insn> visit(BranchWithElseItree itree) {
            return controlGen.branchWithElse(itree, InsnsGenerator.this::generate, this.resultCtx);
        }

        @Override
        public List<Insn> visit(TraitNewValItree itree) {
            return controlGen.traitNewVal(itree, InsnsGenerator.this::generate, this.resultCtx);
        }

        @Override
        public List<Insn> visit(NoTraitNewValItree itree) {
            return controlGen.noTraitNewVal(itree, InsnsGenerator.this::generate, this.resultCtx);
        }

    }

}

// vim: et sw=4 sts=4 fdm=marker
