package org.kink_lang.kink.internal.compile.javaclassir;

import java.util.List;
import java.util.Map;

import org.objectweb.asm.Type;

import org.kink_lang.kink.internal.callstack.Trace;

/**
 * The IR of the generated fun in the precursor form of a Java class.
 *
 * @param valFieldCount the number of val fields which capture free vars or the enclosing binding.
 * @param doResumeInsns the insns of doResume method.
 * @param traces the traces.
 * @param desc the desc of the fun.
 * @param childJcirFactories the factories of the IRs of child funs.
 */
public record JavaClassIr(
        int valFieldCount,
        List<Insn> doResumeInsns,
        Map<Integer, Trace> traces,
        String desc,
        List<JcirFactory> childJcirFactories) {

    /**
     * Constructs an IR.
     */
    public JavaClassIr {
        doResumeInsns = List.copyOf(doResumeInsns);
        traces = Map.copyOf(traces);
        childJcirFactories = List.copyOf(childJcirFactories);
    }

    /**
     * Returns the upper bound of data stack usage within the fun
     * not including spreading.
     *
     * @return the upper bound of data stack usage within the fun not including spreading.
     */
    public int dataStackUsageUpperBound() {
        return (int) doResumeInsns().stream()
            .filter(insn ->
                    // (roughly) the number of temp vals used in the fun
                    insn.equals(InsnsGenerator.INVOKE_PUSH_TO_DATASTACK)

                    // (roughly) the number of local vars allocated on data stack
                    || insn.equals(InsnsGenerator.INVOKE_SET_AT_OFFSET))
            .count();
    }

    /**
     * The base name of the generated class.
     * Note that suffix is added to the class name when the class is defined by defineHiddenClass.
     */
    public static final Type TYPE_BASE = Type.getType("Lorg/kink_lang/kink/GeneratedFunValImpl;");

}

// vim: et sw=4 sts=4 fdm=marker
