package org.kink_lang.kink.internal.compile.javaclassir;

import java.util.Locale;
import java.util.HashMap;
import java.util.Map;

import org.kink_lang.kink.internal.contract.Preconds;

/**
 * Generates unique key strings.
 */
public class KeyStrSupplier {

    /** The number of occurences for each base str. */
    private final Map<String, Integer> baseOccurences = new HashMap<>();

    /**
     * Returns a key str unique in {@code this} supplier.
     *
     * <p>For the first invocation with base=Foo, this method returns "#Foo".</p>
     *
     * <p>For the second invocation with base=Foo, this method returns "#Foo#1",
     * for the third, returns "#Foo#2" ...</p>
     *
     * <p>Why it does not add the count for the first invocation?
     * → Answer: to make it easy to write tests of codegen.</p>
     *
     * @param base the base of the key str. It must not contain "#".
     * @return a uniuqe key str within {@code this} supplier.
     */
    public String newKeyStr(String base) {
        Preconds.checkArg(! base.contains("#"),
                "base must not contain '#' for uniqueness of key str");
        if (baseOccurences.containsKey(base)) {
            int occurrenceNum = baseOccurences.get(base);
            baseOccurences.put(base, occurrenceNum + 1);
            return String.format(Locale.ROOT, "#%s#%d", base, occurrenceNum);
        } else {
            baseOccurences.put(base, 1);
            return String.format(Locale.ROOT, "#%s", base);
        }
    }

}

// vim: et sw=4 sts=4 fdm=marker
