package org.kink_lang.kink.internal.compile.javaclassir;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Function;

import org.objectweb.asm.Type;

import org.kink_lang.kink.Vm;
import org.kink_lang.kink.internal.program.itree.FastFunItree;

/**
 * Generator of insns of making binding-capture fast fun.
 */
public class MakeBindingCaptureFastFunGenerator implements MakeFastFunGenerator {

    /** Compiles fun to non-combinator. */
    private final Function<FastFunItree, JavaClassIr> compileNonCombinator;

    /** Compiles fun to combinator. */
    private final Function<FastFunItree, JavaClassIr> compileCombinator;

    /** Generates binding. */
    private final BindingGenerator bindingGen;

    /** Accumulator of JavaClassIr. */
    private final ChildJcirAccumulator jcirAccum;

    /**
     * Constructs a generator.
     *
     * @param compileNonCombinator compiles fun to non-combinator.
     * @param compileCombinator compiles fun to combinator.
     * @param bindingGen generates binding.
     * @param jcirAccum accumulator of JavaClassIr.
     */
    public MakeBindingCaptureFastFunGenerator(
            Function<FastFunItree, JavaClassIr> compileNonCombinator,
            Function<FastFunItree, JavaClassIr> compileCombinator,
            BindingGenerator bindingGen,
            ChildJcirAccumulator jcirAccum) {
        this.compileNonCombinator = compileNonCombinator;
        this.compileCombinator = compileCombinator;
        this.bindingGen = bindingGen;
        this.jcirAccum = jcirAccum;
    }

    @Override
    public List<Insn> makeFun(FastFunItree fun) {
        if (fun.freeLvars().isEmpty()) {
            int jcirInd = jcirAccum.add(() -> compileCombinator.apply(fun));
            return MakeFastFunGenerator.generateCombinator(jcirInd);
        } else {
            int jcirInd = jcirAccum.add(() -> compileNonCombinator.apply(fun));
            return generateNonCombinator(jcirInd);
        }
    }

    /**
     * Generates non-combinator fun.
     */
    private List<Insn> generateNonCombinator(int jcirInd) {
        // make-fun[jcirInd](vm, binding)
        List<Insn> insns = new ArrayList<>(List.of(
                    new Insn.LoadThis(),
                    new Insn.GetField(JavaClassIr.TYPE_BASE, "vm", Type.getType(Vm.class))));
        insns.addAll(bindingGen.generateBinding());
        insns.add(MakeFastFunGenerator.invokeMakeFun(1, jcirInd));
        return insns;
    }

}

// vim: et sw=4 sts=4 fdm=marker
