package org.kink_lang.kink.internal.compile.javaclassir;

import java.lang.invoke.CallSite;
import java.lang.invoke.MethodHandles.Lookup;
import java.lang.invoke.MethodType;
import java.util.ArrayList;
import java.util.List;

import org.objectweb.asm.Handle;
import org.objectweb.asm.Opcodes;

import org.kink_lang.kink.FunVal;
import org.kink_lang.kink.Val;
import org.kink_lang.kink.Vm;
import org.kink_lang.kink.internal.program.itree.FastFunItree;

/**
 * Makes insns to make SSA funs.
 */
public interface MakeFastFunGenerator {

    /**
     * Makes insns to make an SSA fun.
     *
     * <p>The insns must push the fun to the Java stack.</p>
     *
     * @param fun the fun.
     * @return insns.
     */
    List<Insn> makeFun(FastFunItree fun);

    /** The internal name of FunBootstrapper. */
    String FUN_BOOTSTRAPPER_NAME = "org/kink_lang/kink/FunBootstrapper";

    /** Handle of FunBootstrapper.bootstrapMakeFun. */
    Handle BOOTSTRAP_MAKE_FUN_HANDLE = new Handle(Opcodes.H_INVOKESTATIC,
            FUN_BOOTSTRAPPER_NAME,
            "bootstrapMakeFun",
            MethodType.methodType(
                CallSite.class,
                Lookup.class,
                String.class,
                MethodType.class,
                int.class).descriptorString(),
            false);

    /** Handle of FunBootstrapper.bootstrapCombinator. */
    Handle BOOTSTRAP_COMBINATOR_HANDLE = new Handle(Opcodes.H_INVOKESTATIC,
            FUN_BOOTSTRAPPER_NAME,
            "bootstrapCombinator",
            MethodType.methodType(
                CallSite.class,
                Lookup.class,
                String.class,
                MethodType.class,
                int.class).descriptorString(),
            false);

    /**
     * Generates insns for a combinator.
     *
     * <p>The insns push the combinator to the Java stack.</p>
     *
     * @param jcirInd the index of the jcir in childJcirs list.
     * @return insns.
     */
    static List<Insn> generateCombinator(int jcirInd) {
        return List.of(
                new Insn.InvokeDynamic(
                    MethodType.methodType(FunVal.class),
                    BOOTSTRAP_COMBINATOR_HANDLE,
                    List.of(jcirInd)));
    }

    /**
     * Returns the type of the fun constructor.
     *
     * @param valFieldCount the count of val fields of the fun class.
     * @return the type of the fun constructor.
     */
    static MethodType constructorType(int valFieldCount) {
        List<Class<?>> types = new ArrayList<>();
        types.add(Vm.class);
        for (int i = 0; i < valFieldCount; ++ i) {
            types.add(Val.class);
        }
        return MethodType.methodType(FunVal.class, types);
    }

    /**
     * Returns make-fun indy insn.
     *
     * @param valFieldCount the number of val fields.
     * @param jcirInd the index of the jcir.
     * @return the insn.
     */
    static Insn invokeMakeFun(int valFieldCount, int jcirInd) {
        return new Insn.InvokeDynamic(
                MakeFastFunGenerator.constructorType(valFieldCount),
                BOOTSTRAP_MAKE_FUN_HANDLE,
                List.of(jcirInd));
    }

}

// vim: et sw=4 sts=4 fdm=marker
