package org.kink_lang.kink.internal.compile.javaclassir;

import java.util.List;

import org.kink_lang.kink.internal.callstack.Trace;

/**
 * How the result of evaluation is used.
 */
enum ResultContext {

    /**
     * The tail of the fun call.
     */
    TAIL {
        @Override
        List<Insn> returnOnTail() {
            return List.of(
                    InsnsGenerator.LOAD_STACKMACHINE,
                    InsnsGenerator.LOAD_CONTPARAM,
                    InsnsGenerator.INVOKE_TRANSITION_TO_RESULT,
                    new Insn.ReturnValue());
        }

        @Override
        List<Insn> returnOnTailOrMark(String labelKey) {
            return returnOnTail();
        }

        @Override
        Trace onTailOrNot(Trace trace) {
            return trace.onTail();
        }

        @Override
        ResultContext actAsTail() {
            return TAIL;
        }
    },

    /**
     * Not the tail of the fun call.
     */
    NON_TAIL {
        @Override
        List<Insn> returnOnTail() {
            return List.of();
        }

        @Override
        List<Insn> returnOnTailOrMark(String labelKey) {
            return List.of(new Insn.Mark(labelKey));
        }

        @Override
        Trace onTailOrNot(Trace trace) {
            return trace;
        }

        @Override
        ResultContext actAsTail() {
            return SYNTHETIC_TAIL;
        }
    },

    /**
     * A synthetic tail -- such as the end of if-then clause -- of the fun call.
     */
    SYNTHETIC_TAIL {
        @Override
        List<Insn> returnOnTail() {
            return List.of();
        }

        @Override
        List<Insn> returnOnTailOrMark(String labelKey) {
            return List.of(new Insn.Mark(labelKey));
        }

        @Override
        Trace onTailOrNot(Trace trace) {
            return trace.onTail();
        }

        @Override
        ResultContext actAsTail() {
            return SYNTHETIC_TAIL;
        }
    };

    /**
     * Produces return-from-fun insns on the tail.
     */
    abstract List<Insn> returnOnTail();

    /**
     * Produces return-from-fun insns on the tail,
     * or otherwise mark the label.
     */
    abstract List<Insn> returnOnTailOrMark(String labelKey);

    /**
     * Tail trace for synthetic or true tail,
     * or non-tail trace.
     */
    abstract Trace onTailOrNot(Trace trace);

    /**
     * Returns the context converted to the form
     * which acts as a tail.
     */
    abstract ResultContext actAsTail();

}

// vim: et sw=4 sts=4 fdm=marker
