package org.kink_lang.kink.internal.compile.javaclassir;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

import org.kink_lang.kink.Vm;
import org.kink_lang.kink.internal.program.itree.*;

/**
 * Compiles top level funs.
 */
public class TopLevelCompiler {

    /** The vm. */
    private final Vm vm;

    /** The program name. */
    private final String programName;

    /** The program text. */
    private final String programText;

    /** The logger for the class. */
    private static final System.Logger LOGGER
        = System.getLogger(TopLevelCompiler.class.getName());

    /**
     * Constructs a compiler.
     *
     * @param vm the vm.
     * @param programName the name of the program.
     * @param programText the text of the program.
     */
    public TopLevelCompiler(Vm vm, String programName, String programText) {
        this.vm = vm;
        this.programName = programName;
        this.programText = programText;
    }

    /**
     * Compiles a top level fun.
     *
     * @param topLevel the itree of the top level of the program.
     * @return the compile jcir.
     */
    public JavaClassIr compile(Itree topLevel) {
        var keySup = new KeyStrSupplier();
        var traceAccum = new TraceAccumulator();
        var bindingGen = BindingGenerator.FIELD;
        var lvarAccGen = new SlowLvarAccessGenerator(vm, bindingGen, keySup, traceAccum);
        var nonCombinatorCompiler = new BindingCaptureFastFunCompiler(vm, programName, programText);
        var combinatorCompiler = new ValCaptureFastFunCompiler(vm, programName, programText);
        var jcirAccum = new ChildJcirAccumulator();
        var makeFastFunGen = new MakeBindingCaptureFastFunGenerator(
                nonCombinatorCompiler::compile,
                combinatorCompiler::compileControlUnchanged,
                bindingGen,
                jcirAccum);
        var controlGen = new OverriddenControlGenerator(
                vm, programName, programText, keySup, traceAccum);
        InsnsGenerator insnsGen = new InsnsGenerator(vm, programName, programText,
                bindingGen,
                lvarAccGen,
                makeFastFunGen,
                new InSlowFunLetRecGenerator(),
                controlGen,
                keySup,
                traceAccum,
                new ProgramCounterSupplier(traceAccum),
                jcirAccum);

        List<Insn> insns = new ArrayList<>(CompilerSupport.PROLOGUE);
        insns.addAll(insnsGen.generate(topLevel, ResultContext.TAIL));
        insns.addAll(CompilerSupport.EPILOGUE);
        String desc = String.format(Locale.ROOT, "(top-level-fun location=%s)",
                vm.location.of(programName, programText, topLevel.pos()).desc());
        LOGGER.log(System.Logger.Level.TRACE, "insns for {0}: {1}", desc, insns);
        return new JavaClassIr(
                1,
                insns,
                traceAccum.traces(),
                desc,
                jcirAccum.childJcirFactories());
    }

}

// vim: et sw=4 sts=4 fdm=marker
