package org.kink_lang.kink.internal.compile.tempval;

import java.util.Locale;

import org.kink_lang.kink.Val;
import org.kink_lang.kink.Vm;

/**
 * Val to hold an int number.
 *
 * <p>This val will be never used out of generated code.</p>
 */
public class IntVal extends Val {

    /** The int num. */
    private final int num;

    /**
     * Constructs an int val.
     *
     * @param vm the vm.
     * @param num the int num.
     */
    public IntVal(Vm vm, int num) {
        super(vm);
        this.num = num;
    }

    /**
     * Returns an int val for the number.
     *
     * @param vm the vm.
     * @param num the int num.
     * @return an int val for the int num.
     */
    public static IntVal of(Vm vm, int num) {
        IntVal.Holder ih = vm.component.get(IntVal.Holder.class);
        return ih.of(num);
    }

    /**
     * Returns the int number.
     *
     * @return the int num.
     */
    public int getInt() {
        return this.num;
    }

    @Override
    public String toString() {
        return String.format(Locale.ROOT, "IntVal(%d)", this.num);
    }

    @Override
    public int hashCode() {
        return this.num;
    }

    @Override
    public boolean equals(Object arg) {
        return arg == this
            || arg instanceof IntVal
            && this.vm.equals(((IntVal) arg).vm)
            && this.getInt() == ((IntVal) arg).getInt();
    }

    /**
     * Holder of int vals.
     */
    public static class Holder {

        /** The vm. */
        private final Vm vm;

        /** Cached int vals from MIN_CACHED to MAX_CACHED. */
        private final IntVal[] cache = new IntVal[(MAX_CACHED - MIN_CACHED) + 1];

        /** The minimum number of cached IntVal. */
        private static final int MIN_CACHED = -100;

        /** The maximum number of cached IntVal. */
        private static final int MAX_CACHED = 100;

        /**
         * Constructs a holder of int vals.
         *
         * @param vm the vm.
         */
        public Holder(Vm vm) {
            this.vm = vm;
        }

        /**
         * Returns an int val for the number.
         *
         * @param num the int num.
         * @return an int val for the int num.
         */
        public IntVal of(int num) {
            return MIN_CACHED <= num && num <= MAX_CACHED
                ? this.cache[num - MIN_CACHED]
                : new IntVal(vm, num);
        }

        /**
         * Initializes the holder.
         */
        public void init() {
            for (int i = MIN_CACHED; i <= MAX_CACHED; ++ i) {
                this.cache[i - MIN_CACHED] = new IntVal(vm, i);
            }
        }

    }

}

// vim: et sw=4 sts=4 fdm=marker
