package org.kink_lang.kink.internal.intrinsicsupport;

import org.kink_lang.kink.*;
import org.kink_lang.kink.hostfun.CallContext;
import org.kink_lang.kink.hostfun.HostResult;
import org.kink_lang.kink.hostfun.graph.GraphNode;
import org.kink_lang.kink.internal.compile.tempval.IntVal;

/**
 * Provides funs used in wires as konsts.
 *
 * This class is registered as a component.
 */
public class FunsHolder {

    /** The vm. */
    private final Vm vm;

    /** Raises an exception which indicates not-bool result of cond fun for branch. */
    private final FunVal raiseBranchCondNotBool;

    /**
     * Constructs a holder.
     */
    FunsHolder(Vm vm) {
        this.vm = vm;
        this.raiseBranchCondNotBool = vm.fun.make().take(2).action(this::raiseBranchCondNotBool);
    }

    /**
     * Returns the instance registered to the vm.
     *
     * @param vm the vm.
     * @return the instance for the vm.
     */
    private static FunsHolder instance(Vm vm) {
        return vm.component.getOrRegister(FunsHolder.class, FunsHolder::new);
    }

    /**
     * Returns wrong-number-of-args fun, which takes no args.
     *
     * @param vm the vm.
     * @param paramCount the expected count of params.
     * @param params the desc of params.
     * @param args the actual args.
     * @return wrong-number-of-args fun.
     */
    public static FunVal wrongNumberOfArgs(
            Vm vm, int paramCount, GraphNode params, VecVal args) {
        int argsCount = args.toList().size();
        String template = argsCount < paramCount
            ? "too few args: for {}, got {}"
            : "too many args: for {}, got {}";
        return vm.fun.make().action(c -> c.call(vm.graph.raiseFormat(
                        template,
                        params,
                        vm.graph.repr(args))));
    }

    /**
     * Returns not-vec-rhs fun, which takes no arg.
     *
     * @param vm the vm.
     * @param rhs the right-hand-side of assignment.
     * @return the fun.
     */
    public static FunVal raiseNotVecRhs(Vm vm, Val rhs) {
        return vm.fun.make().action(c -> c.call(vm.graph.raiseFormat(
                        "Vec.op_store(Rhs): the arg must be a vec, but got {}",
                        vm.graph.repr(rhs))));
    }

    /**
     * Raises an exception which indicates not-bool result of cond fun for branch.
     */
    private HostResult raiseBranchCondNotBool(CallContext c) {
        Val indVal = c.arg(0);
        int ind = ((IntVal) indVal).getInt();
        Val actual = c.arg(1);
        return c.call(BranchSupport.condNotBool(vm, ind, actual));
    }

    /**
     * Returns {@code raiseBranchCondNotBool} fun.
     *
     * @param vm the vm.
     * @return {@code raiseBranchCondNotBool} fun.
     */
    public static FunVal getRaiseBranchCondNotBool(Vm vm) {
        return instance(vm).raiseBranchCondNotBool;
    }

}

// vim: et sw=4 sts=4 fdm=marker
