package org.kink_lang.kink.internal.mod.java;

import javax.annotation.Nullable;

import org.kink_lang.kink.JavaVal;
import org.kink_lang.kink.Val;
import org.kink_lang.kink.Vm;
import org.kink_lang.kink.hostfun.CallContext;
import org.kink_lang.kink.hostfun.HostResult;

/**
 * Factory of THROWER mod.
 */
public class ThrowerMod {

    /** The vm. */
    private final Vm vm;

    /** The name of the mod. */
    public static final String MOD_NAME = "kink/javahost/THROWER";

    /**
     * Constructs the factory.
     */
    private ThrowerMod(Vm vm) {
        this.vm = vm;
    }

    /**
     * Makes THROWER mod in the vm.
     *
     * @param vm the vm.
     * @return THROWER mod.
     */
    public static Val makeMod(Vm vm) {
        return new ThrowerMod(vm).makeMod();
    }

    /**
     * Maks a mod.
     */
    private Val makeMod() {
        Val mod = vm.newVal();
        mod.setVar(vm.sym.handleFor("new"),
                vm.fun.make("THROWER.new(Exception_java_val)").take(1).action(this::newFun));
        mod.setVar(vm.sym.handleFor("is?"),
                vm.fun.make("THROWER.is?(Val)").take(1).action(this::isPFun));
        return mod;
    }

    /**
     * Implementation of THROWER.new.
     */
    private HostResult newFun(CallContext c) {
        Throwable th = extractThrowable(c.arg(0));
        if (th == null) {
            String desc = "THROWER.new(Exception_java_val)";
            return c.call(vm.graph.raiseFormat(
                        "{}: the arg must be a java_val of a Throwable, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(c.arg(0))));
        }
        return ThrowerVal.of(vm, th);
    }

    /**
     * Extracts a Throwable instance from arg,
     * or null if it is not an appropriate java_val.
     */
    @Nullable
    private Throwable extractThrowable(Val arg) {
        if (! (arg instanceof JavaVal)) {
            return null;
        }
        JavaVal excJavaVal = (JavaVal) arg;
        Object obj = excJavaVal.objectReference();
        if (! (obj instanceof Throwable)) {
            return null;
        }
        Throwable th = (Throwable) obj;
        return th;
    }

    /**
     * Implementation of THROWER.is?.
     */
    private HostResult isPFun(CallContext c) {
        return vm.bool.of(c.arg(0) instanceof ThrowerVal);
    }

}

// vim: et sw=4 sts=4 fdm=marker
