package org.kink_lang.kink.internal.mod.java;

import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import org.kink_lang.kink.Val;
import org.kink_lang.kink.SharedVars;
import org.kink_lang.kink.Vm;
import org.kink_lang.kink.hostfun.CallContext;
import org.kink_lang.kink.hostfun.HostResult;

/**
 * Thrower val, which is used to throw an exception from the proxy method.
 */
class ThrowerVal extends Val {

    /** Wrapped throwable. */
    private final Throwable thrown;

    /**
     * Constructs a throwable val.
     */
    private ThrowerVal(Vm vm, SharedVars sharedVars, Throwable thrown) {
        super(vm, sharedVars);
        this.thrown = thrown;
    }

    /**
     * Makes a thrower val.
     */
    static ThrowerVal of(Vm vm, Throwable thrown) {
        ThrowerHelper helper = ThrowerHelper.of(vm);
        return new ThrowerVal(vm, helper.sharedVars, thrown);
    }

    /**
     * Returns the wrapped throwable.
     */
    Throwable getThrown() {
        return this.thrown;
    }

    @Override
    public String toString() {
        return String.format(Locale.ROOT, "ThrowerVal(%s)", this.thrown);
    }

    /**
     * Returns the properties of the thrower val.
     */
    private List<Object> getProperties() {
        return List.of(this.vm, this.thrown);
    }

    @Override
    public int hashCode() {
        return getProperties().hashCode();
    }

    @Override
    public boolean equals(Object arg) {
        return arg == this
            || arg instanceof ThrowerVal
            && getProperties().equals(((ThrowerVal) arg).getProperties());
    }

    /**
     * Helper of thrower vals.
     */
    static class ThrowerHelper {

        /** Vm. */
        private final Vm vm;

        /** Shared vars of thrower vals. */
        final SharedVars sharedVars;

        /**
         * Constructs a helper for the vm.
         */
        private ThrowerHelper(Vm vm) {
            this.vm = vm;
            this.sharedVars = makeSharedVars();
        }

        /**
         * Returns the helper instance of the vm.
         */
        static ThrowerHelper of(Vm vm) {
            return vm.component.getOrRegister(ThrowerHelper.class, ThrowerHelper::new);
        }

        /**
         * Makes the set of shared vars for the helper.
         */
        private SharedVars makeSharedVars() {
            Map<Integer, Val> map = new HashMap<>();
            map.put(vm.sym.handleFor("thrown"),
                    vm.fun.make("Thrower.thrown").take(0).action(this::thrownMethod));
            map.put(vm.sym.handleFor("repr"),
                    vm.fun.make("Thrower.repr").take(0).action(this::reprMethod));
            return vm.sharedVars.of(map);
        }

        /**
         * Implementation of Thrower.thrown.
         */
        private HostResult thrownMethod(CallContext c) {
            Val recv = c.recv();
            if (! (recv instanceof ThrowerVal)) {
                return c.call(vm.graph.raiseFormat(
                            "Thrower.thrown: recv must be a thrower, but got {}",
                            vm.graph.repr(recv)));
            }
            ThrowerVal thrower = (ThrowerVal) recv;
            return vm.java.of(thrower.getThrown(), Throwable.class);
        }

        /**
         * Implementation of Thrower.repr.
         */
        private HostResult reprMethod(CallContext c) {
            Val recv = c.recv();
            if (! (recv instanceof ThrowerVal)) {
                return c.call(vm.graph.raiseFormat(
                            "Thrower.repr: recv must be a thrower, but got {}",
                            vm.graph.repr(recv)));
            }
            ThrowerVal thrower = (ThrowerVal) recv;
            return vm.str.of(String.format(Locale.ROOT, "Thrower(%s)", thrower.getThrown()));
        }
    }

}

// vim: et sw=4 sts=4 fdm=marker
