package org.kink_lang.kink.internal.program.ast;

import java.util.List;
import java.util.Locale;
import java.util.function.Function;

import org.kink_lang.kink.internal.contract.Preconds;
import org.kink_lang.kink.internal.function.Function3;
import org.kink_lang.kink.internal.program.lex.EotToken;
import org.kink_lang.kink.internal.program.lex.ErrorToken;
import org.kink_lang.kink.internal.program.lex.Token;

/**
 * A parser of Kink programs.
 *
 * The parser converts a list of tokens to an AST.
 */
public class Parser {

    /** The factory of ParseRun. */
    private final Function<List<Token>, ParseRun> parseRunFactory;

    /**
     * Constructs a parser.
     *
     * @param locale the locale for error messages.
     */
    public Parser(Locale locale) {
        this(locale, tokens -> new ParseRun(locale, tokens));
    }

    /**
     * Constructs a parser with a factory of ParseRun.
     */
    Parser(Locale locale, Function<List<Token>, ParseRun> parseRunFactory) {
        this.parseRunFactory = parseRunFactory;
    }

    /**
     * Parses the token list to produce an AST.
     *
     * <p>When parsing succeeds,
     * the method calls {@code onSucc} with the AST and returns its result.</p>
     *
     * <p>When parsing results in an error,
     * the method calls {@code onError} with the message, startPos, and endPos,
     * then returns its result.</p>
     *
     * <p>The last element of {@code tokens} must be
     * an instance of {@link EotToken} or {@link ErrorToken}.</p>
     *
     * @param tokens the token list of a Kink program.
     * @param onSucc called when the parsing succeeds.
     * @param onError called when the parsing results in a compile error.
     * @param <T> the result type of onSucc and onError.
     * @return the result of onSucc or onError.
     * @throws IllegalArgumentException
     *      when the last element of the {@code tokens} argument
     *      is not an EotToken nor an ErrorToken.
     */
    public <T> T parse(List<Token> tokens,
            Function<? super SeqExpr, ? extends T> onSucc,
            Function3<? super String, ? super Integer, ? super Integer, ? extends T> onError) {
        Preconds.checkArg(! tokens.isEmpty(), "tokens must not be empty");
        Token last = tokens.get(tokens.size() - 1);
        Preconds.checkArg(last instanceof EotToken || last instanceof ErrorToken,
                "last token must be EotToken or ErrorToken");

        ParseRun parseRun = parseRunFactory.apply(tokens);
        try {
            SeqExpr seq = parseRun.run();
            return onSucc.apply(seq);
        } catch (CompileException ce) {
            return onError.apply(ce.getMessage(), ce.getStartPos(), ce.getEndPos());
        }
    }

}

// vim: et sw=4 sts=4 fdm=marker
