package org.kink_lang.kink.internal.program.itree;

import java.util.HashSet;
import java.util.Set;

/**
 * Analyzes usage and definition of local vars in a fun.
 */
class UsageDefinitionAnalysis implements DeepTransformer.Callback {

    /** Lvars used in the fun body. */
    private final Set<LocalVar> used = new HashSet<>();

    /**
     * Lvars defined in the fun body.
     */
    private final Set<LocalVar> defined = new HashSet<>();

    /**
     * Returns the local vars used in the fun body.
     */
    Set<LocalVar> usedLvars() {
        return Set.copyOf(this.used);
    }

    /**
     * Returns the local vars defined in the fun body.
     */
    Set<LocalVar> definedLvars() {
        return Set.copyOf(this.defined);
    }

    /**
     * Calculates free lvars.
     */
    Set<LocalVar> freeLvars() {
        var set = new HashSet<>(this.used);
        set.removeAll(this.defined);
        return Set.copyOf(set);
    }

    /**
     * Analyzes the usage and the definition of {@code body}.
     *
     * @param body the body of a fun.
     * @return the analysis result.
     */
    static UsageDefinitionAnalysis analyzeUseDefine(Itree body) {
        var analysis = new UsageDefinitionAnalysis();
        DeepTransformer.deepTransform(body, analysis);
        return analysis;
    }

    @Override
    public LocalVar storeLvar(LocalVar lvar) {
        this.defined.add(lvar);
        return lvar;
    }

    @Override
    public LocalVar derefLvar(LocalVar lvar) {
        this.used.add(lvar);
        return lvar;
    }

    @Override
    public Itree itree(Itree itree) {
        itree.accept(new Visitor());
        return itree;
    }

    /**
     * Visitor for analysis.
     */
    private class Visitor extends SkeltonItreeVisitor<Void> {

        /**
         * Constructs a visitor.
         */
        private Visitor() {
            super(itree -> null);
        }

        @Override
        public Void visit(FastFunItree nested) {
            used.addAll(nested.freeLvars());
            return null;
        }

        @Override
        public Void visit(IfItree preloadedIf) {
            traverseSub(preloadedIf.trueFun().body());
            preloadedIf.falseFun().map(FastFunItree::body).ifPresent(this::traverseSub);
            return null;
        }

        @Override
        public Void visit(BranchItree branch) {
            branch.condThenPairs().stream().forEach(condThen -> {
                traverseSub(condThen.condFun().body());
                traverseSub(condThen.thenFun().body());
            });
            return null;
        }

        @Override
        public Void visit(BranchWithElseItree branch) {
            branch.condThenPairs().stream().forEach(condThen -> {
                traverseSub(condThen.condFun().body());
                traverseSub(condThen.thenFun().body());
            });
            traverseSub(branch.elseThenFun().body());
            return null;
        }

        /**
         * Traverses sub itree.
         */
        private void traverseSub(Itree sub) {
            DeepTransformer.deepTransform(sub, UsageDefinitionAnalysis.this);
        }

    }

}

// vim: et sw=4 sts=4 fdm=marker
