package org.kink_lang.kink.internal.program.itreeoptimize;

import java.util.ArrayList;
import java.util.List;

import org.kink_lang.kink.internal.program.itree.*;

/**
 * Applies inlining of calls of branch in the form below.
 *
 * <pre>
 * branch(
 *   { cond1 } { body1 }
 *   { cond2 } { body2 }
 *   ,,,
 *   { condn } { bodyn }
 *   $true { elseBody }
 * )
 * </pre>
 *
 * <p>This inlining must take place under the body of IndexFunItree.</p>
 *
 * <p>Conds must not modify local vars, because it wrongly affects dereferences
 * in other conds and result bodies.</p>
 */
public class BranchWithElseInliner extends BaseOptimizer {

    @Override
    public Itree visit(SymcallItree org) {
        if (! org.isLocalCallOf("branch")) {
            return org;
        }

        List<ItreeElem> argElems = org.args();
        if (argElems.stream().anyMatch(ItreeElem::isSpread)) {
            return org;
        }

        List<Itree> args = argElems.stream().map(ItreeElem::expr).toList();
        if (args.size() < 2 || args.size() % 2 != 0) {
            return org;
        }

        List<Itree> beforeElse = args.subList(0, args.size() - 2);
        if (! beforeElse.stream().allMatch(a -> a instanceof FastFunItree)) {
            return org;
        }

        List<FastFunItree> beforeElseFuns = beforeElse.stream()
            .map(a -> (FastFunItree) a)
            .toList();

        List<CondThenPair> condThenPairs = new ArrayList<>();
        for (int i = 0; i < beforeElseFuns.size(); i += 2) {
            FastFunItree condFun = beforeElseFuns.get(i);
            FastFunItree bodyFun = beforeElseFuns.get(i + 1);
            if (RecvArgsSearcher.containsRecvOrArgs(condFun.body())
                    || RecvArgsSearcher.containsRecvOrArgs(bodyFun.body())) {
                return org;
            }
            condThenPairs.add(new CondThenPair(condFun, bodyFun));
        }

        Itree lastCond = args.get(args.size() - 2);
        if (! (lastCond instanceof LderefItree lastCondLderef)
                || ! lastCondLderef.lvar().equals(new LocalVar.Original("true"))) {
            return org;
        }

        Itree lastBody = args.get(args.size() - 1);
        if (! (lastBody instanceof FastFunItree elseBody)
                || RecvArgsSearcher.containsRecvOrArgs(elseBody.body())) {
            return org;
        }

        return new BranchWithElseItree(condThenPairs, elseBody, org.pos());
    }

}

// vim: et sw=4 sts=4 fdm=marker
