package org.kink_lang.kink.internal.sym;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;

import javax.annotation.Nullable;

import org.kink_lang.kink.SymRegistry;
import org.kink_lang.kink.internal.contract.Preconds;
import org.kink_lang.kink.internal.intrinsicsupport.PreloadedFuns;

/**
 * The bidirectional registry of syms and sym-handles.
 *
 * <p>A sym-handle is an int value between 1 and 0x3fff_ffff
 * corresponding to a sym.
 * A handle is generated when {@link #handleFor(String)}
 * is called for the sym for the first time.</p>
 *
 * <p>The registry can be accessed from multiple threads concurrently.</p>
 */
public class SymRegistryImpl implements SymRegistry {

    /** The maximum sym handle. */
    private final int maxSymHandle;

    /** The reference of the sym mapping. */
    private final AtomicReference<Mapping> mappingRef;

    /** The default value of maxSymHandle. */
    private static final int MAX_SYM_HANDLE = 0x3fff_ffff;

    /** The maximum sym handle of preloaded control syms. */
    public static final int MAX_PRELOADED_CONTROL = PreloadedFuns.controlSyms().size();

    /** Sym handle of "repr". */
    public static final int REPR = MAX_PRELOADED_CONTROL + 1;

    /**
     * Constructs a sym handle.
     */
    public SymRegistryImpl() {
        this(MAX_SYM_HANDLE);
    }

    /**
     * Constructs a sym handle specifying maxSymHandle.
     */
    SymRegistryImpl(int maxSymHandle) {
        this.maxSymHandle = maxSymHandle;
        Mapping mapping = new Mapping();
        mapping = mapping.add("");
        for (String controlSym : PreloadedFuns.controlSyms()) {
            mapping = mapping.add(controlSym);
        }
        mapping = mapping.add("repr");
        this.mappingRef = new AtomicReference<>(mapping);
    }

    /**
     * Registers or gets the handle of the sym.
     *
     * @param sym the sym.
     * @return the handle.
     * @throws IllegalStateException
     *      if the sym is not registered yet,
     *      and the registry does not have capacity for the new sym.
     */
    @Override
    public int handleFor(String sym) {
        while (true) {
            Mapping mapping = mappingRef.get();
            Integer handle = mapping.getHandle(sym);
            if (handle != null) {
                return handle;
            }
            Preconds.checkState(
                    mapping.getLastSymHandle() < maxSymHandle, "cannot register any more syms");
            Mapping newMapping = mapping.add(sym);
            mappingRef.compareAndSet(mapping, newMapping);
        }
    }

    /**
     * Returns the corresponding sym for the handle.
     *
     * @param handle the handle.
     * @return the corresponding sym for the handle.
     * @throws IllegalArgumentException if the specified handle is not registered.
     */
    @Override
    public String symFor(int handle) {
        Mapping mapping = mappingRef.get();
        Preconds.checkArg(mapping.isValidHandle(handle), "unregistred handle");
        return mapping.getSym(handle);
    }

    /**
     * Returns true if the handle is registered.
     *
     * @param handle the handle.
     * @return true if the handle is registered.
     */
    @Override
    public boolean isValidHandle(int handle) {
        Mapping mapping = mappingRef.get();
        return mapping.isValidHandle(handle);
    }

    /**
     * Returns the last registered handle.
     */
    int getLastSymHandle() {
        Mapping mapping = mappingRef.get();
        return mapping.getLastSymHandle();
    }

    /**
     * Whether the handle is a preloaded fun.
     *
     * @param handle the handle to test.
     * @return whether the handle is a preloaded fun.
     */
    public static boolean isPreloaded(int handle) {
        return 1 <= handle && handle <= MAX_PRELOADED_CONTROL;
    }

    /**
     * Bidirectional mapping of syms.
     *
     * Instances of this class are immutable.
     */
    private static class Mapping {

        /** Map from syms to handles. */
        private final Map<String, Integer> symToHandle;

        /** Map from (handle - 1) to sym. */
        private final String[] handleM1ToSym;

        /**
         * Constructs an empty mapping.
         */
        Mapping() {
            this(Collections.emptyMap(), new String[0]);
        }

        /**
         * Constructs a mapping.
         */
        Mapping(Map<String, Integer> symToHandle, String[] handleM1ToSym) {
            this.symToHandle = symToHandle;
            this.handleM1ToSym = handleM1ToSym;
        }

        /**
         * Returns a new mapping adding the sym.
         *
         * Precondition: sym must not be registered in the current mapping.
         */
        Mapping add(String sym) {
            String internedSym = sym.intern();
            int newHandle = handleM1ToSym.length;
            Map<String, Integer> newMap = new HashMap<>(symToHandle);
            Integer old = newMap.put(internedSym, newHandle);
            assert old == null;
            String[] newArray = new String[handleM1ToSym.length + 1];
            System.arraycopy(handleM1ToSym, 0, newArray, 0, handleM1ToSym.length);
            newArray[newHandle] = internedSym;
            return new Mapping(newMap, newArray);
        }

        /**
         * Returns the handle corresponding to the sym;
         * or null if absent.
         */
        @Nullable
        Integer getHandle(String sym) {
            return symToHandle.get(sym);
        }

        /**
         * Returns the sym corresponding to the handle.
         *
         * Precondition: handle must be registered.
         */
        String getSym(int handle) {
            return handleM1ToSym[handle];
        }

        /**
         * Returns true if the handle is registered.
         */
        boolean isValidHandle(int handle) {
            return 0 <= handle && handle <= getLastSymHandle();
        }

        /**
         * Returns the lastly registered handle.
         */
        int getLastSymHandle() {
            return handleM1ToSym.length - 1;
        }

    }

}

// vim: et sw=4 sts=4 fdm=marker
