package org.kink_lang.kink;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import org.kink_lang.kink.hostfun.HostContext;
import org.kink_lang.kink.hostfun.CallContext;
import org.kink_lang.kink.hostfun.HostResult;
import org.kink_lang.kink.internal.intrinsicsupport.BranchSupport;
import org.kink_lang.kink.internal.intrinsicsupport.IfSupport;
import org.kink_lang.kink.internal.intrinsicsupport.NewValSupport;

/**
 * The helper class of local binding vals.
 */
public class BindingHelper {

    /** The vm. */
    private final Vm vm;

    /** Shared vars of local binding vals. */
    SharedVars sharedVars;

    /**
     * Constructs a helper.
     */
    BindingHelper(Vm vm) {
        this.vm = vm;
    }

    /**
     * Sets up for local binding vals.
     */
    void init() {
        int stdoutHandle = vm.sym.handleFor("stdout");
        int stdinHandle = vm.sym.handleFor("stdin");
        int stderrHandle = vm.sym.handleFor("stderr");

        Map<Integer, Val> vars = new HashMap<>();
        vars.put(vm.sym.handleFor("nada"), vm.fun.constant(vm.nada));
        vars.put(vm.sym.handleFor("true"), vm.fun.constant(vm.bool.trueVal));
        vars.put(vm.sym.handleFor("false"), vm.fun.constant(vm.bool.falseVal));
        vars.put(vm.sym.handleFor("new_val"), vm.fun.make("new_val(...[Sym1 Val1 Sym2 Val2 ,,,])")
                .action(this::newValFun));
        vars.put(vm.sym.handleFor("raise"),
                vm.fun.make("raise(Msg)").take(1).action(this::raiseFun));
        vars.put(vm.sym.handleFor("op_lognot"), vm.fun.make("op_lognot(Bool)")
                .take(1).action(this::opLogNotFun));
        vars.put(vm.sym.handleFor("op_logor"), vm.fun.make("op_logor(Bool $thunk)")
                .take(2).action(this::opLogOrFun));
        vars.put(vm.sym.handleFor("op_logand"), vm.fun.make("op_logand(Bool $thunk)")
                .take(2).action(this::opLogAndFun));
        vars.put(vm.sym.handleFor("if"), vm.fun.make("if(Bool $true_cont ...[$false_cont={}])")
                .takeMinMax(2, 3)
                .action(this::ifFun));
        vars.put(vm.sym.handleFor("branch"),
                vm.fun.make("branch(...[$cond1 $then1 $cond2 $then2 ,,,])")
                .action(this::branchFun));
        vars.put(stdinHandle, vm.fun.make("stdin")
                .take(0).action(c -> c.call("kink/_io/STDIO", stdinHandle)));
        vars.put(stdoutHandle, vm.fun.make("stdout")
                .take(0).action(c -> c.call("kink/_io/STDIO", stdoutHandle)));
        vars.put(stderrHandle, vm.fun.make("stderr")
                .take(0).action(c -> c.call("kink/_io/STDIO", stderrHandle)));

        vars.put(vm.sym.handleFor("repr"), vm.fun.make("Binding.repr").take(0).action(
                    c -> vm.str.of(String.format(Locale.ROOT,
                            "(binding val_id=%d)", c.recv().identity()))));

        this.sharedVars = vm.sharedVars.of(vars);
    }

    /**
     * Returns a new local binding val.
     *
     * @return a new local binding val.
     */
    public BindingVal newBinding() {
        return new BindingVal(vm);
    }

    // new_val(Name1 Val1 Name2 Val2 ...) {{{1

    /**
     * Implementation of new_val.
     */
    private HostResult newValFun(CallContext c) {
        int argCount = c.argCount();
        if (argCount % 2 != 0) {
            return c.call(NewValSupport.oddNumbefOfArgs(vm, argCount));
        }

        Val val = vm.newVal();
        for (int i = 0; i < argCount; i += 2) {
            Val symVal = c.arg(i);
            if (! (symVal instanceof StrVal sym)) {
                return c.call(NewValSupport.symNotStr(vm, i, symVal));
            }
            val.setVar(vm.sym.handleFor(sym.string()), c.arg(i + 1));
        }
        return val;
    }

    // }}}1

    // raise(Msg) {{{1

    /**
     * Implementation of raise.
     */
    private HostResult raiseFun(CallContext c) {
        Val msgVal = c.arg(0);
        if (! (msgVal instanceof StrVal)) {
            return c.call(vm.graph.raiseFormat("raise(Msg): Msg must be a str, but got {}",
                        vm.graph.repr(msgVal)));
        }
        String msg = ((StrVal) msgVal).string();
        return c.raise(msg);
    }

    // }}}1

    // op_lognot {{{

    /**
     * Implementation of op_lognot.
     */
    private HostResult opLogNotFun(CallContext c) {
        Val arg = c.arg(0);
        if (arg == vm.bool.trueVal) {
            return vm.bool.falseVal;
        } else if (arg == vm.bool.falseVal) {
            return vm.bool.trueVal;
        } else {
            return c.call(vm.graph.raiseFormat(
                        "op_lognot(Bool): Bool must be bool, but got {}",
                        vm.graph.repr(arg)));
        }
    }

    // }}}

    // op_logor {{{

    /**
     * Implementation of op_logor.
     */
    private HostResult opLogOrFun(CallContext c) {
        Val first = c.arg(0);
        Val second = c.arg(1);

        if (! vm.bool.isBool(first)) {
            return c.call(vm.graph.raiseFormat(
                        "op_logor(Bool $thunk): Bool must be bool, but got {}",
                        vm.graph.repr(first)));
        }

        if (! (second instanceof FunVal)) {
            return c.call(vm.graph.raiseFormat(
                        "op_logor(Bool $thunk): $thunk must be fun, but got {}",
                        vm.graph.repr(second)));
        }

        return first == vm.bool.trueVal
            ? vm.bool.trueVal
            : c.call((FunVal) second);
    }

    // }}}

    // op_logand {{{

    /**
     * Implementation of op_logand.
     */
    private HostResult opLogAndFun(CallContext c) {
        Val first = c.arg(0);
        Val second = c.arg(1);

        if (! vm.bool.isBool(first)) {
            return c.call(vm.graph.raiseFormat(
                        "op_logand(Bool $thunk): Bool must be bool, but got {}",
                        vm.graph.repr(first)));
        }

        if (! (second instanceof FunVal)) {
            return c.call(vm.graph.raiseFormat(
                        "op_logand(Bool $thunk): $thunk must be fun, but got {}",
                        vm.graph.repr(second)));
        }

        return first == vm.bool.trueVal
            ? c.call((FunVal) second)
            : vm.bool.falseVal;
    }

    // }}}

    // if {{{

    /**
     * Implementation of if.
     */
    private HostResult ifFun(CallContext c) {
        Val bool = c.arg(0);
        Val trueCont = c.arg(1);

        if (! vm.bool.isBool(bool)) {
            return c.call(IfSupport.condNotBool(vm, bool));
        }

        if (! (trueCont instanceof FunVal)) {
            return c.call(vm.graph.raiseFormat(
                        "if(Bool $true_cont ...[$false_cont={{}}]):"
                        + " $true_cont must be a fun, but got {}",
                        vm.graph.repr(trueCont)));
        }

        if (c.argCount() == 2) {
            return bool == vm.bool.trueVal
                ? c.call((FunVal) trueCont)
                : vm.nada;
        } else {
            Val falseCont = c.arg(2);
            if (! (falseCont instanceof FunVal)) {
                return c.call(vm.graph.raiseFormat(
                            "if(Bool $true_cont ...[$false_cont={{}}]):"
                            + " $false_cont must be a fun, but got {}",
                            vm.graph.repr(falseCont)));
            }

            return c.call((FunVal) (bool == vm.bool.trueVal ? trueCont : falseCont));
        }
    }

    // }}}

    // branch(...[$cond1 $then1 $cond2 $then2,,,]) {{{1

    /** The description of branch. */
    private static final String BRANCH_DESC = "branch(...[$cond1 $then1 $cond2 $then2 ,,,])";

    /**
     * Implementation of branch.
     */
    private HostResult branchFun(CallContext c) {
        int argCount = c.argCount();
        List<Val> args = new ArrayList<>(argCount);
        for (int i = 0; i < argCount; ++ i) {
            args.add(c.arg(i));
        }

        if (argCount % 2 != 0) {
            return c.call(vm.graph.raiseFormat("{}: odd number of args: {}",
                        vm.graph.of(vm.str.of(BRANCH_DESC)),
                        vm.graph.repr(vm.num.of(argCount))));
        }
        for (int i = 0; i < argCount; i += 2) {
            if (! (args.get(i) instanceof FunVal)) {
                String param = "$cond" + (i / 2 + 1);
                return c.call(vm.graph.raiseFormat("{}: {} must be fun, but got {}",
                            vm.graph.of(vm.str.of(BRANCH_DESC)),
                            vm.graph.of(vm.str.of(param)),
                            vm.graph.repr(args.get(i))));
            }
            if (! (c.arg(i + 1) instanceof FunVal)) {
                String param = "$then" + (i / 2 + 1);
                return c.call(vm.graph.raiseFormat("{}: {} must be fun, but got {}",
                            vm.graph.of(vm.str.of(BRANCH_DESC)),
                            vm.graph.of(vm.str.of(param)),
                            vm.graph.repr(args.get(i + 1))));
            }
        }

        return branchLoop(c, args, 0);
    }

    /**
     * Loop of branch.
     */
    private HostResult branchLoop(HostContext c, List<Val> args, int index) {
        if (index >= args.size()) {
            return c.call(BranchSupport.noMatchingCond(vm));
        }
        FunVal cond = (FunVal) args.get(index);
        return c.call(cond).on((cc, b) -> {
            if (! vm.bool.isBool(b)) {
                return cc.call(BranchSupport.condNotBool(vm, index, b));
            } else if (b.equals(vm.bool.trueVal)) {
                FunVal body = (FunVal) args.get(index + 1);
                return cc.call(body);
            } else {
                return branchLoop(cc, args, index + 2);
            }
        });
    }

    // }}}1

}

// vim: et sw=4 sts=4 fdm=marker
