package org.kink_lang.kink.hostfun.graph;

import org.kink_lang.kink.FunVal;
import org.kink_lang.kink.Val;

/**
 * The facade interface of the Execution Graph DSL.
 *
 * <p>The facade provides factory methods of graph nodes of the Execution Graph DSL.</p>
 *
 * <p>CAUTION: This interface is not expected to be implemented by external programs.
 * Therefore, abstract methods can be added to this interface without prior announcements.</p>
 *
 * <p>See {@link org.kink_lang.kink.hostfun.graph} for usage.</p>
 *
 * @see org.kink_lang.kink.Vm#graph
 */
public interface GraphFacade {

    /**
     * Returns a graph node which emits the val.
     *
     * @param val the val to be emitted.
     * @return the graph node which emits the val.
     */
    public abstract GraphNode of(Val val);

    /**
     * Returns a graph node which calls the fun.
     *
     * @param fun the fun to call.
     * @return the call graph node which calls the fun.
     */
    public abstract CallGraphNodeToRecv call(FunVal fun);

    /**
     * Returns a call graph node which calls a method in the owner.
     *
     * @param owner the owner of the method.
     * @param symHandle the sym handle of the method fun.
     * @return the call graph node which calls a method in the owner.
     */
    public abstract CallGraphNodeToRecv call(GraphNode owner, int symHandle);

    /**
     * Returns a call graph node which calls a fun in the mod.
     *
     * @param modName the name of the mod.
     * @param symHandle the sym handle of the fun.
     * @return the call graph node which calls a fun in the mod.
     */
    public abstract CallGraphNodeToArgs call(String modName, int symHandle);

    /**
     * Convenient method to call val.repr.
     *
     * @param val the val for which the .repr method is called.
     * @return the execution graph to call val.repr.
     */
    public abstract GraphNode repr(Val val);

    /**
     * Convenient method to call the Kink method template.format(...args).
     *
     * @param template the template for Str.format method, such as "Catch {}".
     * @param args the args for Str.format method.
     * @return the execution graph to call template.format(...args).
     */
    public abstract GraphNode format(String template, GraphNode... args);

    /**
     * Convenient method to call raise(template.format(...args)).
     *
     * @param template the template for Str.format method, such as "expected a num, but got {}".
     * @param args the args for Str.format method.
     * @return the execution graph to call raise(template.format(...args)).
     */
    public abstract GraphNode raiseFormat(String template, GraphNode... args);

}

// vim: et sw=4 sts=4 fdm=marker
