package org.kink_lang.kink.internal.compile;

import java.lang.invoke.MethodHandles.Lookup;
import java.lang.invoke.MethodType;

import org.kink_lang.kink.BindingVal;
import org.kink_lang.kink.FunVal;
import org.kink_lang.kink.Val;
import org.kink_lang.kink.Vm;
import org.kink_lang.kink.internal.compile.classgen.ClassGenerator;
import org.kink_lang.kink.internal.compile.javaclassir.TopLevelCompiler;
import org.kink_lang.kink.internal.control.Control;
import org.kink_lang.kink.internal.program.itree.Itree;

/**
 * Compiles itree to FunVal by runtime bytecode generation.
 */
public class ItreeCompiler {

    /** The vm. */
    private final Vm vm;

    /** Lookup of org.kink_lang.kink package. */
    private final Lookup packageLookup;

    /** The name of the program. */
    private final String programName;

    /** The text of the program. */
    private final String programText;

    /** The binding of the top level fun. */
    private final BindingVal binding;

    /**
     * Constructs a compiler.
     *
     * @param vm the vm.
     * @param packageLookup lookup of org.kink_lang.kink package.
     * @param programName the name of the program.
     * @param programText the text of the program.
     * @param binding the binding of the top level fun.
     */
    public ItreeCompiler(
            Vm vm,
            Lookup packageLookup,
            String programName,
            String programText,
            BindingVal binding) {
        this.vm = vm;
        this.packageLookup = packageLookup;
        this.programName = programName;
        this.programText = programText;
        this.binding = binding;
    }

    /**
     * Returns a generated toplevel fun.
     *
     * @param topLevel the top of the top level of the program.
     * @return a generated toplevel fun.
     */
    public FunVal compile(Itree topLevel) {
        var topLevelCompiler = new TopLevelCompiler(vm, programName, programText);
        var jcir = topLevelCompiler.compile(topLevel);
        return Control.runWrappingThrowable(() -> {
            var classGen = new ClassGenerator(vm, packageLookup);
            var generated = classGen.generate(jcir);
            var mh = generated.findConstructor(generated.lookupClass(),
                    MethodType.methodType(void.class, Vm.class, Val.class));
            return (FunVal) mh.invoke(vm, this.binding);
        });
    }

}

// vim: et sw=4 sts=4 fdm=marker
