package org.kink_lang.kink.internal.compile.classgen;

import java.util.List;
import java.util.Map;

import org.objectweb.asm.ClassWriter;
import org.objectweb.asm.FieldVisitor;
import org.objectweb.asm.Opcodes;
import org.objectweb.asm.Type;
import org.objectweb.asm.commons.GeneratorAdapter;
import org.objectweb.asm.commons.Method;

import org.kink_lang.kink.Val;
import org.kink_lang.kink.Vm;
import org.kink_lang.kink.internal.callstack.Trace;
import org.kink_lang.kink.internal.compile.javaclassir.BytecodeGenState;
import org.kink_lang.kink.internal.compile.javaclassir.Insn;
import org.kink_lang.kink.internal.compile.javaclassir.JavaClassIr;

/**
 * Generates bytecode of a fun.
 */
public class BytecodeGenerator {

    /** The base class name. */
    private static final String BASE_CLASS_NAME = "org/kink_lang/kink/GeneratedFunValBase";

    /**
     * Generates a bytecode of a fun.
     *
     * @param jcir the Java class IR.
     * @return the bytecode of the class to generate.
     */
    public byte[] generate(JavaClassIr jcir) {
        ClassWriter cw = new ClassWriter(ClassWriter.COMPUTE_FRAMES);
        cw.visit(Opcodes.V17, 0,
                JavaClassIr.TYPE_BASE.getInternalName(),
                null, BASE_CLASS_NAME, null);

        genTraceMapField(cw);
        genVmStaticField(cw);
        genChildJcirFactoriesField(cw);
        genValFields(cw, jcir.valFieldCount());
        genConstructor(cw, jcir.valFieldCount());
        genDoResume(cw, jcir.doResumeInsns());
        genDesc(cw, jcir.desc());
        genDataStackUsageUpperBound(cw, jcir.dataStackUsageUpperBound());
        genTrace(cw);

        cw.visitEnd();
        return cw.toByteArray();
    }

    /**
     * Generates traceMap field.
     */
    private void genTraceMapField(ClassWriter cw) {
        FieldVisitor fv = cw.visitField(
                Opcodes.ACC_PRIVATE | Opcodes.ACC_STATIC | Opcodes.ACC_VOLATILE,
                "traceMap",
                Type.getDescriptor(Map.class),
                null,
                null);
        fv.visitEnd();
    }

    /**
     * Generates childJcirFactories field.
     */
    private void genChildJcirFactoriesField(ClassWriter cw) {
        FieldVisitor fv = cw.visitField(
                Opcodes.ACC_PUBLIC | Opcodes.ACC_STATIC | Opcodes.ACC_VOLATILE,
                "childJcirFactories",
                Type.getDescriptor(List.class),
                null,
                null);
        fv.visitEnd();
    }

    /**
     * Generates vmStatic field.
     */
    private void genVmStaticField(ClassWriter cw) {
        FieldVisitor fv = cw.visitField(
                Opcodes.ACC_PRIVATE | Opcodes.ACC_STATIC | Opcodes.ACC_VOLATILE,
                "vmStatic",
                Type.getDescriptor(Vm.class),
                null,
                null);
        fv.visitEnd();
    }

    /**
     * Generates val fields.
     */
    private void genValFields(ClassWriter cw, int valFieldCount) {
        for (int i = 0; i < valFieldCount; ++ i) {
            FieldVisitor fv = cw.visitField(
                    Opcodes.ACC_PRIVATE | Opcodes.ACC_FINAL,
                    valFieldName(i),
                    Type.getDescriptor(Val.class),
                    null,
                    null);
            fv.visitEnd();
        }
    }

    /**
     * Generates the constructor.
     */
    private void genConstructor(ClassWriter cw, int valFieldCount) {
        Type[] paramTypes = new Type[valFieldCount + 1];
        paramTypes[0] = Type.getType(Vm.class);
        for (int i = 0; i < valFieldCount; ++ i) {
            paramTypes[i + 1] = Type.getType(Val.class);
        }

        Method m = new Method("<init>", Type.VOID_TYPE, paramTypes);
        GeneratorAdapter mg = new GeneratorAdapter(Opcodes.ACC_PRIVATE, m, null, null, cw);

        mg.loadThis();
        mg.loadArg(0);
        mg.invokeConstructor(
                Type.getType("L" + BASE_CLASS_NAME + ";"),
                new Method("<init>", Type.VOID_TYPE, new Type[] { Type.getType(Vm.class) }));
        for (int i = 0; i < valFieldCount; ++ i) {
            mg.loadThis();
            mg.loadArg(i + 1);
            mg.putField(JavaClassIr.TYPE_BASE,
                    valFieldName(i),
                    Type.getType(Val.class));
        }
        mg.returnValue();

        mg.endMethod();
    }

    /**
     * Name of the val field for the index.
     */
    private static String valFieldName(int index) {
        return "valField" + index;
    }

    /**
     * Generates doResume method.
     */
    private void genDoResume(ClassWriter cw, List<Insn> doResumeInsns) {
        Method m = new Method("doResume", Type.VOID_TYPE, new Type[] {
            Type.getType("Lorg/kink_lang/kink/StackMachine;"),
            Type.getType("Lorg/kink_lang/kink/Val;"),
            Type.INT_TYPE,
            Type.getType("Lorg/kink_lang/kink/internal/callstack/CallStack;"),
            Type.getType("Lorg/kink_lang/kink/DataStack;"),
            Type.INT_TYPE,
        });
        GeneratorAdapter mg = new GeneratorAdapter(Opcodes.ACC_PUBLIC, m, null, null, cw);

        var state = new BytecodeGenState(mg, doResumeInsns);
        doResumeInsns.forEach(insn -> insn.generateBytecodeSnippet(state));

        mg.endMethod();
    }

    /**
     * Generates desc method.
     */
    private void genDesc(ClassWriter cw, String desc) {
        Method m = new Method("desc", Type.getType(String.class), new Type[0]);
        GeneratorAdapter mg = new GeneratorAdapter(Opcodes.ACC_PUBLIC, m, null, null, cw);

        mg.push(desc);
        mg.returnValue();
        mg.endMethod();
    }

    /**
     * Generates dataStackUsageUpperBound method.
     */
    private void genDataStackUsageUpperBound(ClassWriter cw, int dataStackUsageUpperBound) {
        Method m = new Method("dataStackUsageUpperBound", Type.getType(int.class), new Type[0]);
        GeneratorAdapter mg = new GeneratorAdapter(Opcodes.ACC_PUBLIC, m, null, null, cw);

        mg.push(dataStackUsageUpperBound);
        mg.returnValue();
        mg.endMethod();
    }

    /**
     * Generates trace method.
     */
    private void genTrace(ClassWriter cw) {
        Method m = new Method("trace",
                Type.getType(Trace.class),
                new Type[] { Type.getType(int.class) });
        GeneratorAdapter mg = new GeneratorAdapter(Opcodes.ACC_PUBLIC, m, null, null, cw);

        mg.getStatic(
                JavaClassIr.TYPE_BASE,
                "traceMap",
                Type.getType(Map.class));
        mg.loadArg(0);
        mg.box(Type.getType(int.class));
        mg.invokeInterface(
                Type.getType(Map.class),
                new Method(
                    "get", Type.getType(Object.class), new Type[] { Type.getType(Object.class) }));
        mg.checkCast(Type.getType(Trace.class));
        mg.returnValue();
        mg.endMethod();
    }

}

// vim: et sw=4 sts=4 fdm=marker
