package org.kink_lang.kink.internal.compile.javaclassir;

import java.lang.invoke.MethodType;
import java.util.ArrayList;
import java.util.List;

import org.objectweb.asm.Type;
import org.objectweb.asm.commons.Method;

import org.kink_lang.kink.internal.callstack.Location;
import org.kink_lang.kink.internal.callstack.Trace;
import org.kink_lang.kink.internal.program.itree.LocalVar;

/**
 * Generates insns to access local vars.
 */
public interface LvarAccessGenerator {

    /**
     * Generates insns to load a local var, checking absence.
     *
     * @param lvar the local var.
     * @param loc the location of the dereference.
     * @return insns to load a local var.
     */
    List<Insn> loadLvar(LocalVar lvar, Location loc);

    /**
     * Generates insns to load a local var, not checking absence.
     *
     * @param lvar the local var.
     * @return insns to load a local var.
     */
    List<Insn> loadLvarAllowNull(LocalVar lvar);

    /**
     * Generates insns to store a local var.
     *
     * @param lvar the local var.
     * @return insns to store a local var.
     */
    List<Insn> storeLvar(LocalVar lvar);

    /**
     * Generates insns to pass the recv.
     *
     * @param lvar the local var.
     * @return insns to pass recv.
     */
    List<Insn> passRecv(LocalVar lvar);

    /**
     * Generates insns to pass the specified arg.
     *
     * @param lvar the local var.
     * @param argIndex the index of the arg.
     * @return insns to pass arg.
     */
    List<Insn> passArg(LocalVar lvar, int argIndex);

    /**
     * Returns whether {@code lvar} is unused.
     *
     * @param lvar local var to test.
     * @return whether lvar is unused.
     */
    boolean isUnused(LocalVar lvar);


    /**
     * If contParam is null, transition to raise and return.
     *
     * @param lvarName the name of the local var.
     * @param loc the location of dereference.
     * @param keySup key str supplier.
     * @param traceAccum trace accumulator.
     * @return insns.
     */
    static List<Insn> checkNull(
            String lvarName,
            Location loc,
            KeyStrSupplier keySup,
            TraceAccumulator traceAccum) {
        List<Insn> insns = new ArrayList<>();

        // if contParam != null; goto #nonnull
        insns.add(InsnsGenerator.LOAD_CONTPARAM);
        String nonnullLabel = keySup.newKeyStr("nonnull");
        insns.add(new Insn.IfNonNull(nonnullLabel));

        // stackMachine.transitionToRaiseOn(msg, trace); return
        insns.add(InsnsGenerator.LOAD_STACKMACHINE);
        insns.add(new Insn.PushString("no such var: " + lvarName));
        int traceKey = traceAccum.add(Trace.of(loc));
        insns.add(new Insn.InvokeDynamic(
                    MethodType.methodType(Trace.class),
                    InsnsGenerator.BOOTSTRAP_TRACE_HANDLE,
                    List.of(traceKey)));
        insns.add(new Insn.InvokeVirtual(Type.getType("Lorg/kink_lang/kink/StackMachine;"),
                    new Method("transitionToRaiseOn",
                        Type.VOID_TYPE,
                        new Type[] {
                            Type.getType(String.class),
                            Type.getType(Trace.class) })));
        insns.add(new Insn.ReturnValue());

        // #nonnull:
        insns.add(new Insn.Mark(nonnullLabel));

        return insns;
    }

}

// vim: et sw=4 sts=4 fdm=marker
