package org.kink_lang.kink.internal.compile.javaclassir;

import java.lang.invoke.CallSite;
import java.lang.invoke.MethodHandles.Lookup;
import java.lang.invoke.MethodType;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;
import java.util.function.BiFunction;
import java.util.stream.IntStream;

import org.objectweb.asm.Handle;
import org.objectweb.asm.Opcodes;
import org.objectweb.asm.Type;
import org.objectweb.asm.commons.Method;

import org.kink_lang.kink.FunVal;
import org.kink_lang.kink.SharedVars;
import org.kink_lang.kink.SharedVarsFactory;
import org.kink_lang.kink.Val;
import org.kink_lang.kink.VecVal;
import org.kink_lang.kink.Vm;
import org.kink_lang.kink.internal.callstack.FakeCallTraceCse;
import org.kink_lang.kink.internal.callstack.Location;
import org.kink_lang.kink.internal.callstack.Trace;
import org.kink_lang.kink.internal.compile.bootstrap.ConstBootstrapper;
import org.kink_lang.kink.internal.compile.tempval.MaybeTraitVal;
import org.kink_lang.kink.internal.intrinsicsupport.NewValSupport;
import org.kink_lang.kink.internal.ovis.OwnVarIndexes;
import org.kink_lang.kink.internal.program.itree.*;
import org.kink_lang.kink.internal.vec.MaybeTrait;
import org.kink_lang.kink.internal.vec.TraitError;
import org.kink_lang.kink.internal.vec.TraitVecInternal;

/**
 * Common parts of new_val.
 */
public final class NewVal {

    /**
     * Cannot instantiate the class.
     */
    private NewVal() {
    }

    /** Insn of invoking VecVal.getTrait. */
    static final Insn INVOKE_GET_TRAIT_OF_VEC_VAL = new Insn.InvokeVirtual(
            Type.getType(VecVal.class),
            new Method("getTrait", Type.getType(MaybeTrait.class), new Type[0]));

    /** Insn of calling constructor of MaybeTraitVal. */
    static final Insn INVOKE_MAYBE_TRAIT_VAL_CONSTRUCTOR = new Insn.InvokeConstructor(
            Type.getType(MaybeTraitVal.class),
            new Method("<init>", Type.VOID_TYPE,
                new Type[] {
                    Type.getType(Vm.class),
                    Type.getType(MaybeTrait.class) }));

    /** Insn of calling constructor of Val. */
    static final Insn INVOKE_VAL_CONSTRUCTOR = new Insn.InvokeConstructor(
            Type.getType(Val.class),
            new Method("<init>", Type.VOID_TYPE,
                new Type[] {
                    Type.getType(Vm.class),
                    Type.getType(SharedVars.class),
                    Type.getType(OwnVarIndexes.class),
                    Type.getType(Val.class),
                    Type.getType(Val.class),
                    Type.getType(Val.class),
                    Type.getType(Val.class),
                    Type.getType(Val.class),
                    Type.getType(Val.class),
                    Type.getType(Val[].class) }));

    /** Insn of invoking MaybeTrait.getMaybeTrait. */
    static final Insn INVOKE_GET_MAYBE_TRAIT_OF_TEMPVAL = new Insn.InvokeVirtual(
            Type.getType(MaybeTraitVal.class),
            new Method("getMaybeTrait", Type.getType(MaybeTrait.class), new Type[0]));

    /** Handler of ConstBootstrapper.bootstrapCoreFun. */
    static final Handle BOOTSTRAP_PRELOADED_HANDLE = new Handle(Opcodes.H_INVOKESTATIC,
            Type.getType(ConstBootstrapper.class).getInternalName(),
            "bootstrapPreloaded",
            MethodType.methodType(
                CallSite.class,
                Lookup.class,
                String.class,
                MethodType.class,
                String.class).descriptorString(),
            false);

    /** Handle of OvisBootstrapper.bootstrapOvis. */
    static final Handle BOOTSTRAP_OVIS_HANDLE = new Handle(Opcodes.H_INVOKESTATIC,
            "org/kink_lang/kink/OvisBootstrapper",
            "bootstrapOvis",
            Type.getMethodDescriptor(
                Type.getType(CallSite.class),
                Type.getType(Lookup.class),
                Type.getType(String.class),
                Type.getType(MethodType.class),
                Type.getType(int[].class)),
            false);

    /** The number of val fields in Val class. */
    private static final int VAL_FIELDS = 6;

    /**
     * Common part of new_val with trait.
     */
    static List<Insn> traitNewValCommonPart(
            Vm vm,
            TraitNewValItree itree,
            BiFunction<Itree, ResultContext, List<Insn>> generate,
            ResultContext resultCtx,
            String programName,
            String programText,
            KeyStrSupplier keySup,
            TraceAccumulator traceAccum) {
        List<Insn> insns = new ArrayList<>();

        insns.addAll(loadTraitVec(itree, generate, programName, programText, keySup, traceAccum));
        List<List<Insn>> insnsForVals = makeInsnsForVals(itree.symValPairs(), generate);
        String sharedVars = keySup.newKeyStr("sharedVars");
        List<String> argSyms = makeArgSyms(itree.symValPairs());
        List<String> ovisSyms = makeOvisSyms(argSyms);
        List<String> vlocals = makeVlocals(keySup);
        String moreVars = keySup.newKeyStr("moreVars");

        if (containsCall(insnsForVals)) {
            insns.addAll(PUSH_MAYBE_TRAIT);
            insns.addAll(pushVals(insnsForVals));
            String topOffset = keySup.newKeyStr("topOffset");
            insns.addAll(getTopOffset(topOffset));
            String maybeTrait = keySup.newKeyStr("maybeTrait");
            insns.addAll(maybeTraitFromStack(topOffset, maybeTrait, itree.symValPairs().size()));
            insns.addAll(sharedVarsFromMaybeTrait(vm, itree, resultCtx, programName, programText,
                        maybeTrait, sharedVars, keySup, traceAccum));
            insns.addAll(loadVsFromStack(vlocals, topOffset, argSyms, ovisSyms));
            insns.addAll(makeMoreVarsFromStack(moreVars, topOffset, argSyms, ovisSyms));
            insns.addAll(clearDataStack(topOffset, argSyms.size() + 1));
        } else {
            String maybeTrait = keySup.newKeyStr("maybeTrait");
            insns.addAll(maybeTraitFromContParam(maybeTrait));
            insns.addAll(produceArrayForMoreVars(ovisSyms.size()));
            insns.add(storeMoreVars(moreVars));
            insns.addAll(calcValsUnstacked(insnsForVals, argSyms, ovisSyms, vlocals, moreVars));
            insns.addAll(nullifyRemainingVs(ovisSyms, vlocals));
            insns.addAll(sharedVarsFromMaybeTrait(vm, itree, resultCtx, programName, programText,
                        maybeTrait, sharedVars, keySup, traceAccum));
        }

        insns.addAll(invokeValConstructor(vm, ovisSyms, sharedVars, vlocals, moreVars));
        return insns;
    }

    /**
     * Load trait vec into contParam.
     */
    private static List<Insn> loadTraitVec(
            TraitNewValItree itree,
            BiFunction<Itree, ResultContext, List<Insn>> generate,
            String programName,
            String programText,
            KeyStrSupplier keySup,
            TraceAccumulator traceAccum) {
        List<Insn> insns = new ArrayList<>();

        // contParam = eval trait
        insns.addAll(generate.apply(itree.trait(), ResultContext.NON_TAIL));

        // if contParam instanceof VecVal; goto #trait-is-vec
        insns.add(InsnsGenerator.LOAD_CONTPARAM);
        insns.add(new Insn.InstanceOf(Type.getType(VecVal.class)));
        String traitIsVec = keySup.newKeyStr("trait-is-vec");
        insns.add(new Insn.IfNonZero(traitIsVec));

        // stackMachine.transitionToCannotSpread(trace, contParam); return
        insns.add(InsnsGenerator.LOAD_STACKMACHINE);
        Trace spreadTrace = Trace.of(new Location(programName, programText, itree.spreadPos()));
        insns.add(new Insn.InvokeDynamic(
                        MethodType.methodType(Trace.class),
                        InsnsGenerator.BOOTSTRAP_TRACE_HANDLE,
                        List.of(traceAccum.add(spreadTrace))));
        insns.add(InsnsGenerator.LOAD_CONTPARAM);
        insns.add(new Insn.InvokeVirtual(InsnsGenerator.STACKMACHINE_TYPE,
                    new Method("transitionToCannotSpread", Type.VOID_TYPE,
                        new Type[] { Type.getType(Trace.class), Type.getType(Val.class) })));
        insns.add(new Insn.ReturnValue());

        // #trait-is-vec:
        insns.add(new Insn.Mark(traitIsVec));

        return insns;
    }

    // dataStack.push(new MaybeTraitVal(vm, ((VecVal) contParam).getTrait()))
    /** Push temp val of maybe-trait made of contParam. */
    private static final List<Insn> PUSH_MAYBE_TRAIT = List.of(
            InsnsGenerator.LOAD_DATASTACK,
            new Insn.NewInstance(Type.getType(MaybeTraitVal.class)),
            new Insn.Dup(),
            new Insn.LoadThis(),
            new Insn.GetField(JavaClassIr.TYPE_BASE, "vm", Type.getType(Vm.class)),
            InsnsGenerator.LOAD_CONTPARAM,
            new Insn.CheckCast(Type.getType(VecVal.class)),
            INVOKE_GET_TRAIT_OF_VEC_VAL,
            INVOKE_MAYBE_TRAIT_VAL_CONSTRUCTOR,
            InsnsGenerator.INVOKE_PUSH_TO_DATASTACK);

    /**
     * Get maybeTrait from stack.
     */
    private static final List<Insn> maybeTraitFromStack(
            String topOffset, String maybeTrait, int numPairs) {
        // #maybeTrait =
        // ((MaybeTraitVal) dataStack.getAtOffset(#topOffset - (numPairs + 1))).getMaybeTrait()
        List<Insn> insns = new ArrayList<>();
        insns.add(InsnsGenerator.LOAD_DATASTACK);
        insns.add(new Insn.LoadLocal(topOffset));
        insns.add(new Insn.PushInt(numPairs + 1));
        insns.add(new Insn.SubInt());
        insns.add(InsnsGenerator.INVOKE_AT_OFFSET);
        insns.add(new Insn.CheckCast(Type.getType(MaybeTraitVal.class)));
        insns.add(INVOKE_GET_MAYBE_TRAIT_OF_TEMPVAL);
        insns.add(new Insn.StoreNewLocal(maybeTrait, Type.getType(MaybeTrait.class)));
        return insns;
    }

    /**
     * Get maybeTrait from contParam.
     */
    private static List<Insn> maybeTraitFromContParam(String maybeTrait) {
        // #maybeTrait = ((VecVal) contParam).getTrait()
        return List.of(
                InsnsGenerator.LOAD_CONTPARAM,
                new Insn.CheckCast(Type.getType(VecVal.class)),
                INVOKE_GET_TRAIT_OF_VEC_VAL,
                new Insn.StoreNewLocal(maybeTrait, Type.getType(MaybeTrait.class)));
    }

    /**
     * Make #sharedVars from #maybeTrait.
     */
    private static final List<Insn> sharedVarsFromMaybeTrait(
            Vm vm,
            TraitNewValItree itree,
            ResultContext resultCtx,
            String programName,
            String programText,
            String maybeTrait,
            String sharedVars,
            KeyStrSupplier keySup,
            TraceAccumulator traceAccum) {
        List<Insn> insns = new ArrayList<>();

        // if #maybeTrait instanceof TraitVecInternal; goto #is-trait
        insns.add(new Insn.LoadLocal(maybeTrait));
        insns.add(new Insn.InstanceOf(Type.getType(TraitVecInternal.class)));
        String isTrait = keySup.newKeyStr("is-trait");
        insns.add(new Insn.IfNonZero(isTrait));

        // callStack.pushCse(fakeCallCse, 0, 0, 0)
        insns.add(InsnsGenerator.LOAD_CALLSTACK);
        Trace callTrace = resultCtx.onTailOrNot(Trace.of(
                    vm.sym.handleFor("new_val"),
                    new Location(programName, programText, itree.pos())));
        insns.add(new Insn.InvokeDynamic(
                    MethodType.methodType(FakeCallTraceCse.class),
                    InsnsGenerator.BOOTSTRAP_FAKE_CALL_CSE,
                    List.of(traceAccum.add(callTrace))));
        insns.add(new Insn.PushInt(0));
        insns.add(new Insn.PushInt(0));
        insns.add(new Insn.PushInt(0));
        insns.add(InsnsGenerator.INVOKE_PUSH_CSE);

        // dataStack.removeFromOffset(0)
        insns.add(InsnsGenerator.LOAD_DATASTACK);
        insns.add(new Insn.PushInt(0));
        insns.add(new Insn.InvokeVirtual(InsnsGenerator.DATASTACK_TYPE,
                    new Method("removeFromOffset", Type.VOID_TYPE,
                        new Type[] { Type.INT_TYPE })));

        // dataStack.push(nada)
        insns.add(InsnsGenerator.LOAD_DATASTACK);
        insns.add(InsnsGenerator.PRODUCE_NADA);
        insns.add(InsnsGenerator.INVOKE_PUSH_TO_DATASTACK);

        // stackMachine.transitionToCall(
        //  NewValFuns.traitError(vm, (TraitError) #maybeTrait, 6)): return
        insns.add(InsnsGenerator.LOAD_STACKMACHINE);
        insns.add(new Insn.LoadThis());
        insns.add(new Insn.GetField(JavaClassIr.TYPE_BASE, "vm", Type.getType(Vm.class)));
        insns.add(new Insn.LoadLocal(maybeTrait));
        insns.add(new Insn.CheckCast(Type.getType(TraitError.class)));
        insns.add(new Insn.PushInt(itree.symValPairs().size() * 2));
        insns.add(new Insn.InvokeStatic(Type.getType(NewValSupport.class),
                    new Method("traitError", Type.getType(FunVal.class),
                        new Type[] {
                            Type.getType(Vm.class),
                            Type.getType(TraitError.class),
                            Type.INT_TYPE })));
        insns.add(new Insn.InvokeVirtual(InsnsGenerator.STACKMACHINE_TYPE,
                    new Method("transitionToCall", Type.VOID_TYPE,
                        new Type[] { Type.getType(FunVal.class) })));
        insns.add(new Insn.ReturnValue());

        // #is-trait:
        insns.add(new Insn.Mark(isTrait));

        // #sharedVars = ((TraitVecInternal) #maybeTrait).getTraitVars()
        insns.add(new Insn.LoadLocal(maybeTrait));
        insns.add(new Insn.CheckCast(Type.getType(TraitVecInternal.class)));
        insns.add(new Insn.InvokeVirtual(Type.getType(TraitVecInternal.class),
                    new Method("getTraitVars", Type.getType(SharedVars.class), new Type[0])));
        insns.add(new Insn.StoreNewLocal(sharedVars, Type.getType(SharedVars.class)));

        return insns;
    }

    /**
     * Common part of new_val without trait.
     */
    static List<Insn> noTraitNewValCommonPart(
            Vm vm,
            NoTraitNewValItree itree,
            BiFunction<Itree, ResultContext, List<Insn>> generate,
            KeyStrSupplier keySup) {
        List<Insn> insns = new ArrayList<>();
        List<List<Insn>> insnsForVals = makeInsnsForVals(itree.symValPairs(), generate);
        List<String> argSyms = makeArgSyms(itree.symValPairs());
        List<String> ovisSyms = makeOvisSyms(argSyms);
        String sharedVars = keySup.newKeyStr("sharedVars");
        List<String> vlocals = makeVlocals(keySup);
        String moreVars = keySup.newKeyStr("moreVars");

        if (containsCall(insnsForVals)) {
            insns.addAll(pushVals(insnsForVals));
            String topOffset = keySup.newKeyStr("topOffset");
            insns.addAll(getTopOffset(topOffset));
            insns.addAll(loadMinimalSharedVars(sharedVars));
            insns.addAll(loadVsFromStack(vlocals, topOffset, argSyms, ovisSyms));
            insns.addAll(makeMoreVarsFromStack(moreVars, topOffset, argSyms, ovisSyms));
            insns.addAll(clearDataStack(topOffset, argSyms.size()));
        } else {
            insns.addAll(loadMinimalSharedVars(sharedVars));
            insns.addAll(produceArrayForMoreVars(ovisSyms.size()));
            insns.add(storeMoreVars(moreVars));
            insns.addAll(calcValsUnstacked(insnsForVals, argSyms, ovisSyms, vlocals, moreVars));
            insns.addAll(nullifyRemainingVs(ovisSyms, vlocals));
        }

        insns.addAll(invokeValConstructor(vm, ovisSyms, sharedVars, vlocals, moreVars));
        return insns;
    }

    /**
     * List of insns for each val in the order of the args.
     */
    private static List<List<Insn>> makeInsnsForVals(
            List<SymValPair> symValPairs,
            BiFunction<Itree, ResultContext, List<Insn>> generate) {
        return symValPairs.stream()
            .map(SymValPair::val)
            .map(val -> generate.apply(val, ResultContext.NON_TAIL))
            .toList();
    }

    /**
     * List of syms in the order of the args.
     */
    private static List<String> makeArgSyms(List<SymValPair> symValPairs) {
        return symValPairs.stream()
            .map(SymValPair::sym)
            .toList();
    }

    /**
     * List of unique syms for OVIS.
     */
    private static List<String> makeOvisSyms(List<String> argSyms) {
        Set<String> ovisSyms = new TreeSet<>(argSyms);
        return ovisSyms.stream().toList();
    }

    /**
     * Key strs for local vars of vals.
     */
    private static List<String> makeVlocals(KeyStrSupplier keySup) {
        return IntStream.range(0, VAL_FIELDS)
            .mapToObj(vi -> keySup.newKeyStr("v" + vi))
            .toList();
    }

    /**
     * Whether the insns contains non-tail call.
     */
    private static boolean containsCall(Collection<List<Insn>> insnsForVals) {
        return insnsForVals.stream()
            .anyMatch(is -> is.contains(InsnsGenerator.INVOKE_CALL_FUN));
    }

    /**
     * More-vars array to the local var.
     */
    private static Insn storeMoreVars(String moreVars) {
        return new Insn.StoreNewLocal(moreVars, Type.getType(Val[].class));
    }

    /**
     * Insns to produce produce an array for #moreVars.
     *
     * @param attrCount the count of attributes of the new val.
     * @return the insns.
     */
    private static List<Insn> produceArrayForMoreVars(int attrCount) {
        return attrCount <= VAL_FIELDS
            ? List.of(new Insn.GetStatic(
                        Type.getType(Val.class),
                        "EMPTY_VS",
                        Type.getType(Val[].class)))
            : List.of(
                    new Insn.PushInt(attrCount - VAL_FIELDS),
                    new Insn.NewArray(Type.getType(Val.class)));
    }

    /**
     * Insns to push vals to the data stack.
     *
     * @param insnsForVals the insns to produce each val.
     * @return the insns.
     */
    private static List<Insn> pushVals(List<List<Insn>> insnsForVals) {
        return insnsForVals.stream()
            .flatMap(valInsns -> {
                // contParam = val
                List<Insn> insns = new ArrayList<>(valInsns);
                // dataStack.push(contParam)
                insns.add(InsnsGenerator.LOAD_DATASTACK);
                insns.add(InsnsGenerator.LOAD_CONTPARAM);
                insns.add(InsnsGenerator.INVOKE_PUSH_TO_DATASTACK);
                return insns.stream();
            })
            .toList();
    }

    /**
     * Evaluate vals and store them into local vars or moreVars elements,
     * not using data stack.
     * This can be used when evaluation of vals does not contain non-tail call.
     */
    private static List<Insn> calcValsUnstacked(
            List<List<Insn>> insnsForVals,
            List<String> argSyms,
            List<String> ovisSyms,
            List<String> vlocals,
            String moreVars) {
        List<Insn> insns = new ArrayList<>();
        for (int pairInd = 0; pairInd < insnsForVals.size(); ++ pairInd) {
            insns.addAll(insnsForVals.get(pairInd));
            String sym = argSyms.get(pairInd);
            if (argSyms.lastIndexOf(sym) != pairInd) {
                continue;
            }

            int ovisInd = ovisSyms.indexOf(sym);
            if (ovisInd < VAL_FIELDS) {
                String vlocal = vlocals.get(ovisInd);
                insns.add(InsnsGenerator.LOAD_CONTPARAM);
                insns.add(new Insn.StoreNewLocal(vlocal, Type.getType(Val.class)));
            } else {
                insns.add(new Insn.LoadLocal(moreVars));
                insns.add(new Insn.PushInt(ovisInd - VAL_FIELDS));
                insns.add(InsnsGenerator.LOAD_CONTPARAM);
                insns.add(new Insn.ArrayStore(Type.getType(Val.class)));
            }
        }
        return insns;
    }

    /**
     * Nullify local vars which don't contain a val.
     */
    private static List<Insn> nullifyRemainingVs(List<String> ovisSyms, List<String> vlocals) {
        List<Insn> insns = new ArrayList<>();
        for (int vi = ovisSyms.size(); vi < VAL_FIELDS; ++ vi) {
            String vlocal = vlocals.get(vi);
            insns.add(new Insn.PushString(null));
            insns.add(new Insn.StoreNewLocal(vlocal, Type.getType(Val.class)));
        }
        return insns;
    }

    /**
     * Generates {@code #topOffset = dataStack.getTopOffset()}.
     *
     * @param topOffset the key str of the local var of #topOffset.
     * @return the insns.
     */
    private static List<Insn> getTopOffset(String topOffset) {
        List<Insn> insns = new ArrayList<>();
        insns.add(InsnsGenerator.LOAD_DATASTACK);
        insns.add(InsnsGenerator.INVOKE_TOP_OFFSET);
        insns.add(new Insn.StoreNewLocal(topOffset, Type.INT_TYPE));
        return insns;
    }

    /**
     * Generates {@code #sharedVars = vm.sharedVars.minimal}.
     *
     * @param sharedVars the key str of the local var of #sharedVars.
     * @return the insns.
     */
    private static List<Insn> loadMinimalSharedVars(String sharedVars) {
        List<Insn> insns = new ArrayList<>();
        insns.add(new Insn.LoadThis());
        insns.add(new Insn.GetField(JavaClassIr.TYPE_BASE, "vm", Type.getType(Vm.class)));
        insns.add(new Insn.GetField(Type.getType(Vm.class),
                    "sharedVars",
                    Type.getType(SharedVarsFactory.class)));
        insns.add(new Insn.GetField(Type.getType(SharedVarsFactory.class),
                    "minimal",
                    Type.getType(SharedVars.class)));
        insns.add(new Insn.StoreNewLocal(sharedVars, Type.getType(SharedVars.class)));
        return insns;
    }

    /**
     * Generates insns to load v0...v5 from the data stack.
     *
     * @param vlocals the key strs of the local vars of the vals.
     * @param topOffset the top offset key str.
     * @param argSyms the syms in the order of the args of new_val.
     * @param ovisSyms the uniq syms in the order of the ovis.
     * @return the insns.
     */
    private static List<Insn> loadVsFromStack(
            List<String> vlocals,
            String topOffset,
            List<String> argSyms,
            List<String> ovisSyms) {
        List<Insn> insns = new ArrayList<>();
        for (int vi = 0; vi < vlocals.size(); ++ vi) {
            if (vi < ovisSyms.size()) {
                String sym = ovisSyms.get(vi);
                int pairInd = argSyms.lastIndexOf(sym);
                // _ = dataStack.getAtOffset(#topOffset - (pairCount - pairInd))
                insns.add(InsnsGenerator.LOAD_DATASTACK);
                insns.add(new Insn.LoadLocal(topOffset));
                insns.add(new Insn.PushInt(argSyms.size() - pairInd));
                insns.add(new Insn.SubInt());
                insns.add(InsnsGenerator.INVOKE_AT_OFFSET);
            } else {
                // _ = null
                insns.add(new Insn.PushString(null));
            }
            // #vx = _
            insns.add(new Insn.StoreNewLocal(vlocals.get(vi), Type.getType(Val.class)));
        }
        return insns;
    }

    /**
     * Genrates insns to make #moreVars from the data stack.
     *
     * @param moreVars the key str of #moreVars local var.
     * @param topOffset the top offset key str.
     * @param argSyms the syms in the order of the args of new_val.
     * @param ovisSyms the uniq syms in the order of the ovis.
     * @return the insns.
     */
    private static List<Insn> makeMoreVarsFromStack(String moreVars,
            String topOffset,
            List<String> argSyms,
            List<String> ovisSyms) {
        List<Insn> insns = new ArrayList<>(produceArrayForMoreVars(ovisSyms.size()));
        if (ovisSyms.size() > VAL_FIELDS) {
            for (int mvind = 0; mvind < ovisSyms.size() - VAL_FIELDS; ++ mvind) {
                String sym = ovisSyms.get(mvind + VAL_FIELDS);
                int pairInd = argSyms.lastIndexOf(sym);
                // array[mvind] = dataStack.getAtOffset(#topOffset - (pairCount - pairInd))
                insns.add(new Insn.Dup());
                insns.add(new Insn.PushInt(mvind));
                insns.add(InsnsGenerator.LOAD_DATASTACK);
                insns.add(new Insn.LoadLocal(topOffset));
                insns.add(new Insn.PushInt(argSyms.size() - pairInd));
                insns.add(new Insn.SubInt());
                insns.add(InsnsGenerator.INVOKE_AT_OFFSET);
                insns.add(new Insn.ArrayStore(Type.getType(Val.class)));
            }
        }
        insns.add(storeMoreVars(moreVars));
        return insns;
    }

    /**
     * Insns to clear data stack.
     *
     * @param topOffset the key str of the top offset local var.
     * @param sizeToClear the size of the slice to clear.
     * @return the insns.
     */
    private static List<Insn> clearDataStack(String topOffset, int sizeToClear) {
        List<Insn> insns = new ArrayList<>();
        // dataStack.removeFromOffset(#topOffset - sizeToClear)
        insns.add(InsnsGenerator.LOAD_DATASTACK);
        insns.add(new Insn.LoadLocal(topOffset));
        insns.add(new Insn.PushInt(sizeToClear));
        insns.add(new Insn.SubInt());
        insns.add(InsnsGenerator.INVOKE_REMOVE_FROM_OFFSET);
        return insns;
    }

    /**
     * Insns to invoke Val constructor.
     *
     * @param vm the vm.
     * @param ovisSyms the uniq syms in the order of the ovis.
     * @param sharedVars the local var of the shared vars.
     * @param vlocals the key strs of the local vars of the vals.
     * @param moreVars the key str of #moreVars local var.
     * @return the insns.
     */
    private static List<Insn> invokeValConstructor(Vm vm,
            List<String> ovisSyms,
            String sharedVars,
            List<String> vlocals,
            String moreVars) {
        List<Insn> insns = new ArrayList<>();
        insns.add(new Insn.NewInstance(Type.getType(Val.class)));
        insns.add(new Insn.Dup());
        insns.add(new Insn.LoadThis());
        insns.add(new Insn.GetField(JavaClassIr.TYPE_BASE, "vm", Type.getType(Vm.class)));
        insns.add(new Insn.LoadLocal(sharedVars));
        List<Object> symHandles = ovisSyms.stream()
            .map(sym -> (Object) vm.sym.handleFor(sym))
            .toList();
        insns.add(new Insn.InvokeDynamic(
                    MethodType.methodType(OwnVarIndexes.class),
                    BOOTSTRAP_OVIS_HANDLE,
                    symHandles));
        for (int vi = 0; vi < VAL_FIELDS; ++ vi) {
            insns.add(new Insn.LoadLocal(vlocals.get(vi)));
        }
        insns.add(new Insn.LoadLocal(moreVars));
        insns.add(INVOKE_VAL_CONSTRUCTOR);
        insns.add(InsnsGenerator.STORE_CONTPARAM);
        return insns;
    }

}

// vim: et sw=4 sts=4 fdm=marker
