package org.kink_lang.kink.internal.contract;

import java.util.Locale;

/**
 * Methods for preconditions.
 */
public final class Preconds {

    /**
     * This class should not be instantiated.
     */
    Preconds() {
        throw new UnsupportedOperationException("should not be instantiated");
    }

    /**
     * Checks the precondition for an argument.
     *
     * <p>If the condition is not met, it throws an IllegalArgumentException
     * with the given message.</p>
     *
     * @param cond the precondition.
     * @param msg the message of the IllegalArgumentException.
     * @throws IllegalArgumentException when the condition is not met.
     */
    public static void checkArg(boolean cond, String msg) {
        if (! cond) {
            throw new IllegalArgumentException(msg);
        }
    }

    /**
     * Checks that precondition for a state.
     *
     * <p>If the condition is not met, it throws an IllegalStateException
     * with the given message.</p>
     *
     * @param cond the precondition.
     * @param msg the mesage of the IllegalStateException.
     * @throws IllegalStateException when the condition is not met.
     */
    public static void checkState(boolean cond, String msg) {
        if (! cond) {
            throw new IllegalStateException(msg);
        }
    }

    /**
     * Checks that the {@code checkedIndex} argument is a valid element index
     * of a sequence which contains {@code size} elements.
     * Namely, the method checks that {@code checkedIndex} is not negative,
     * and smaller than {@code size}.
     *
     * @param checkedIndex the index to be checked.
     * @param size the size of a sequence.
     * @throws IndexOutOfBoundsException when {@code checkedIndex} is not a valid element index.
     */
    public static void checkElemIndex(int checkedIndex, int size) {
        if (0 <= checkedIndex && checkedIndex < size) {
            return;
        }

        String msg = String.format(Locale.ROOT, "index must be 0<= and <%d, but actually was %d",
                size, checkedIndex);
        throw new IndexOutOfBoundsException(msg);
    }

    /**
     * Checks that the {@code checkedIndex} argument is a valid position index
     * of a sequence which contains {@code size} elements.
     * Namely, the method checks that {@code checkedIndex} is not negative,
     * and smaller than or equal to {@code size}.
     *
     * @param checkedIndex the index to be checked.
     * @param size the size of a sequence.
     * @throws IndexOutOfBoundsException when {@code checkedIndex} is not a valid position index.
     */
    public static void checkPosIndex(int checkedIndex, int size) {
        if (0 <= checkedIndex && checkedIndex <= size) {
            return;
        }

        String msg = String.format(Locale.ROOT, "index must be 0<= and <=%d, but actually was %d",
                size, checkedIndex);
        throw new IndexOutOfBoundsException(msg);
    }

    /**
     * Checks that the index range {@code from} and {@code to} is valid
     * in the sequence with {@code size}.
     *
     * @param from the from index.
     * @param to the to index.
     * @param size the size of sequence.
     * @throws IndexOutOfBoundsException when {@code from} or {@code to} is out of bounds.
     * @throws IllegalArgumentException when {@code from} is bigger than {@code to}.
     */
    public static void checkRange(int from, int to, int size) {
        if (0 > from) {
            String msg = String.format(Locale.ROOT, "negative from index %d", from);
            throw new IndexOutOfBoundsException(msg);
        }

        if (from > to) {
            String msg = String.format(Locale.ROOT, "from index %d > to index %d", from, to);
            throw new IllegalArgumentException(msg);
        }

        if (to > size) {
            String msg = String.format(Locale.ROOT, "to index %d > size %d", to, size);
            throw new IndexOutOfBoundsException(msg);
        }
    }

}

// vim: et sw=4 sts=4 fdm=marker
