package org.kink_lang.kink.internal.mod.java;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.List;
import java.util.Locale;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import javax.annotation.Nullable;

import org.kink_lang.kink.*;
import org.kink_lang.kink.hostfun.CallContext;
import org.kink_lang.kink.hostfun.HostResult;

/**
 * Factory of JAVA_PROXY mod.
 */
public class JavaProxyMod {

    /** The vm. */
    private final Vm vm;

    /** The name of the mod. */
    public static final String MOD_NAME = "kink/javahost/JAVA_PROXY";

    /**
     * Constructs the factory.
     */
    private JavaProxyMod(Vm vm) {
        this.vm = vm;
    }

    /**
     * Makes JAVA_PROXY mod in the VM.
     *
     * @param vm the vm.
     * @return JAVA_PROXY mod.
     */
    public static Val makeMod(Vm vm) {
        return new JavaProxyMod(vm).makeMod();
    }

    /**
     * Makes a mod.
     */
    private Val makeMod() {
        Val mod = vm.newVal();
        mod.setVar(vm.sym.handleFor("new"),
                vm.fun.make("JAVA_PROXY.new(Intfs $handler)")
                .take(2).action(this::newFun));
        return mod;
    }

    /**
     * Implementation of JAVA_PROXY.new.
     */
    private HostResult newFun(CallContext c) throws ReflectiveOperationException {
        ClassLoader cl = getClass().getClassLoader();

        Val intfsVal = c.arg(0);
        if (! (intfsVal instanceof VecVal)) {
            return c.call(vm.graph.raiseFormat("JAVA_PROXY.new(Class_loader Intfs $handler):"
                        + " required a vec of interfaces for Intfs, but got {}",
                        vm.graph.repr(intfsVal)));
        }
        List<Val> intfVals = ((VecVal) intfsVal).toList();
        Class<?>[] interfaces = new Class<?>[intfVals.size()];
        for (int i = 0; i < interfaces.length; ++ i) {
            Class<?> klass = extractClass(intfVals.get(i));
            if (klass == null) {
                return c.call(vm.graph.raiseFormat("JAVA_PROXY.new(Class_loader Intfs $handler):"
                            + " the #{} elem of Intfs is not a java val of class, but {}",
                            vm.graph.of(vm.num.of(i)),
                            vm.graph.repr(intfVals.get(i))));
            }
            interfaces[i] = klass;
        }

        Val handlerVal = c.arg(1);
        if (! (handlerVal instanceof FunVal)) {
            return c.call(vm.graph.raiseFormat("JAVA_PROXY.new(Class_loader Intfs $handler):"
                        + " required fun for $handler, but got {}",
                        vm.graph.repr(handlerVal)));
        }
        InvocationHandler invHandler = makeInvocationHandler((FunVal) handlerVal);
        Object proxy = Proxy.newProxyInstance(cl, interfaces, invHandler);
        return vm.java.of(proxy, proxy.getClass());
    }

    /**
     * Extracts a Class from val.
     */
    @Nullable
    private Class<?> extractClass(Val val) {
        if (! (val instanceof JavaVal)) {
            return null;
        }

        Object obj = ((JavaVal) val).objectReference();
        if (! (obj instanceof Class)) {
            return null;
        }

        return (Class<?>) obj;
    }

    /**
     * Converts the args array to a vec.
     */
    private final VecVal toArgsVec(Method method, @Nullable Object[] args) {
        // InvocationHandler takes null as args when the size of the arguments is zero
        if (args == null) {
            return vm.vec.of();
        }

        Class<?>[] paramTypes = method.getParameterTypes();
        List<Val> argVals = IntStream.range(0, args.length)
            .mapToObj(i -> vm.java.of(args[i], paramTypes[i]))
            .collect(Collectors.toList());
        return vm.vec.of(argVals);
    }

    /**
     * Result of proxy method.
     */
    private sealed interface ProxyResult {}

    /**
     * Proxy fun returns a val.
     *
     * @param val the returned val.
     */
    private record Success(Val val) implements ProxyResult {}

    /**
     * Proxy fun raises an exception.
     *
     * @param exception the raised exception.
     */
    private record Failure(ExceptionVal exception) implements ProxyResult {}

    /**
     * Makes an InvocationHandler from the fun.
     */
    private InvocationHandler makeInvocationHandler(FunVal fun) {
        return (proxy, method, args) -> {
            JavaVal methodVal = vm.java.of(method, Method.class);
            VecVal argsVal = toArgsVec(method, args);
            ProxyResult result = vm.run(
                    c -> c.call(fun).args(methodVal, argsVal),
                    Success::new,
                    Failure::new);

            if (result instanceof Failure failure) {
                throw new RuntimeException(failure.exception().toRuntimeException());
            }

            Val resultVal = ((Success) result).val();
            if (resultVal instanceof ProxyThrowResultVal) {
                ProxyThrowResultVal thrower = (ProxyThrowResultVal) resultVal;
                throw thrower.thrown();
            }

            return extractResult(method, resultVal);
        };
    }

    /**
     * Extracts the result value of the proxied method.
     */
    @Nullable
    private Object extractResult(Method method, Val resVal) {
        Class<?> returnType = method.getReturnType();
        if (returnType.equals(void.class)) {
            return null;
        }

        if (! (resVal instanceof JavaVal resJava)) {
            throw new IllegalStateException("expected java val is returned, but was not");
        }

        Object resObj = resJava.objectReference();
        if (! JavaHelper.isTypable(resObj, returnType)) {
            throw new IllegalStateException(String.format(Locale.ROOT,
                        "unmatched result type of proxy fun for method «%s», got %s",
                        method, resObj));
        }

        return resObj;
    }

}

// vim: et sw=4 sts=4 fdm=marker
