package org.kink_lang.kink.internal.program.itreeoptimize;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import org.kink_lang.kink.internal.program.itree.*;

/**
 * Converts an assignment to args passing.
 *
 * @param <S> the type of assignment.
 * @param <D> the type of args passing.
 */
public abstract class TemplateArgsPassingOptimizer<S extends ItreeWithRhs, D extends Itree>
    extends BaseOptimizer {

    @Override
    public Itree visit(FastFunItree fun) {
        var body = optimizeBody(fun.body());
        return new FastFunItree(body, fun.pos());
    }

    @Override
    public Itree visit(SlowFunItree fun) {
        var body = optimizeBody(fun.body());
        return new SlowFunItree(body, fun.pos());
    }

    /**
     * If {@code itree} is a type of S and it might be able to replace to an args passing,
     * returns it casting to S. If not, returns empty.
     */
    abstract Optional<S> testReplaceable(Itree itree);

    /**
     * Converts assignment to args passing.
     */
    abstract D convert(S assignment);

    /**
     * Optimizes fun body.
     */
    Itree optimizeBody(Itree body) {
        var steps = Operations.toSteps(body);
        var opt = findTargetAssignment(steps);
        if (! opt.isPresent()) {
            return body;
        }
        var assignmentInd = opt.get().index();

        var prologue = steps.subList(0, assignmentInd);
        if (! prologue.stream().allMatch(s -> s instanceof LstoreItree)) {
            return body;
        }
        var prologueLstores = prologue.stream().map(s -> (LstoreItree) s).toList();

        if (! prologueLstores.stream().map(LstoreItree::rhs).allMatch(this::safeRhs)) {
            return body;
        }

        S assignment = opt.get().element();
        if (! (assignment.rhs() instanceof LderefItree lderef)) {
            return body;
        }

        LocalVar lvar = lderef.lvar();
        var lstoresToArgsVar = prologueLstores.stream()
            .filter(lstore -> lstore.lvar().equals(lvar))
            .toList();
        if (lstoresToArgsVar.size() != 1) {
            return body;
        }

        var lstoreToArgsVar = lstoresToArgsVar.get(0);
        if (! (lstoreToArgsVar.rhs() instanceof ArgVecItree)) {
            return body;
        }

        var argsPassing = convert(assignment);
        var newSteps = new ArrayList<Itree>(prologueLstores);
        newSteps.add(argsPassing);
        newSteps.addAll(steps.subList(assignmentInd + 1, steps.size()));
        return new SeqItree(newSteps, body.pos());
    }

    /**
     * Pair of the index and the element.
     *
     * @param index the index.
     * @param element the element corresponding to the index.
     * @param <E> the type of the element.
     */
    private record IndexElementPair<E> (int index, E element) {}

    /**
     * Returns the index of assignment which might be able to replace.
     */
    private Optional<IndexElementPair<S>> findTargetAssignment(List<Itree> steps) {
        for (int i = 0; i < steps.size(); ++ i) {
            var step = steps.get(i);
            Optional<S> opt = testReplaceable(step);
            if (opt.isPresent()) {
                return Optional.of(new IndexElementPair<>(i, opt.get()));
            }
        }
        return Optional.empty();
    }

    /**
     * Allowlist of rhs types which does not invoke a fun.
     */
    private boolean safeRhs(Itree rhs) {
        return rhs instanceof ArgVecItree
            || rhs instanceof RecvItree
            || rhs instanceof LderefItree;
    }

}

// vim: et sw=4 sts=4 fdm=marker
